/*
Copyright (c) 2007, Dennis M. Sosnoski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.jibx.ws.wsdl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import junit.framework.TestCase;

import org.jibx.binding.Loader;
import org.jibx.binding.Utility;
import org.jibx.binding.classes.BoundClass;
import org.jibx.binding.classes.ClassCache;
import org.jibx.binding.classes.ClassFile;
import org.jibx.binding.classes.MungedClass;
import org.jibx.binding.def.BindingDefinition;
import org.jibx.binding.generator.GlobalCustom;
import org.jibx.runtime.BindingDirectory;
import org.jibx.runtime.IBindingFactory;
import org.jibx.runtime.IUnmarshallingContext;
import org.jibx.runtime.JiBXException;

/**
 * Test code for class handling.
 */
public class CustomizationTestBase extends TestCase
{
    protected static final String ROOT_CLASS = "org.jibx.ws.wsdl.WsdlCustom";
    protected static final IBindingFactory m_bindingFactory;
    static {
        try {
            
            // set paths to be used for loading referenced classes
            URL[] urls = Loader.getClassPaths();
            String[] paths = new String[urls.length];
            for (int i = 0; i < urls.length; i++) {
                paths[i] = urls[i].getFile();
            }
            ClassCache.setPaths(paths);
            ClassFile.setPaths(paths);
            
            // find the binding definition
            ClassLoader loader = CustomizationTestBase.class.getClassLoader();
            URL url = loader.getResource("org/jibx/ws/wsdl/binding.xml");
            InputStream is = url.openStream();
            if (is == null) {
                throw new RuntimeException("Customizations binding definition not found");
            }
            
            // process the binding
            BoundClass.reset();
            MungedClass.reset();
            BindingDefinition.reset();
            BindingDefinition def = Utility.loadBinding("binding.xml",
                "binding", is, url, true);
            def.generateCode(false);
            
            // output the modified class files
            MungedClass.fixChanges(true);
            
            // look up the mapped class and associated binding factory
            Class mclas = Class.forName(ROOT_CLASS);
            m_bindingFactory = BindingDirectory.getFactory(mclas);
            
        } catch (JiBXException e) {
            throw new RuntimeException("JiBXException: " + e.getMessage());
        } catch (IOException e) {
            throw new RuntimeException("IOException: " + e.getMessage());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("ClassNotFoundException: " +
                e.getMessage());
        }
    }
    
    /**
     * Read a customization into model from input stream.
     * 
     * @param is input stream
     * @return root element
     * @throws Exception 
     */
    protected GlobalCustom readCustom(InputStream is)
        throws Exception {
        IUnmarshallingContext ictx =
            m_bindingFactory.createUnmarshallingContext();
        GlobalCustom global = (GlobalCustom)ictx.unmarshalDocument(is, null);
        global.fillClasses();
        return global;
    }
    
    /**
     * Read a customization into model from string.
     * 
     * @param text customization document text
     * @return root element
     * @throws Exception 
     */
    protected GlobalCustom readCustom(String text)
        throws Exception {
        return readCustom(new ByteArrayInputStream(text.getBytes("utf-8")));
    }
}