/*
 * $Id: util.c,v 1.9 2006/04/11 13:48:45 lorenzo Exp $
 * 
 * Copyright (C) 2006 RIPE NCC
 * 
 * Original developer: Lorenzo Colitti <lorenzo@ripe.net>
 * Contributor(s):
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <string.h>

#include "pinger.h"

extern char *optarg;
extern int opterr, optind;

void usage(char *progname) {
	fprintf(stderr, "\
Usage: %s [-i interface] [-s srcaddr] [-t numthreads] [-n numpackets] [-u] <infile|->\
\n\t-i: interface to use for outgoing packets\
\n\t-s: source IP address for packets\
\n\t-t: number of threads to launch, default 3\
\n\t-n: number of pings to send to each address, default 5\
\n\t-u: UDP hack: send (but don't receive) DNS UDP packets instead of ICMP\
\n\tinfile: file of IP addresses to ping, one per line\
\n\t\tIf infile is '-', read from standard input\n", progname);
}

struct options getoptions(int argc, char * argv[]) {
	int opt;
	struct options options;
	int ret;

	opterr = 0;

	/* Default values */
	options.numthreads = 3;
	options.numpings = 5;
	options.port = 0;
	options.filename = NULL;
	memset(&options.srcaddr, 0, sizeof(options.srcaddr));
	options.ifname = NULL;

	while( (opt = getopt(argc, argv, "i:s:t:n:u")) != -1) {
		switch(opt) {
			case 's':
				ret = parse_ip(optarg, 0, (struct sockaddr *) &options.srcaddr, sizeof(options.srcaddr));
				if(ret) {
					fprintf(stderr, "Invalid IP address %s\n", optarg);
					exit(1);
				}
				break;
			case 'i':
				options.ifname = optarg;
				break;
			case 't':
				options.numthreads = atoi(optarg);
				if(options.numthreads < 1) {
					fprintf(stderr, "Invalid number of threads %d\n", options.numthreads);
					exit(1);
				}
				break;
			case 'n':
				options.numpings = atoi(optarg);
				if(options.numpings < 1) {
					fprintf(stderr, "Invalid number of pings %d\n", options.numpings);
					exit(1);
				}
				break;
			case 'f':
				options.filename = optarg;
				break;
			case 'u':
				options.port = 53;
				break;
			default:
				usage(argv[0]);
				exit(1);
		}
	}

	if(optind == argc - 1) {
		options.filename = argv[optind];
	} else {
		usage(argv[0]);
		exit(1);
	}

	return options;
}

/**
 * Prints a hex dump of a buffer to standard output
 * @param buf the buffer to hexdump
 * @param len the length in bytes
 */
void hexdump(char *buf, int len) {
	int i;
	unsigned char c;

	for(i = 0; i < len; i++) {
		c = buf[i];
		printf("%02x ", c);
		if((i + 1) % 16 == 0)
			printf("\n");
	}
	printf("\n");
}

/**
 * Subtracts two struct timeval values
 * @param tv1 the struct timeval to subtract FROM
 * @param tv2 the struct timeval to subtract
 * @return tv1 - tv2
 */
struct timeval tv_subtract(struct timeval tv1, struct timeval tv2) {
	struct timeval result;

	/* Check for carry */
	if(tv2.tv_usec > tv1.tv_usec) {
		tv1.tv_sec -= 1;
		tv1.tv_usec += 1000000;
	}

	result.tv_sec = tv1.tv_sec - tv2.tv_sec;
	result.tv_usec = tv1.tv_usec - tv2.tv_usec;

	return result;
}
