/*
 * Copyright (c) 1996 by Internet Software Consortium.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM DISCLAIMS
 * ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL INTERNET SOFTWARE
 * CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#if defined(LIBC_SCCS) && !defined(lint)
static char rcsid[] = "$Id: gen_ho.c,v 1.8 1996/12/29 06:40:49 vixie Exp $";
#endif /* LIBC_SCCS and not lint */

/* Imports */

#include "port_before.h"

#include <errno.h>
#include <stdlib.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>

#include "port_after.h"

#include "irs_p.h"
#include "gen_p.h"

extern int h_errno;

/* Definitions */

struct pvt {
	struct irs_rule *	rules;
	struct irs_rule *	rule;
	struct irs_ho *		ho;
};

/* Forwards */

static void		ho_close(struct irs_ho *this);
static struct hostent *	ho_byname(struct irs_ho *this, const char *name);
static struct hostent *	ho_byname2(struct irs_ho *this, const char *name,
				   int af);
static struct hostent *	ho_byaddr(struct irs_ho *this, const void *addr,
				  int len, int af);
static struct hostent *	ho_next(struct irs_ho *this);
static void		ho_rewind(struct irs_ho *this);
static void		ho_minimize(struct irs_ho *this);

/* Exports */

struct irs_ho *
irs_gen_ho(struct irs_acc *this) {
	struct gen_p *accpvt = (struct gen_p *)this->private;
	struct irs_ho *ho;
	struct pvt *pvt;

	if (!(ho = malloc(sizeof *ho))) {
		errno = ENOMEM;
		return (NULL);
	}
	memset(ho, 0x5e, sizeof *ho);
	if (!(pvt = (struct pvt *)malloc(sizeof *pvt))) {
		free(ho);
		errno = ENOMEM;
		return (NULL);
	}
	memset(pvt, 0, sizeof *pvt);
	pvt->rules = accpvt->map_rules[irs_ho];
	pvt->rule = pvt->rules;
	ho->private = pvt;
	ho->close = ho_close;
	ho->byname = ho_byname;
	ho->byname2 = ho_byname2;
	ho->byaddr = ho_byaddr;
	ho->next = ho_next;
	ho->rewind = ho_rewind;
	ho->minimize = ho_minimize;
	return (ho);
}

/* Methods. */

static void
ho_close(struct irs_ho *this) {
	struct pvt *pvt = (struct pvt *)this->private;

	free(pvt);
	free(this);
}

static struct hostent *
ho_byname(struct irs_ho *this, const char *name) {
	struct pvt *pvt = (struct pvt *)this->private;
	struct irs_rule *rule;
	struct hostent *rval;
	struct irs_ho *ho;

	for (rule = pvt->rules; rule; rule = rule->next) {
		ho = rule->inst->ho;
		h_errno = NETDB_INTERNAL;
		errno = 0;
		rval = (*ho->byname)(ho, name);
		if (rval != NULL)
			return (rval);
		if (rule->flags & IRS_CONTINUE)
			continue;
		/*
		 * The value TRY_AGAIN can mean that the service
		 * is not available, or just that this particular name
		 * cannot be resolved now.  We use the errno ECONNREFUSED
		 * to distinguish.  If a lookup sets that errno when
		 * h_errno is TRY_AGAIN, we continue to try other lookup
		 * functions, otherwise we return the TRY_AGAIN error.
		 */
		if (h_errno != TRY_AGAIN || errno != ECONNREFUSED)
			break;
	}
	return (NULL);
}

static struct hostent *
ho_byname2(struct irs_ho *this, const char *name, int af) {
	struct pvt *pvt = (struct pvt *)this->private;
	struct irs_rule *rule;
	struct hostent *rval;
	struct irs_ho *ho;

	for (rule = pvt->rules; rule; rule = rule->next) {
		ho = rule->inst->ho;
		h_errno = NETDB_INTERNAL;
		errno = 0;
		rval = (*ho->byname2)(ho, name, af);
		if (rval != NULL)
			return (rval);
		if (rule->flags & IRS_CONTINUE)
			continue;
		/*
		 * See the comments in ho_byname() explaining
		 * the interpretation of TRY_AGAIN and ECONNREFUSED.
		 */
		if (h_errno != TRY_AGAIN || errno != ECONNREFUSED)
			break;
	}
	return (NULL);
}

static struct hostent *
ho_byaddr(struct irs_ho *this, const void *addr, int len, int af) {
	struct pvt *pvt = (struct pvt *)this->private;
	struct irs_rule *rule;
	struct hostent *rval;
	struct irs_ho *ho;

	for (rule = pvt->rules; rule; rule = rule->next) {
		ho = rule->inst->ho;
		h_errno = NETDB_INTERNAL;
		errno = 0;
		rval = (*ho->byaddr)(ho, addr, len, af);
		if (rval != NULL)
			return (rval);
		if (rule->flags & IRS_CONTINUE)
			continue;
		/*
		 * See the comments in ho_byname() explaining
		 * the interpretation of TRY_AGAIN and ECONNREFUSED.
		 */
		if (h_errno != TRY_AGAIN || errno != ECONNREFUSED)
			break;
	}
	return (NULL);
}

static struct hostent *
ho_next(struct irs_ho *this) {
	struct pvt *pvt = (struct pvt *)this->private;
	struct hostent *rval;
	struct irs_ho *ho;

	while (pvt->rule) {
		ho = pvt->rule->inst->ho;
		rval = (*ho->next)(ho);
		if (rval)
			return (rval);
		if (!(pvt->rule->flags & IRS_CONTINUE))
			break;
		pvt->rule = pvt->rule->next;
		if (pvt->rule) {
			ho = pvt->rule->inst->ho;
			(*ho->rewind)(ho);
		}
	}
	return (NULL);
}

static void
ho_rewind(struct irs_ho *this) {
	struct pvt *pvt = (struct pvt *)this->private;
	struct irs_ho *ho;

	pvt->rule = pvt->rules;
	if (pvt->rule) {
		ho = pvt->rule->inst->ho;
		(*ho->rewind)(ho);
	}
}

static void
ho_minimize(struct irs_ho *this) {
	struct pvt *pvt = (struct pvt *)this->private;
	struct irs_rule *rule;

	for (rule = pvt->rules; rule != NULL; rule = rule->next) {
		struct irs_ho *ho = rule->inst->ho;

		(*ho->minimize)(ho);
	}
}
