/* Copyright (c) 1995 by Internet Software Consortium
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM DISCLAIMS
 * ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL INTERNET SOFTWARE
 * CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* signals.c - implement asynch signals for the eventlib
 * vix 16sep95 [initial]
 */

#if !defined(LINT) && !defined(CODECENTER)
static const char rcsid[] = "$Id: signals.c,v 1.1 1995/09/17 05:30:26 vixie Exp $";
#endif

#include "eventlib.h"
#include "eventlib_p.h"

static evSignal *FindSig(const evContext_p *ctx, int sig);
static void EatMe(int mystery);

int
evSetSignal(evContext opaqueCtx, int sig,
	    evSignalFunc func, void *uap,
	    evSignalID *opaqueID
) {
	evContext_p *ctx = opaqueCtx.opaque;
	evSignal *old, *new;
	struct sigaction act, oact;
	sigset_t set, oset;
	int save;

	/*
	 * POSIX gives us no way to validate the signal number, since there's
	 * no requirement that sigaddset() return (-1) if given a bad signal
	 * (and BSD's sigaddset() macro doesn't do so in any case.)  So we'll
	 * try a sigaddset(), check it for errors, and then check its results.
	 */
	OK(sigemptyset(&set));
	OK(sigaddset(&set, sig));
	if (sigismember(&set, sig) <= 0)
		ERR(EINVAL);

	/*
	 * If we already have something in the signal list for this signal,
	 * then we want its ``oact,'' elsewise we need to get the ``oact''
	 * from the system.
	 */
	old = FindSig(ctx, sig);

	/*
	 * Block the signal.  If it wasn't previously blocked and we had the
	 * signal on the list, exit now.  If it was previously blocked and we
	 * didn't have it on the list, remember the blockedness so that we
	 * leave it blocked after evClearSignal().  Note that we block the
	 * signal even if it's already on our signal list, since this gives
	 * us a chance to blue chunks if it's not in fact blocked any more.
	 *
	 * Note that blocking the signal has the effect that our select() in
	 * evGetNext() will not return with -1 and errno==EINTR when a signal
	 * occurs.  We would rather not have that happen, but to fix it we
	 * would have to let the signal call a handler, and that handler would
	 * either have to have static global storage (to remember that the
	 * signal had occured -- had for thread safety!).  We prefer polling,
	 * while noting that POSIX should have defined a signal "uap" argument.
	 */
	OK(sigprocmask(SIG_BLOCK, &set, &oset));
	if (!old) {
		/* First catch of this signal.  Make blockedBefore right. */

		if (sigismember(&oset, sig) > 0) {
			/* Note that an error below doesn't need any fixups. */
			OK(sigaddset(&ctx->blockedBefore, sig));
		} else {
			if (sigismember(&ctx->blockedBefore, sig) > 0)
				OK(sigdelset(&ctx->blockedBefore, sig));
		}

		/*
		 * Install a signal handler.  We don't do anything in the
		 * handler, but the POSIX docs are unclear as to whether
		 * SIG_IGN will preclude the sigpending() mask from ever
		 * showing that the signal happened.  Since we rely on the
		 * (obscure) POSIX semantics of sigprocmask() (see the
		 * comments in evGetNext for more details), we want to make
		 * sure that the kernel has no excuse not to set our
		 * sigpending() mask bit.
		 */
		sigemptyset(&set);
		act.sa_handler = EatMe;
		act.sa_mask = set;
		act.sa_flags = 0;
		if (sigaction(sig, &act, &oact) < 0) {
			save = errno;
			(void) sigprocmask(SIG_SETMASK, &oset, NULL);/*Fixup.*/
			ERR(errno);
		}
	} else {
		/* This signal is already being caught.  Make sure, though. */

		if (sigismember(&oset, sig) <= 0) {
			(void) sigprocmask(SIG_SETMASK, &oset, NULL);/*Fixup.*/
			ERR(EBADF);
		}
		oact = old->oact;
	}

	/* Allocate, fill, link. */
	NEW(new);
	if (!new) {
		(void) sigprocmask(SIG_SETMASK, &oset, NULL);   /* Fixup. */
		if (!old)
			(void) sigaction(sig, &oact, NULL);
		ERR(ENOMEM);
	}
	new->func = func;
	new->uap = uap;
	new->sig = sig;
	new->oact = oact;
	new->next = ctx->signals;
	ctx->signals = new;

	/* Save handle if we've been given one. */
	if (opaqueID)
		opaqueID->opaque = new;

	return (0);
}

int
evClearSignal(evContext opaqueCtx, evSignalID opaqueID) {
	evContext_p *ctx = opaqueCtx.opaque;
	evSignal *del = opaqueID.opaque;
	evSignal *old, *cur;

	/* Unlink. */
	for (old = NULL, cur = ctx->signals;
	     cur != NULL && cur != del;
	     old = cur, cur = cur->next)
		NULL;
	if (! cur)
		ERR(ENOENT);
	if (! old)
		ctx->signals = del->next;
	else
		old->next = del->next;

	/*
	 * Was the signal caught elsewise?  If not, we should restore the
	 * signal handler and blocking status to whatever it used to be.
	 */
	if (!(cur = FindSig(ctx, del->sig))) {
		if (sigismember(&ctx->blockedBefore, del->sig) <= 0) {
			sigset_t set;

			(void) sigemptyset(&set);
			(void) sigaddset(&set, del->sig);
			(void) sigprocmask(SIG_UNBLOCK, &set, NULL);
		}
		(void) sigaction(del->sig, &del->oact, NULL);
	}

	/* Deallocate and get out. */
	free(del);
	return (0);
}

static evSignal *
FindSig(const evContext_p *ctx, int sig) {
	register evSignal *cur;

	for (cur = ctx->signals; cur && cur->sig != sig; cur = cur->next)
		NULL;
	return (cur);
}

static void
EatMe(int mystery) {
	NULL;
}
