/* $Id: tif.c,v 1.8 2005/06/19 19:39:36 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * imp/src/tif.c,v 1.2 2004/10/08 20:28:06 flaw
 * if/src/tif.c,v 1.1 2004/09/28 15:40:49 flaw
 *//*
 * Common tuple interface routines
 */
#include <postgres.h>
#include <access/attnum.h>
#include <access/heapam.h>
#include <access/htup.h>
#include <access/hio.h>
#include <catalog/pg_type.h>
#include <lib/stringinfo.h>
#include <utils/array.h>
#include <utils/rel.h>
#include <utils/relcache.h>
#include <utils/syscache.h>
#include <utils/tuplestore.h>
#include <tcop/tcopprot.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/conv.h>
#include <pypg/error.h>

/*
 * HeapTuple_FromTupleDescAndPySequence()
 *
 * XXX: this really doesn't properly clean up in the case of an ERROR.
 */
HeapTuple
HeapTuple_FromTupleDescAndPySequence(TupleDesc td, PyObj args)
{
	int i, natts, slen;
	Datum *datums = NULL;
	char *nulls = NULL;
	bool *frees = NULL; /* Explicit marker for datums that need to be freed */
	PyObj ob;
	volatile HeapTuple ht = NULL;

	natts = td->natts;

	if ((slen = PySequence_Length(args)) != natts)
	{
		PyErr_Format(PyExc_TypeError,
			"This TupleDesc has exactly %d attributes, got %d arguments for tuple",
			natts, slen
		);
		return(NULL);
	}

	if (natts == 0)
	{
		Datum d = 0;
		char n = 'n';
		PgError_TRAP(ht = heap_formtuple(td, &d, &n));
		return(ht);
	}
	slen = 0;

	datums = malloc(sizeof(Datum) * natts);
	if (datums == NULL)
	{
		return(NULL);
	}
	nulls = malloc(sizeof(bool) * natts);
	if (nulls == NULL)
	{
		free(datums);
		return(NULL);
	}
	frees = malloc(sizeof(bool) * natts);
	if (frees == NULL)
	{
		pfree(datums);
		pfree(nulls);
		return(NULL);
	}

	/* make the datums for the tuple */
	for (i = 0; i < natts; ++i)
	{
		Form_pg_attribute att;
		att = td->attrs[i];
		frees[i] = false;

		ob = PySequence_GetItem(args, i);
		if (ob == PyPgDEFAULT)
		{
			/* XXX: evaluate the default expr */
			datums[i] = 0;
			nulls[i] = 'n';
		}
		else if (ob == Py_None || (PyPgObject_Check(ob) && PyPgObject_IsNULL(ob)))
		{
			datums[i] = 0;
			nulls[i] = 'n';
		}
		else
		{
			nulls[i] = ' ';
			if (PyPgObject_Check(ob))
			{
				Oid toid;
				toid = PyPgObject_FetchTypeOid(ob);
				/* XXX: check errors */

				if (toid != att->atttypid)
				{
					PyErr_Format(PyExc_TypeError,
						"expecting %d type for attribute %d, got %d",
						(int) att->atttypid, i, (int) toid
					);

					slen = i;
					break;
				}
				datums[i] = PyPgObject_FetchDatum(ob);
				/* XXX: check errors */
			}
			else
			{
				datums[i] = Datum_FromTypeOidAndPyObject(att->atttypid, ob);
				if (PyErr_Occurred())
				{
					slen = i;
					break;
				}
				if (!att->attbyval)
					frees[i] = true;
				nulls[i] = ' ';
			}
		}
	}

	/* No error if slen == 0 */
	if (slen == 0)
		ht = heap_formtuple(td, datums, nulls);
	else
		natts = slen;	/* limit the clean up to those datums created */

	/* free up allocated memory; formtuple create a copy, so obliterate ours */
	for (i = 0; i < natts; ++i)
	{
		if (frees[i] == true)
			pfree(DatumGetPointer(datums[i]));
	}
	free(datums);
	free(nulls);
	free(frees);

	return(ht);
}

/*
 * PyPgTIF_FetchAttribute - Fetch a part that every HeapTuple has
 *
 * This is NOT where HeapTuple attributes are fetched, rather those are fetched
 * through the sequence protocol and the mapping protocol. These are meant to be
 * OBJECT attributes fetched through tp_getattr.
 */
PyObj
PyPgTIF_FetchAttribute(HeapTuple ht, char *attr)
{
	PyObj rob = NULL;

	if (!strcmp(attr, "oid"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(OIDOID, HeapTuple_FetchOid(ht));
	}
	else if (!strcmp(attr, "tableoid"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(OIDOID, HeapTuple_FetchTableOid(ht));
	}
	else if (!strcmp(attr, "Xmin"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(XIDOID, HeapTuple_FetchXmin(ht));
	}
	else if (!strcmp(attr, "Xmax"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(XIDOID, HeapTuple_FetchXmax(ht));
	}
	else if (!strcmp(attr, "Xvac"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(XIDOID, HeapTuple_FetchXvac(ht));
	}
	else if (!strcmp(attr, "Cmin"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(CIDOID, HeapTuple_FetchCmin(ht));
	}
	else if (!strcmp(attr, "Cmax"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(CIDOID, HeapTuple_FetchCmax(ht));
	}
	else if (!strcmp(attr, "TID"))
	{
		ItemPointer to, from;
		from = &HeapTuple_FetchTID(ht);
		to = palloc(SizeOfIptrData);
		if (to)
		{
			ItemPointerCopy(from, to);
			rob = PyPgObject_FromTypeOidAndDatum(TIDOID, PointerGetDatum(to));
		}
	}

	return(rob);
}

/*
 * PyPgTIF_FetchItem - fetch a tuple's attribute as a PyPgObject.
 */
PyObj
PyPgTIF_FetchItem(TupleDesc td, HeapTuple ht, unsigned int item)
{
	bool isnull = false;
	int natts;
	Datum datum;
	PyObj to;
	PyObj rob = NULL;

	natts = HeapTuple_FetchNatts(ht);
	if (item >= natts)
	{
		PyErr_Format(PyExc_IndexError,
				"index(%d) out of range(%d)", item, natts);
		return(NULL);
	}

	datum = HeapTuple_FetchAttribute(ht, item, td, &isnull);
	if (PyErr_Occurred())
		return(NULL);

	to = PyPgType_FromOid(td->attrs[item]->atttypid);
	if (to == NULL)
		return(NULL);
	DECREF(to);

	if (isnull)
	{
		rob = PyPgType_FetchNULL(to);
		Py_INCREF(rob);
	}
	else
		rob = PyPgObject_FromPyPgTypeAndDatum(to, datum);

	return(rob);
}

/*
 * PyPgTIF_FetchSubscript - fetch a tuple's attribute as a PyPgObject using a
 * subscript
 */
PyObj
PyPgTIF_FetchSubscript(TupleDesc td, HeapTuple ht, PyObj ob)
{
	AttrNumber an;
	PyObj rob = NULL;
	an = AttrNumber_FromTupleDescAndPyObject(td, ob);
	if (!PyErr_Occurred())
		rob = PyPgTIF_FetchItem(td, ht, an);
	return(rob);
}

PyObj
PyPgTIF_FetchSlice
(
	TupleDesc td,
	HeapTuple ht,
	unsigned int from,
	unsigned int to
)
{
	PyObj rob, v;
	unsigned int i;

	rob = PyTuple_New(to - from);
	for (i = from; i < to; ++i)
	{
		v = PyPgTIF_FetchItem(td, ht, i);
		if (v == NULL)
		{
			DECREF(rob);
			return(NULL);
		}
		PyTuple_SET_ITEM(rob, i - from, v);
	}

	return(rob);
}

/*
 * PyPgTIF_String - Effectively record_out, but on the given TD and HT
 *
 * This is a PyPg function as it is meant to return NULL on error, and with a
 * raised exception.
 *
 * This code should reflect the implementation in 8's recordout function.
 * (Some of this code is taken from src/backend/utils/adt/rowtypes.c)
 */
char *
PyPgTIF_String(TupleDesc td, HeapTuple ht)
{
	StringInfoData buf;
	volatile unsigned long i, natts;
	
	natts = td->natts;
	initStringInfo(&buf);
	appendStringInfoChar(&buf, '(');

	for (i = 0; i < natts; ++i)
	{
		Form_pg_attribute att;
		HeapTuple ttup;
		Form_pg_type ts;
		Oid typoutput, typrelid;
		bool isnull;
		volatile Datum d;
		Oid atttypid;

		att = td->attrs[i];
		if (att->attisdropped)
			continue;

		atttypid = att->atttypid;

		ttup = SearchSysCache(TYPEOID, atttypid, 0, 0, 0);
		ts = TYPESTRUCT(ttup);
		typoutput = ts->typoutput;
		typrelid = ts->typrelid;
		ReleaseSysCache(ttup);

		d = HeapTuple_FetchAttribute(ht, i, td, &isnull);
		if (PyErr_Occurred())
		{
			pfree(buf.data);
			return(NULL);
		}

		if (!isnull)
		{
			bool quote_it = false;
			char * volatile str = NULL, *tmp;

#if !defined(PG_HAS_DATUMTUPLES)
/*
 * This is to handle a case prior to 8.0 where record_out resulted in an error.
 */
			if (typrelid != InvalidOid)
			{
				Relation rd;
				TupleDesc td;
				HeapTuple ht;
				rd = RelationIdGetRelation(typrelid);
				td = RelationGetDescr(rd);
				RelationClose(rd); /* Cheat a bit to avoid catching errors */
				ht = (HeapTuple) d;
				str = PyPgTIF_String(td, ht);
			}
			else
#endif
			{
				/*
				 * XXX: Could use some optimizing
				 */
				PgError_TRAP(
					str = DatumGetCString(OidFunctionCall1(typoutput, d))
				);
			}

			if (PyErr_Occurred())
			{
				pfree(buf.data);
				return(NULL);
			}
			
			quote_it = (str[0] == '\0');
			for (tmp = str; *tmp; ++tmp)
			{
				char ch = *tmp;
				if (
						ch == '"'
					|| ch == '\\'
					|| ch == '('
					|| ch == ')'
					|| ch == ','
					|| isspace((unsigned char) ch))
				{
					quote_it = true;
					break;
				}
			}

			if (quote_it)
				appendStringInfoChar(&buf, '"');
			for (tmp = str; *tmp; ++tmp)
			{
				char ch = *tmp;

				if (ch == '"' || ch == '\\')
					appendStringInfoChar(&buf, ch);
				appendStringInfoChar(&buf, ch); 
			}
			if (quote_it)
				appendStringInfoChar(&buf, '"');
			
			pfree(str);
		}

		if (i+1 < natts)
			appendStringInfoChar(&buf, ',');
	}
	appendStringInfoChar(&buf, ')');

	return(buf.data);
}
/*
 * vim: ts=3:sw=3:noet:
 */
