#=====================================================================
# SQL-Ledger Accounting
# Copyright (C) 2001
#
#  Author: Dieter Simader
#   Email: dsimader@sql-ledger.org
#     Web: http://www.sql-ledger.org
#
# Contributors: Thomas Bayen <bayen@gmx.de>
#               Antti Kaihola <akaihola@siba.fi>
#               Moritz Bunkus (tex code)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#======================================================================
# Utilities for parsing forms
# and supporting routines for linking account numbers
# used in AR, AP and IS modules
#
#======================================================================

package Form;

use POSIX qw(strftime);

sub new {
  my $type = shift;
  
  my $self = {};

  read(STDIN, $_, $ENV{CONTENT_LENGTH});
  
  if ($ENV{QUERY_STRING}) {
    $_ = $ENV{QUERY_STRING};
  }

  if ($ARGV[0]) {
    $_ = $ARGV[0];
  }

  foreach $item (split(/&/)) {
    ($key, $value) = split(/=/, $item);
    $self->{$key} = &unescape("",$value);
  }

  $self->{action} = lc $self->{action};
  $self->{action} =~ s/( |-|,)/_/g;

  $self->{version} = "3.0.0";
  $self->{dbversion} = "3.0.0";

  bless $self, $type;
  
}


sub debug {
  my ($self) = @_;
  
  print "\n";
  
  map { print "$_ = $self->{$_}\n" } (sort keys %{$self});
  
} 

  
sub set_cookie {
  my ($self, $init) = @_;

  $self->{timeout} = ($self->{timeout} > 0) ? $self->{timeout} : 3600;

  if ($self->{endsession}) {
    $_ = time;
  } else {
    $_ = time + $self->{timeout};
  }

  if ($ENV{HTTP_USER_AGENT}) {

    my @d = split / +/, scalar gmtime($_);
    my $today = "$d[0], $d[2]-$d[1]-$d[4] $d[3] GMT";

    if ($init) {
      $self->{sessionid} = time;
    }
    print qq|Set-Cookie: SQL-Ledger-$self->{login}=$self->{sessionid}; expires=$today; path=/;\n| if $self->{login};
  }

}

sub escape {
  my ($self, $str, $beenthere) = @_;

  # for Apache 2 we escape strings twice
  if (($ENV{SERVER_SIGNATURE} =~ /Apache\/2\.(\d+)\.(\d+)/) && !$beenthere) {
    $str = $self->escape($str, 1) if $2 < 44;
  }
	    
  $str =~ s/([^a-zA-Z0-9_.-])/sprintf("%%%02x", ord($1))/ge;
  $str;

}

sub unescape {
  my ($self, $str) = @_;
  
  $str =~ tr/+/ /;
  $str =~ s/\\$//;

  $str =~ s/%([0-9a-fA-Z]{2})/pack("c",hex($1))/eg;

  $str;

}


 
sub error {
  my ($self, $msg) = @_;

  if ($ENV{HTTP_USER_AGENT}) {
    $msg =~ s/\n/<br>/g;

    delete $self->{pre};

    if (!$self->{header}) {
      $self->header;
    }

    print qq|<body><h2 class=error>Error!</h2>

    <p><b>$msg</b>|;

    exit;

  } else {
  
    if ($self->{error_function}) {
      &{ $self->{error_function} }($msg);
    } else {
      die "Error: $msg\n";
    }
  }
  
}


sub save_query {
  my ($self, $form,$dbname, $query) = @_;
  
  $date=strftime "20%y%m%d", localtime;
    $date =~ s/\n//g;

  open(TEMPLATE, ">>log/$dbname$date.sql") or $self->error("log/$dbname$date.sql : $!");
# $self->error($dbname);
 $query =~ s/\r\n/\n/g;
  print TEMPLATE "$query;\n";

  close(TEMPLATE);

}

sub info {
  my ($self, $msg) = @_;

  if ($ENV{HTTP_USER_AGENT}) {
    $msg =~ s/\n/<br>/g;

    print qq|Content-Type: text/html

    <body bgcolor=ffffff>

    <h2><font color=blue>Status!</font></h2>

    <p><b>$msg</b>
    
    </body>
    </html>
    |;

  } else {
    print "Status: $msg\n";
  }
}


sub numtextrows {
  my ($self, $str, $cols, $maxrows) = @_;

  my $rows;

  map { $rows += int ((length $_)/$cols) + 1 } (split /\r/, $str);

  $rows = $maxrows if (defined $maxrows && ($rows > $maxrows));
  
  $rows;

}


sub dberror {
  my ($self, $msg) = @_;

  $self->error("$msg\n".$DBI::errstr);
  
}


sub isblank {
  my ($self, $name, $msg) = @_;

  if ($self->{$name} =~ /^\s*$/) {
    $self->error($msg);
  }
}
  

sub header {
  my ($self, $init) = @_;

  return if $self->{header};

  my ($nocache, $stylesheet, $charset,$favicon);
  $self->{title} = 'Open Business Accounting version '.$self->{version}   if $self->{title} eq '';
  
  # Opera caches scripts
  # use expire tag to prevent caching
  if ($ENV{HTTP_USER_AGENT} =~ /Opera/) {
    $nocache = qq|<META HTTP-EQUIV="Expires" CONTENT="Tue, 01 Jan 1980 1:00:00 GMT">
  <META HTTP-EQUIV="Pragma" CONTENT="no-cache">
|;
  }

  if ($self->{stylesheet} && (-f "css/$self->{stylesheet}")) {
    $stylesheet = qq|<LINK REL="stylesheet" HREF="css/$self->{stylesheet}" TYPE="text/css" TITLE="OBA style sheet">
|;
  }
      if ($self->{favicon} && (-f "$self->{favicon}")) {
      $favicon = qq|<LINK REL="shortcut icon" HREF="$self->{favicon}" TYPE="image/x-icon">
  |;
    }

    $self->set_cookie($init);

  if ($self->{charset}) {
#    $charset = qq|<META HTTP-EQUIV="Charset" CONTENT="$self->{charset}">
#|;
   $charset = qq|<meta http-equiv="Content-Type" content="text/html; charset=$self->{charset}">|;
  }

  print qq|Content-Type: text/html

<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0//EN">
<html>
<head>
  $charset
    $favicon

  <title>$self->{title}</title>
  $nocache
  $stylesheet
</head>
 <script language="javascript" type="text/javascript" src="javascript/date.js"></script>
 <script language="javascript" type="text/javascript" src="javascript/nextfocus.js"></script>
 <script language="javascript" type="text/javascript" src="javascript/functions.js"></script>
$self->{pre}

|;
  $self->{header} = 1;

}


sub info {
  my ($self, $msg) = @_;

  if ($ENV{HTTP_USER_AGENT}) {
    $msg =~ s/\n/<br>/g;

    delete $self->{pre};

    if (!$self->{header}) {
      $self->header;
      print qq|
      <body>|;
      $self->{header} = 1;
    }

    print "<br><b>$msg</b>";

  } else {
  
    if ($self->{info_function}) {
      &{ $self->{info_function} }($msg);
    } else {
      print "$msg\n";
    }
  }
  
}



sub quote {
  my ($self, $str) = @_;

  if ($str && ! ref($str)) {
    $str =~ s/"/&quot;/g;
  }

  $str;

}

sub hide_form {
  my $self = shift;

  map { print qq|<input type=hidden name=$_ value="|.$self->quote($self->{$_}).qq|">\n| } sort keys %$self;
  
}

  
sub redirect {
 # my $self = shift;
 my ($self, $msg) = @_;
 #  $self->error($msg);

  if ($self->{callback}) {
    ($script, $argv) = split(/\?/, $self->{callback});
   # $argv= &unescape("",$argv);
	#     	  $self->dberror($script." ".$argv);

# for Windows, OS/2 uncomment the next line
  exec ("perl","$script",$argv);  
  } else {
   # if ($ENV{HTTP_USER_AGENT}) {
     # $self->header("text/html");
    #}
    $self->info($msg);


	}
}


sub isposted {

  my ($self, $rc) = @_;

  if ($rc) {
    $self->redirect;
  }

  $rc;
  
}


sub isdeleted {
  my ($self, $rc) = @_;

  if ($rc) {
    $self->redirect;
  }

  $rc;
  
}


sub sort_columns {
  my ($self, @columns) = @_;

  @columns = grep !/^$self->{sort}$/, @columns;
  splice @columns, 0, 0, $self->{sort};

  @columns;
  
}


sub format_amount2 {
  my ($self, $myconfig, $amount, $places, $dash) = @_;

  if (defined $places) {
    $amount = $self->round_amount($amount, $places) if ($places >= 0);
  }

  # is the amount negative
  my $negative = ($amount < 0);
  
  if ($amount != 0) {
    if ($myconfig->{numberformat} && ($myconfig->{numberformat} ne '1000.00')) {
      my ($whole, $dec) = split /\./, "$amount";
      $whole =~ s/-//;
      $amount = join '', reverse split //, $whole;
      
      if ($myconfig->{numberformat} eq '1,000.00') {
	$amount =~ s/\d{3,}?/$&,/g;
	$amount =~ s/,$//;
	$amount = join '', reverse split //, $amount;
	$amount .= "\.$dec" if $dec;
      }
      
      if ($myconfig->{numberformat} eq '1.000,00') {
	$amount =~ s/\d{3,}?/$&./g;
	$amount =~ s/\.$//;
	$amount = join '', reverse split //, $amount;
	$amount .= ",$dec" if $dec;
      }
      
      if ($myconfig->{numberformat} eq '1000,00') {
	$amount = "$whole";
	$amount .= ",$dec" if $dec;
      }

      if ($dash =~ /-/) {
	$amount = ($negative) ? "($amount)" : "$amount";
      } elsif ($dash =~ /DRCR/) {
	$amount = ($negative) ? "$amount DR" : "$amount CR";
      } else {
	$amount = ($negative) ? "-$amount" : "$amount";
      }
    }
  } else {
    $amount = ($dash) ? "0.00" : "$dash";
  }

  $amount;

}

sub format_amount {
  my ($self, $myconfig, $amount, $places, $dash) = @_;

  if ($places =~ /\d/) {
    $amount = $self->round_amount($amount, $places);
  }

  # is the amount negative
  my $negative = ($amount < 0);
  
  if ($amount != 0) {
    if ($myconfig->{numberformat} && ($myconfig->{numberformat} ne '1000.00')) {
      my ($whole, $dec) = split /\./, "$amount";
      $whole =~ s/-//;
      $amount = join '', reverse split //, $whole;
      
      if ($myconfig->{numberformat} eq '1,000.00') {
	$amount =~ s/\d{3,}?/$&,/g;
	$amount =~ s/,$//;
	$amount = join '', reverse split //, $amount;
	$amount .= "\.$dec" if ($dec ne "");
      }
      
      if ($myconfig->{numberformat} eq "1'000.00") {
	$amount =~ s/\d{3,}?/$&'/g;
	$amount =~ s/'$//;
	$amount = join '', reverse split //, $amount;
	$amount .= "\.$dec" if ($dec ne "");
      }
      
      if ($myconfig->{numberformat} eq '1.000,00') {
	$amount =~ s/\d{3,}?/$&./g;
	$amount =~ s/\.$//;
	$amount = join '', reverse split //, $amount;
	$amount .= ",$dec" if ($dec ne "");
      }
      
      if ($myconfig->{numberformat} eq '1000,00') {
	$amount = "$whole";
	$amount .= ",$dec" if ($dec ne "");
      }

      if ($dash =~ /-/) {
	$amount = ($negative) ? "($amount)" : "$amount";
      } elsif ($dash =~ /DRCR/) {
	$amount = ($negative) ? "$amount DR" : "$amount CR";
      } else {
	$amount = ($negative) ? "-$amount" : "$amount";
      }
    }
  } else {
    if ($dash eq "0" && $places) {
      if ($myconfig->{numberformat} eq '1.000,00') {
	$amount = "0".","."0" x $places;
      } else {
	$amount = "0"."."."0" x $places;
      }
    } else {
      $amount = ($dash ne "") ? "$dash" : "";
    }
  }

  $amount;

}

sub parse_amount {
  my ($self, $myconfig, $amount) = @_;

  if (($myconfig->{numberformat} eq '1.000,00') ||
      ($myconfig->{numberformat} eq '1000,00')) {
    $amount =~ s/\.//g;
    $amount =~ s/,/\./;
  }

  $amount =~ s/,//g;
  
  return ($amount * 1);

}


sub round_amount {
  my ($self, $amount, $places) = @_;

#  $places = 3 if $places == 2;
  
  # compensate for perl bug, add 1/10^$places+2
  sprintf("%.${places}f", $amount + (1 / (10 ** ($places + 2))) * (($amount > 0) ? 1 : -1));

}


sub parse_template {
  my ($self, $myconfig, $userspath) = @_;

  # { Moritz Bunkus
  # Some variables used for page breaks
  my ($chars_per_line, $lines_on_first_page, $lines_on_second_page) = (0, 0, 0);
  my ($current_page, $current_line) = (1, 1);
  my $pagebreak = "";
  my $sum = 0;
  # } Moritz Bunkus

  open(IN, "$self->{templates}/$self->{IN}") or $self->error("$self->{IN} : $!");

  @_ = <IN>;
  close(IN);
  
  $self->{copies} = 1 if (($self->{copies} *= 1) <= 0);
  
  # OUT is used for the media, screen, printer, email
  # for postscript we store a copy in a temporary file
  my $fileid = time;
  $self->{tmpfile} = "$userspath/${fileid}.$self->{IN}";
  if ($self->{format} =~ /(postscript|pdf)/ || $self->{media} eq 'email') {
    $out = $self->{OUT};
    $self->{OUT} = ">$self->{tmpfile}";
  }
  
  
  if ($self->{OUT}) {
    open(OUT, "$self->{OUT}") or $self->error("$self->{OUT} : $!");
  } else {
    open(OUT, ">-") or $self->error("STDOUT : $!");
    $self->header;
  }


  # first we generate a tmpfile
  # read file and replace <%variable%>
  while ($_ = shift) {
      
    $par = "";
    $var = $_;


    # { Moritz Bunkus
    # detect pagebreak block and its parameters
    if (/<%pagebreak ([0-9]+) ([0-9]+) ([0-9]+)%>/) {
      $chars_per_line = $1;
      $lines_on_first_page = $2;
      $lines_on_second_page = $3;
      
      while ($_ = shift) {
        last if (/<\%end pagebreak%>/);
        $pagebreak .= $_;
      }
    }
    # } Moritz Bunkus

    
    if (/<%foreach /) {
      # this one we need for the count
      chomp $var;
      $var =~ s/<%foreach (.+?)%>/$1/;
      while ($_ = shift) {
	last if (/<%end /);
	# store line in $par
	$par .= $_;
    }
      
      # display contents of $self->{number}[] array
      for $i (0 .. $#{ $self->{$var} }) {

        # { Moritz Bunkus
        # Try to detect whether a manual page break is necessary
        # but only if there was a <%pagebreak ...%> block before
	
        if ($chars_per_line) {
          my $lines = int(length($self->{"description"}[$i]) / $chars_per_line + 0.95);
          my $lpp;
	  
          if ($current_page == 1) {
            $lpp = $lines_on_first_page;
          } else {
            $lpp = $lines_on_second_page;
          }

          # Yes we need a manual page break
          if (($current_line + $lines) > $lpp) {
            my $pb = $pagebreak;
	    
            # replace the special variables <%sumcarriedforward%>
            # and <%lastpage%>
	    
            my $psum = $self->format_amount($myconfig, $sum, 2);
            $pb =~ s/<%sumcarriedforward%>/$psum/g;
            $pb =~ s/<%lastpage%>/$current_page/g;
            
	    # only "normal" variables are supported here
            # (no <%if, no <%foreach, no <%include)
            
	    $pb =~ s/<%(.+?)%>/$self->{$1}/g;
            
	    # page break block is ready to rock
            print(OUT $pb);
            $current_page++;
            $current_line = 1;
          }
          $current_line += $lines;
        }
        $sum += $self->parse_amount($myconfig, $self->{"linetotal"}[$i]);
        # } Moritz Bunkus


	# don't parse par, we need it for each line
	$_ = $par;
	s/<%(.+?)%>/$self->{$1}[$i]/mg;
	print OUT;
      }
      next;
    }

    # if not comes before if!
    if (/<%if not /) {
      # check if it is not set and display
      chop;
      s/<%if not (.+?)%>/$1/;

      unless ($self->{$_}) {
	while ($_ = shift) {
	  last if (/<%end /);

	  # store line in $par
	  $par .= $_;
	}
	
	$_ = $par;
	
      } else {
	while ($_ = shift) {
	  last if (/<%end /);
	}
	next;
      }
    }
 
    if (/<%if /) {
      # check if it is set and display
      chop;
      s/<%if (.+?)%>/$1/;

      if ($self->{$_}) {
	while ($_ = shift) {
	  last if (/<%end /);

	  # store line in $par
	  $par .= $_;
	}
	
	$_ = $par;
	
      } else {
	while ($_ = shift) {
	  last if (/<%end /);
	}
	next;
      }
    }
   
    # check for <%include filename%>
    if (/<%include /) {
      
      # get the filename
      chomp $var;
      $var =~ s/<%include (.+?)%>/$1/;

      # mangle filename if someone tries to be cute
      $var =~ s/\///g;

      # prevent the infinite loop!
      next if ($self->{"$var"});

      open(INC, "$self->{templates}/$var") or $self->error($self->cleanup."$self->{templates}/$var : $!");
      unshift(@_, <INC>);
      close(INC);

      $self->{"$var"} = 1;

      next;
    }
    
    s/<%(.+?)%>/$self->{$1}/g;
    print OUT;
  }

  close(OUT);


  # { Moritz Bunkus
  # Convert the tex file to postscript
  if ($self->{format} =~ /(postscript|pdf)/) {

    $self->{pwd} = getcwd;
    chdir("$userspath") or $self->error($self->cleanup."chdir : $!");

    $self->{tmpfile} =~ s/$userspath\///g;

    # for printer we generate postscript
    if ($self->{media} eq 'printer') {
      system("latex --interaction=nonstopmode $self->{tmpfile} > $self->{tmpfile}.err");
      $self->error($self->cleanup) if ($?);
 
      $self->{tmpfile} =~ s/tex$/dvi/;
 
      system("dvips $self->{tmpfile} -o -q > /dev/null");
      $self->error($self->cleanup."dvips : $!") if ($?);
      $self->{tmpfile} =~ s/dvi$/ps/;
    }

    # DS. added screen and email option in addition to printer
    # screen
    if ($self->{media} =~ /(screen|email)/) {
      if ($self->{format} eq 'postscript') {
	system("latex --interaction=nonstopmode $self->{tmpfile} > $self->{tmpfile}.err");
	$self->error($self->cleanup) if ($?);
	
	$self->{tmpfile} =~ s/tex$/dvi/;
 
	system("dvips $self->{tmpfile} -o -q > /dev/null");
	$self->error($self->cleanup."dvips : $!") if ($?);
	$self->{tmpfile} =~ s/dvi$/ps/;
      }
      if ($self->{format} eq 'pdf') {
	system("pdflatex --interaction=nonstopmode $self->{tmpfile} > $self->{tmpfile}.err");
	$self->error($self->cleanup) if ($?);
	$self->{tmpfile} =~ s/tex$/pdf/;
      }
    }

  }

  if ($self->{format} =~ /(postscript|pdf)/ || $self->{media} eq 'email') {

    if ($self->{media} eq 'email') {
      
      use SL::Mailer;

      my $mail = new Mailer;
      
      $self->{email} =~ s/,/>,</g;
      
      map { $mail->{$_} = $self->{$_} } qw(cc bcc subject message version format charset);
      $mail->{to} = qq|"$self->{name}" <$self->{email}>|;
      $mail->{from} = qq|"$myconfig->{name}" <$myconfig->{email}>|;
      $mail->{fileid} = "$fileid.";

      # if we send html or plain text inline
      if (($self->{format} eq 'html') && ($self->{sendmode} eq 'inline')) {
	$mail->{contenttype} = "text/html";

        $mail->{message} =~ s/\r\n/<br>\n/g;
	$myconfig->{signature} =~ s/\\n/<br>\n/g;
	$mail->{message} .= "<br>\n--<br>\n$myconfig->{signature}\n<br>";
	
	open(IN, $self->{tmpfile}) or $self->error($self->cleanup."$self->{tmpfile} : $!");
	while (<IN>) {
	  $mail->{message} .= $_;
	}

	close(IN);

      } else {
	
	@{ $mail->{attachments} } = ($self->{tmpfile});

	$myconfig->{signature} =~ s/\\n/\r\n/g;
	$mail->{message} .= "\r\n--\r\n$myconfig->{signature}";

      }
 
      my $err = $mail->send($out);
      $self->error($self->cleanup."$err") if ($err);
      
    } else {
      
      $self->{OUT} = $out;
      open(IN, $self->{tmpfile}) or $self->error($self->cleanup."$self->{tmpfile} : $!");
      
      if ($self->{OUT}) {
	open(OUT, $self->{OUT}) or $self->error($self->cleanup."$self->{OUT} : $!");
      } else {
	open(OUT, ">-") or $self->error($self->cleanup."$!: STDOUT");
	
	# launch application
	print qq|Content-Type: application/$self->{format}; name="$self->{tmpfile}"
Content-Disposition: filename="$self->{tmpfile}"

|;
        $self->{copies} = 1;
      }
     
     
      for my $i (1 .. $self->{copies}) {
	while (<IN>) {
	  print OUT $_;
	}
	seek IN, 0, 0;
      }

      close(OUT);
      close(IN);
    }

    $self->cleanup;

  }
  # } Moritz Bunkus

}


sub cleanup {
  my $self = shift;

  my @err = ();
  if (-f "$self->{tmpfile}.err") {
    open(FH, "$self->{tmpfile}.err");
    @err = <FH>;
    close(FH);
  }
  
  if ($self->{tmpfile}) {
    # strip extension
    $self->{tmpfile} =~ s/\.\w+$//g;
    my $tmpfile = $self->{tmpfile};
    unlink(<$tmpfile.*>);
  }

  chdir("$self->{pwd}");
  
  "@err";
  
}


sub format_string {
  my ($self, @fields) = @_;

  my $format = $self->{format};
  if ($self->{format} =~ /(postscript|pdf)/) {
    $format = 'tex';
  }

  my %replace = ( 'order' => { 'html' => [ quotemeta('\n'), '
' ],
                               'tex'  => [ '&', quotemeta('\n'), '
',
					   '\$', '%', '_', '#', quotemeta('^'),
					   '{', '}', '<', '>' ] },
                  'html' => {
                quotemeta('\n') => '<br>', '
' => '<br>'
		            },
	           'tex' => {
	        '&' => '\&', '\$' => '\$', '%' => '\%', '_' => '\_',
		'#' => '\#', quotemeta('^') => '\^\\', '{' => '\{', '}' => '\}',
		'<' => '$<$', '>' => '$>$',
		quotemeta('\n') => '\newline ', '
' => '\newline '
                            }
	        );

  foreach my $key (@{ $replace{order}{$format} }) {
    map { $self->{$_} =~ s/$key/$replace{$format}{$key}/g; } @fields;
  }

}


sub datetonum {
  my ($self, $date, $myconfig) = @_;

  if ($date) {
    # get separator
    my $spc = $myconfig->{dateformat};
    $spc =~ s/\w//g;
    $spc = substr($spc, 1, 1);

    if ($spc eq '.') {
      $spc = '\.';
    }
    if ($spc eq '/') {
      $spc = '\/';
    }

    if ($myconfig->{dateformat} =~ /^yy/) {
      ($yy, $mm, $dd) = split /$spc/, $date;
    }
    if ($myconfig->{dateformat} =~ /^mm/) {
      ($mm, $dd, $yy) = split /$spc/, $date;
    }
    if ($myconfig->{dateformat} =~ /^dd/) {
      ($dd, $mm, $yy) = split /$spc/, $date;
    }
    
    $dd *= 1;
    $mm *= 1;
    $yy = ($yy < 70) ? $yy + 2000 : $yy;
    $yy = ($yy >= 70 && $yy <= 99) ? $yy + 1900 : $yy;

    $dd = "0$dd" if ($dd < 10);
    $mm = "0$mm" if ($mm < 10);
    
    $date = "$yy$mm$dd";
  }

  $date;
  
}



# Database routines used throughout

sub dbconnect {
  my ($self, $myconfig) = @_;

  # connect to database
  my $dbh = DBI->connect($myconfig->{dbconnect}, $myconfig->{dbuser}, $myconfig->{dbpasswd}) or $self->dberror;

  # set db options
  if ($myconfig->{dboptions}) {
    $dbh->do($myconfig->{dboptions}) || $self->dberror($myconfig->{dboptions});
  }

  $dbh;

}


sub dbconnect_noauto {
  my ($self, $myconfig) = @_;

  # connect to database
  $dbh = DBI->connect($myconfig->{dbconnect}, $myconfig->{dbuser}, $myconfig->{dbpasswd}, {AutoCommit => 0}) or $self->dberror;

  # set db options
  if ($myconfig->{dboptions}) {
    $dbh->do($myconfig->{dboptions}) || $self->dberror($myconfig->{dboptions});
  }

  $dbh;

}

sub update_field {
  my ($self, $dbh, $table, $field, $where, $value) = @_;

  # if we have a value, go do it
  if ($value != 0) {
    # retrieve balance from table

    # update balance
    $query = "UPDATE $table SET $field = $value WHERE $where";
    $dbh->do($query) || $self->dberror($query);
  #  $self->dberror($query);
  }
}


sub update_balance {
  my ($self, $dbh, $table, $field, $where, $value) = @_;

  # if we have a value, go do it
  if ($value != 0) {
    # retrieve balance from table
    my $query = "SELECT $field FROM $table WHERE $where";
    my $sth = $dbh->prepare($query);

    $sth->execute || $self->dberror($query);
    my ($balance) = $sth->fetchrow_array;
    $sth->finish;

    $balance += $value;
    # update balance
    $query = "UPDATE $table SET $field = $balance WHERE $where";
    $dbh->do($query) || $self->dberror($query);
  #  $self->dberror($query);
  }
}



sub update_exchangerate {
  my ($self, $dbh, $curr, $transdate, $buy, $sell) = @_;

  # some sanity check for currency
  return if ($curr eq '');

  my $query = qq|SELECT curr FROM exchangerate
                 WHERE curr = '$curr'
	         AND transdate = '$transdate'|;
  my $sth = $dbh->prepare($query);
  $sth->execute || $self->dberror($query);
  
  my $set;
  if ($buy != 0 && $sell != 0) {
    $set = "buy = $buy, sell = $sell";
  } elsif ($buy != 0) {
    $set = "buy = $buy";
  } elsif ($sell != 0) {
    $set = "sell = $sell";
  }
  
  if ($sth->fetchrow_array) {
    $query = qq|UPDATE exchangerate
                SET $set
		WHERE curr = '$curr'
		AND transdate = '$transdate'|;
  } else {
    $query = qq|INSERT INTO exchangerate (curr, buy, sell, transdate)
                VALUES ('$curr', $buy, $sell, '$transdate')|;
  }
  $sth->finish;
  $dbh->do($query) || $self->dberror($query);
  
}


sub get_exchangerate {
  my ($self, $dbh, $curr, $transdate, $fld) = @_;
  
  my $query = qq|SELECT $fld FROM exchangerate
                 WHERE curr = '$curr'
		 AND transdate = '$transdate'|;
  my $sth = $dbh->prepare($query);
  
 # $self->dberror($query);

  $sth->execute || $self->dberror($query);

  my ($exchangerate) = $sth->fetchrow_array;
  $sth->finish;

  ($exchangerate) ? $exchangerate : 1;

}


sub delete_exchangerate {
  my ($self, $dbh) = @_;

  my @transdate = ();
  my $transdate;

  my $query = qq|SELECT DISTINCT ON (transdate) transdate
                 FROM acc_trans
		 WHERE trans_id = $self->{id}|;
  my $sth = $dbh->prepare($query);
  $sth->execute || $self->dberror($query);

  while ($transdate = $sth->fetchrow_array) {
    push @transdate, $transdate;
  }
  $sth->finish;

  $query = qq|SELECT transdate FROM acc_trans
              WHERE ar.id = trans_id
	      AND ar.curr = '$self->{currency}'
	      AND transdate IN
	          (SELECT transdate FROM acc_trans
		  WHERE trans_id = $self->{id})
              AND trans_id != $self->{id}
        UNION SELECT transdate FROM acc_trans
	      WHERE ap.id = trans_id
	      AND ap.curr = '$self->{currency}'
	      AND transdate IN
	          (SELECT transdate FROM acc_trans
		  WHERE trans_id = $self->{id})
              AND trans_id != $self->{id}
        UNION SELECT transdate FROM oe
	        WHERE oe.curr = '$self->{currency}'
		AND transdate IN
		    (SELECT transdate FROM acc_trans
		    WHERE trans_id = $self->{id})|;
  $sth = $dbh->prepare($query);
  $sth->execute || $self->dberror($query);

  while ($transdate = $sth->fetchrow_array) {
    @transdate = grep !/^$transdate$/, @transdate;
  }
  $sth->finish;

  foreach $transdate (@transdate) {
    $query = qq|DELETE FROM exchangerate
                WHERE curr = '$self->{currency}'
		AND transdate = '$transdate'|;
    $dbh->do($query) || $self->dberror($query);
  }
  
}


sub check_exchangerate {
  my ($self, $myconfig, $currency, $transdate, $fld) = @_;

  return "" unless $transdate;
  
  my $dbh = $self->dbconnect($myconfig);

  my $query = qq|SELECT $fld FROM exchangerate
                 WHERE curr = '$currency'
		 AND transdate = '$transdate'|;
  my $sth = $dbh->prepare($query);
  $sth->execute || $self->dberror($query);

  ($_) = $sth->fetchrow_array;
  $sth->finish;
  $dbh->disconnect;
  
  $_ = "" unless $_;
  
  $_;
  
}


# the selection sub is used in the AR, AP, IS, IR and OE module
#
sub all_vc {
  my ($self, $myconfig, $table,$code) = @_;

  # create array for vendor or customer
  my $dbh = $self->dbconnect($myconfig);

  my ($query, $sth);
  
  unless ($self->{"${table}_id"}) {
    my $arap = ($table eq 'customer') ? "ar" : "ap";
    $arap = 'oe' if ($self->{type} =~ /_order/);
    my %oid = ( 'Pg'		=> 'oid',
                'Oracle'	=> 'rowid',
		'DB2'		=> '' );

    $query = qq|SELECT ${table}_id FROM $arap
                WHERE $oid{$myconfig->{dbdriver}}
		= (SELECT max($oid{$myconfig->{dbdriver}}) FROM $arap
		             WHERE ${table}_id > 0) |;
    $sth = $dbh->prepare($query);
    $sth->execute || $self->dberror($query);

    unless (($self->{"${table}_id"}) = $sth->fetchrow_array) {
      $self->{"${table}_id"} = 0;
    }
    $sth->finish;
  }
  
  $query = qq|SELECT id, name,${table}number
  		FROM $table where ${table}code like '$code%'
	      ORDER BY ${table}number|;
  $sth = $dbh->prepare($query);
  $sth->execute || $self->dberror($query);
#$self->dberror($query);
 # my $ref = $sth->fetchrow_hashref(NAME_lc);
 # push @{ $self->{"all_$table"} }, $ref;

  while (my $ref = $sth->fetchrow_hashref(NAME_lc)) {
    push @{ $self->{"all_$table"} }, $ref;
  }

  $sth->finish;
  $dbh->disconnect;

}


sub create_links {
  my ($self, $module, $myconfig, $table,$code) = @_;

  # get all the customers or vendors
  &all_vc($self, $myconfig, $table,$code);

  my %xkeyref = ();

  my $dbh = $self->dbconnect($myconfig);
  # now get the account numbers
  my $query = qq|SELECT accno, description, link
                 FROM chart
		 WHERE link LIKE '%$module%'
		 ORDER BY accno|;
  my $sth = $dbh->prepare($query);
  $sth->execute || $self->dberror($query);

  while (my $ref = $sth->fetchrow_hashref(NAME_lc)) {

    foreach my $key (split(/:/, $ref->{link})) {
      if ($key =~ /$module/) {
	# cross reference for keys
	$xkeyref{$ref->{accno}} = $key;

	push @{ $self->{"${module}_links"}{$key} }, { accno => $ref->{accno},
                                       description => $ref->{description} };
      }
    }
  }
  $sth->finish;


  if ($self->{id}) {
    my $arap = ($table eq 'customer') ? 'ar' : 'ap';

    $query = qq|SELECT invnumber, transdate, ${table}_id, datepaid, duedate,
		ordnumber, taxincluded, curr AS currency, notes,transtype,batch,transno,subtransno,subb,project
		FROM $arap
		WHERE id = $self->{id}|;
    $sth = $dbh->prepare($query);
    $sth->execute || $self->dberror($query);

    $ref = $sth->fetchrow_hashref(NAME_lc);
    foreach $key (keys %$ref) {
      $self->{$key} = $ref->{$key};
    }
    $sth->finish;

    # get amounts from individual entries
    $query = qq|SELECT accno, description, source, amount, transdate, cleared
		FROM acc_trans, chart
		WHERE chart.id = acc_trans.chart_id
		AND trans_id = $self->{id}
		AND fx_transaction = '0'
		ORDER BY transdate|;
    $sth = $dbh->prepare($query);
    $sth->execute || $self->dberror($query);


    my $fld = ($module eq 'AR') ? 'buy' : 'sell';
    # get exchangerate for currency
    $self->{exchangerate} = $self->get_exchangerate($dbh, $self->{currency}, $self->{transdate}, $fld);
    
    # store amounts in {acc_trans}{$key} for multiple accounts
    while (my $ref = $sth->fetchrow_hashref(NAME_lc)) {
      $ref->{exchangerate} = $self->get_exchangerate($dbh, $self->{currency}, $ref->{transdate}, $fld);

      push @{ $self->{acc_trans}{$xkeyref{$ref->{accno}}} }, $ref;
    }

    $sth->finish;

    $query = qq|SELECT d.curr AS currencies, d.closedto, d.revtrans,
                  (SELECT c.accno FROM chart c
		   WHERE d.fxgain_accno_id = c.id) AS fxgain_accno,
                  (SELECT c.accno FROM chart c
		   WHERE d.fxloss_accno_id = c.id) AS fxloss_accno
		FROM defaults d|;
    $sth = $dbh->prepare($query);
    $sth->execute || $self->dberror($query);

    $ref = $sth->fetchrow_hashref(NAME_lc);
    map { $self->{$_} = $ref->{$_} } keys %$ref;
    $sth->finish;

  } else {
    # get date
    $query = qq|SELECT current_date AS transdate, current_date + 30 AS duedate,
                d.curr AS currencies, d.closedto, d.revtrans,
                  (SELECT c.accno FROM chart c
		   WHERE d.fxgain_accno_id = c.id) AS fxgain_accno,
                  (SELECT c.accno FROM chart c
		   WHERE d.fxloss_accno_id = c.id) AS fxloss_accno
		FROM defaults d|;
    $sth = $dbh->prepare($query);
    $sth->execute || $self->dberror($query);

    $ref = $sth->fetchrow_hashref(NAME_lc);
    map { $self->{$_} = $ref->{$_} } keys %$ref;
    $sth->finish;
  }

  $dbh->disconnect;

}


sub current_date {
  my ($self, $myconfig, $thisdate, $days) = @_;
  
  my $dbh = $self->dbconnect($myconfig);
  my $query = qq|SELECT current_date AS thisdate
                 FROM defaults|;

  $days *= 1;
  if ($thisdate) {
    $query = qq|SELECT date '$thisdate' + $days AS thisdate
                FROM defaults|;
  }
  
  my $sth = $dbh->prepare($query);
  $sth->execute || $self->dberror($query);

  ($thisdate) = $sth->fetchrow_array;
  $sth->finish;

  $dbh->disconnect;

  $thisdate;

}

sub first_day_current_Date {
  my ($self, $myconfig, $thisdate) = @_;
  
  my $dbh = $self->dbconnect($myconfig);
  my $query = qq|select to_date(extract(year from current_Date)\|\|'-'\|\|extract(month from current_date)\|\|'-'\|\|1 ,'yyyy-MM-dd') as thisdate|;

  my $sth = $dbh->prepare($query);
  $sth->execute || $self->dberror($query);

  ($thisdate) = $sth->fetchrow_array;
  $sth->finish;

  $dbh->disconnect;

  $thisdate;

}


sub like {
  my ($self, $string) = @_;
  
  unless ($string =~ /%/) {
    $string = "%$string%";
  }

  $string;
  
}


package Locale;


sub new {
  my ($type, $country, $NLS_file) = @_;
  my $self = {};

  ($country) = split /--/, $country;
  
  %self = ();
  if ($country && -d "locale/$country") {
    $self->{countrycode} = $country;
    eval { require "locale/$country/$NLS_file"; };
  }


  $self->{NLS_file} = $NLS_file;
  
  push @{ $self->{LONG_MONTH} }, ("January", "February", "March", "April", "May ", "June", "July", "August", "September", "October", "November", "December");
  push @{ $self->{SHORT_MONTH} }, (qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec));
  
  bless $self, $type;

}


sub text {
  my ($self, $text) = @_;
  
  return (exists $self{texts}{$text}) ? $self{texts}{$text} : $text;
  
}


sub findsub {
  my ($self, $text) = @_;
#Form->error($self{subs}{$text}."asdfdsfda");
  if (exists $self{subs}{$text}) {
    $text = $self{subs}{$text};
  } else {
    if ($self->{countrycode} && $self->{NLS_file}) {
      Form->error("$text not defined in locale/$self->{countrycode}/$self->{NLS_file}");
    }
  }

  $text;

}


sub date {
  my ($self, $myconfig, $date, $longformat) = @_;

  my $longdate = "";
  my $longmonth = ($longformat) ? 'LONG_MONTH' : 'SHORT_MONTH';

  if ($date) {
    # get separator
    $spc = $myconfig->{dateformat};
    $spc =~ s/\w//g;
    $spc = substr($spc, 1, 1);

    if ($spc eq '.') {
      $spc = '\.';
    }
    if ($spc eq '/') {
      $spc = '\/';
    }

    if ($myconfig->{dateformat} =~ /^yy/) {
      ($yy, $mm, $dd) = split /$spc/, $date;
    }
    if ($myconfig->{dateformat} =~ /^MM/) {
      ($mm, $dd, $yy) = split /$spc/, $date;
    }
    if ($myconfig->{dateformat} =~ /^dd/) {
      ($dd, $mm, $yy) = split /$spc/, $date;
    }
    
    $dd *= 1;
    $mm--;
    $yy = ($yy < 70) ? $yy + 2000 : $yy;
    $yy = ($yy >= 70 && $yy <= 99) ? $yy + 1900 : $yy;

    if ($myconfig->{dateformat} =~ /^dd/) {
      $longdate = "$dd. ".&text($self, $self->{$longmonth}[$mm])." $yy";
    } else {
      $longdate = &text($self, $self->{$longmonth}[$mm])." $dd, $yy";
    }

  }

  $longdate;

}


1;

