/* $PostgresPy: if/src/tupd.c,v 1.28 2004/07/27 16:15:30 flaw Exp $
 * 
 * † Instrument:
 *     Copyright 2004, rhid development. All Rights Reserved.
 *     
 *     Usage of the works is permitted provided that this
 *     instrument is retained with the works, so that any entity
 *     that uses the works is notified of this instrument.
 *     
 *     DISCLAIMER: THE WORKS ARE WITHOUT WARRANTY.
 *     
 *     [2004, Fair License; rhid.com/fair]
 *     
 * Description:
 *    The TupleDesc Python type implementation.
 */
#include <setjmp.h>
#include <pputils.h>

#include <postgres.h>
#include <tcop/tcopprot.h>
#include <access/htup.h>
#include <access/hio.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <utils/array.h>
#include <utils/palloc.h>
#include <utils/memutils.h>
#include <utils/rel.h>
#include <PGExcept.h>

#include <Python.h>
#include <structmember.h>
#include <abstract.h>
#include <py.h>

#include "datum.h"
#include "tupd.h"
#include "tup.h"
#include "type.h"
#include "obj.h"
#include "rel.h"
#include "module.h"
#include "utils.h"


const char PyPgTupleDesc_Doc[] = "Postgres Tuple Descriptor";

HeapTuple
PgTupD_AttTuple(PgTupD self, long attnum)
{
	HeapTuple aht;

	Assert(self != NULL && attnum >= 0);

	aht = heap_addheader(Natts_pg_attribute, true,
		sizeof(FormData_pg_attribute), PgTupD_FetchTD(self)->attrs[attnum]);

	return(aht);
}

long
PgTupD_AttOffset(PgTupD self, PyObj ob)
{
	TupleDesc tupd;
	long i = -1;

	Assert(self != NULL && ob != NULL);
	
	tupd = PgTupD_FetchTD(self);
	Assert(tupd != NULL);

	if (PyObject_TypeCheck(ob, &PyInt_Type))
		i = PYASINT(ob);
	else if (PyObject_TypeCheck(ob, &PyLong_Type))
		i = PyLong_AsLong(ob);
	else if (PgObj_TypeCheck(ob))
	{
		switch (PgObj_FetchTypeOid(ob))
		{
			case INT4OID:
			case INT2OID:
				i = (long)PgObj(ob)->ob_datum;
			break;
		}
	}
	else
	{
		PyObj obstr = NULL;

		if (PyObject_TypeCheck(ob, &PyString_Type))
			obstr = ob;
		else
			obstr = PyObject_Str(ob);

		if (PyObject_Size(obstr) < NAMEDATALEN)
		{
			char *str = NULL;
			str = PYASSTR(obstr);
			for (i=0; i <= (tupd->natts-1); ++i)
			{
				if (strcmp(NameStr(tupd->attrs[i]->attname), str) == 0)
					break;
			}
		}
		
		if (obstr != ob) DECREF(obstr);
	}

	if (i >= tupd->natts)
		i = -1;
	
	return(i);
}

static int
tupd_length(PgTupD self)
{
	return(PgTupD_FetchTD(self)->natts);
}

static PyObj
tupd_item(PgTupD self, int attr)
{
	PyObj rob;
	HeapTuple cht;

	if (attr >= tupd_length(self))
	{
		PyErr_Format(PyExc_IndexError,
			"Postgres.TupleDesc index(%d) is out of range", attr);
		return(NULL);
	}

	cht = PgTupD_AttTuple(self, attr);
	cht = heap_copytuple(cht);
	rob = PyPgTuple_New(PgTupD_AttTuple(self, attr), PyPgAttrTDO);
	return(rob);
}

static PyObj
tupd_slice(PgTupD self, int from, int to)
{
	PyObj rob;
	int c;

	rob = PyTuple_New(to - from);

	for (c = from; c < to; ++c)
	{
		PyTuple_SetItem(rob, c - from, tupd_item(self, c));
	}

	return(rob);
}

static int
tupd_contains(PgTupD self, PyObj ob)
{
	if (PgTupD_AttOffset(self, ob) >= 0)
		return(1);
	
	return(0);
}

static PySequenceMethods PyPgTupleDescAsSequence = {
	(inquiry)tupd_length,					/* sq_length */
	(binaryfunc)NULL,							/* sq_concat */
	(intargfunc)NULL,							/* sq_repeat */
	(intargfunc)tupd_item,					/* sq_item */
	(intintargfunc)tupd_slice,				/* sq_slice */
	(intobjargproc)NULL,						/* sq_ass_item */
	(intintobjargproc)NULL,					/* sq_ass_slice */
	(objobjproc)tupd_contains,				/* sq_contains */
	(binaryfunc)NULL,							/* sq_inplace_concat */
	(intargfunc)NULL,							/* sq_inplace_repeat */
};

static PyObj
tupd_subscript(PgTupD self, PyObj ob)
{
	int off;
	off = PgTupD_AttOffset(self, ob);
	
	return(tupd_item(self, off));
}

static PyMappingMethods PyPgTupleDescAsMapping = {
	(inquiry)tupd_length,					/* mp_length */
	(binaryfunc)tupd_subscript,			/* mp_subscript */
	(objobjargproc)NULL,						/* mp_ass_subscript */
};

static PyMethodDef PyPgTupleDesc_Methods[] = {
	//{"name", (PyCFunction), flags, "docstring"},
	{NULL}
};

static void
tupd_dealloc(PgTupD self)
{
	FreeTupleDesc((TupleDesc)self->ob_datum);
	PyObject_Del((PyObj)self);
}

static int
tupd_compare(PgTupD self, PyObj ob)
{
	TupleDesc pri, sec;

	if (!PgTupD_TypeCheck(ob))
		return(-1);

	pri = PgTupD_FetchTD(self);
	sec = PgTupD_FetchTD(ob);

	if (pri->natts > sec->natts)
		return(1);
	else if (pri->natts < sec->natts)
		return(-1);
	else if (equalTupleDescs(pri, sec))
		return(0);
		
	return(-1);
}

static PyObj
tupd_str(PgTupD self)
{
	PyObj rob;
	TupleDesc td;
	unsigned long i, natts;
	
	td = PgTupD_FetchTD(self);
	natts = td->natts;

	for (i = 0; i < natts; ++i)
	{
		Form_pg_attribute att;
		Oid atttypid;
		char *str = NULL;

		if (i == 0)
			rob = PYSTR("");
		else
			PyString_ConcatAndDel(&rob, PYSTR(" "));

		att = td->attrs[i];
		atttypid = att->atttypid;

		str = NameStr(att->attname);
	
		if (strlen(str))
		{
			PyString_ConcatAndDel(&rob, PYSTR(str));
			PyString_ConcatAndDel(&rob, PYSTR(" "));
		}
		
		str = PgTypeName_FromOid(atttypid);
		PyString_ConcatAndDel(&rob, PYSTR(str));
		pfree(str);

		if (att->atttypmod > 0)
		{
			char *tmod = NULL;
			asprintf(&tmod, "(%d)", att->atttypmod);
			PyString_ConcatAndDel(&rob, PYSTR(tmod));
			free(tmod);
		}

		if (att->attndims > 0)
		{
			char *dims = NULL;
			asprintf(&dims, "[%d]", att->attndims);
			PyString_ConcatAndDel(&rob, PYSTR(dims));
			free(dims);
		}

		if (att->attnotnull)
			PyString_ConcatAndDel(&rob, PYSTR(" NOT NULL"));

		if (att->atthasdef)
		{
			AttrDefault *defval;
			uint16 num_defval;
			int ii;

			defval = td->constr->defval;
			num_defval = td->constr->num_defval;
			for (ii = 0; ii < num_defval; ++ii)
			{
				if (defval[ii].adnum == i)
					break;
			}

			if (ii < num_defval)
			{
				char *def;
				asprintf(&def, " DEFAULT '%s'", defval[ii].adbin);
				PyString_ConcatAndDel(&rob, PYSTR(def));
				free(def);
			}
		}

		if (i+1 < natts)
			PyString_ConcatAndDel(&rob, PYSTR(","));
	}

	return(rob);
}

static PyObj
tupd_repr(PgTupD self)
{
	PyObj rob;
	rob = PYSTR("( ");
	PyString_ConcatAndDel(&rob, tupd_str(self));
	PyString_ConcatAndDel(&rob, PYSTR(" )"));
	return(rob);
}

/*
 * tupd_call - create a Postgres.Tuple based on the descriptor
 */
static PyObj
tupd_call(PgTupD self, PyObj args, PyObj kw)
{
	TupleDesc td;
	int i, natts, slen;
	Datum *datums;
	bool *nulls;
	bool *frees; /* Explicit marker for datums that need to be freed */
	PyObj ob;
	HeapTuple ht;
	PyObj rob;

	if (PyCFunctionErr_NoKeywordsAllowed(kw))
		return(NULL);

	td = PgTupD_FetchTD(self);
	natts = td->natts;

	if ((slen = PySequence_Length(args)) != natts)
	{
		PyErr_Format(PyExc_TypeError,
			"This TupleDesc has exactly %d attributes, got %d for tuple",
			natts, slen
		);
		return(NULL);
	}
	slen = 0;

	datums = palloc(sizeof(Datum) * natts);
	nulls = palloc(sizeof(bool) * natts);
	frees = palloc(sizeof(bool) * natts);

	/* make the datums for the tuple */
	for (i = 0; i < natts; ++i)
	{
		Form_pg_attribute att;
		att = td->attrs[i];

		ob = PySequence_GetItem(args, i);
		if (ob == PyPgDEFAULT)
		{
			/* XXX: evaluate the default expr */
			datums[i] = 0;
			nulls[i] = true;
			frees[i] = false;
		}
		else if (PyObject_Compare(ob, Py_None) == 0 ||
					(PgObj_TypeCheck(ob) && PgObj_IsNull(ob)))
		{
			datums[i] = 0;
			nulls[i] = true;
			frees[i] = false;
		}
		else
		{
			nulls[i] = false;
			if (PgObj_TypeCheck(ob))
			{
				if (PgObj_FetchTypeOid(ob) != att->atttypid)
				{
					PyErr_Format(PyExc_TypeError,
						"Expecting %d type for attribute %d, got %d",
						(int)att->atttypid, i, (int)PgObj_FetchTypeOid(ob)
					);

					slen = i;
				}
				frees[i] = false;
				datums[i] = PgObj(ob)->ob_datum;
			}
			else
			{
				datums[i] = Datum_FromPyObjectAndTypeOid(ob, att->atttypid);
				if (att->attbyval)
					frees[i] = false;
				else
					frees[i] = true;
				nulls[i] = false;
			}
		}
	}

	/* No error occurred if slen == 0 */
	if (slen == 0)
	{
		ht = heap_formtuple(td, datums, nulls);
		rob = PyPgTuple_New(ht, PyObj(self));
	}
	else
	{
		rob = NULL;		/* an error occurred */
		natts = slen;	/* limit the clean up to those datums created */
	}

	/* free up allocated memory; formtuple create a copy, so obliterate ours */
	for (i = 0; i < natts; ++i)
	{
		if (frees[i])
			pfree(DatumGetPointer(datums[i]));
	}
	pfree(datums);
	pfree(nulls);
	pfree(frees);

	return(rob);
}

static PyObj
tupd_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj rob;
	PgTupD nd;
	
	nd = (PgTupD)subtype->tp_alloc(subtype, 0);

	/* XXX: fill in custom tuple descriptor creation  */

	rob = PyObj(nd);
	return(rob);
}

PyTypeObject PyPgTupleDesc_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.TupleDesc",			/* tp_name */
	sizeof(struct PyPgTupleDesc),	/* tp_basicsize */
	NULL,									/* tp_itemsize */
	(destructor)tupd_dealloc,		/* tp_dealloc */
	NULL,									/* tp_print */
	(getattrfunc)NULL,				/* tp_getattr */
	(setattrfunc)NULL,				/* tp_setattr */
	(cmpfunc)tupd_compare,			/* tp_compare */
	(reprfunc)tupd_repr,				/* tp_repr */
	NULL,									/* tp_as_number */
	&PyPgTupleDescAsSequence,		/* tp_as_sequence */
	&PyPgTupleDescAsMapping,		/* tp_as_mapping */
	(hashfunc)NULL,					/* tp_hash */
	(ternaryfunc)tupd_call,			/* tp_call */
	(reprfunc)tupd_str,				/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	(char*)PyPgTupleDesc_Doc,		/* tp_doc */
	(traverseproc)NULL,				/* tp_traverse */
	(inquiry)NULL,						/* tp_clear */
	(richcmpfunc)NULL,				/* tp_richcompare */
	(long)0,								/* tp_weaklistoffset */
	(getiterfunc)PySeqIter_New,	/* tp_iter */
	(iternextfunc)NULL,				/* tp_iternext */
	PyPgTupleDesc_Methods,			/* tp_methods */
	NULL,									/* tp_members */
	NULL,									/* tp_getset */
	&PyPgDatum_Type,					/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	NULL,									/* tp_dictoffset */
	(initproc)NULL,					/* tp_init */
	NULL,									/* tp_alloc */
	(newfunc)tupd_new,				/* tp_new */
};

PyObj
PyPgTupleDesc_New(TupleDesc tupd)
{
	PgTupD tdo;

	tdo = (PgTupD)PgTupD_NEW();
	tdo->ob_datum = (Datum)tupd;

	return(PyObj(tdo));
}

PyObj
PgTupD_FromRelation(Relation rel)
{
	PyObj rob;
	rob = PyPgTupleDesc_New(CreateTupleDescCopyConstr(RelationGetDescr(rel)));
	return(rob);
}

PyObj
PyPgTupleDesc_FromRelationOid(Oid ro)
{
	/* XXX: Implement Oid based desciptor cache */
	return(PyPgTupleDesc_New(RelationOidGetDescr(ro)));
}
