/*
 *  PostgreSQLProtocol.h
 *  PostgresSQL
 *
 *  Created by Pascal on Thu Dec 19 2002.
 *  Copyright (c) 2002 P3 Consulting. All rights reserved.
 *
 */

#import <Cocoa/Cocoa.h>
#import "libpq-fe.h"
#import "PgSQLListener.h"

/*!
@header PostgreSQLProtocol.h

 P3 Consulting, 2003 All Rights Reserved<br><br>
 This is a Cocoa-wrapper around the libcpg functions for connecting to a PostgreSQL database.
 
 As some of you will notice most of the following API is designed to be 
 - as far as possible - source code compatible with OpenBase Basic API.<br>
 This is due to the simple fact that I also used OpenBase and want to migrate as easely 
 as possible some of my own code to PostgreSQL. 
 <br>
 In order to support uniqueRowIdForTable functionnality of OpenBase, you have to had a 
	<br><code>_rowid BIGSERIAL NOT NULL UNIQUE PRIMARY KEY</code><br>
 field to every table you plan to use with uniqueRowIdForTable.
 <br><br>
 Differences with OpenBase API:
	markRow and removeMarkOnRow are not implemented - <br>
		LOCKING strategy is not impemented that way in PostgreSQL
		please refer to PostgreSQL doc and use executeCommand to
		achieve desired functionality
	<br><br>
	databaseEncodingName is not implemented
<br><br>	
	notification mechanism is different <br>
		we use <code>LISTEN</code> (and <code>UNLISTEN</code> to disable a notification)
		to register a PgSQLListener object that will be called
		by an NSTimer at a user specified interval, if a PostgreSQL
		notification is detected at that time, the pgSQLNotify method
		(defined in PgSQLListener protocol) of a user specified object 
		will be called.
<br><br> 
	large object support has different - but close - API<br>
	-	Oid are used instead of char[] to hold large object reference<br>
	-	more functionnality is available including reading and 
		writing part of large object.
<br><br>		
 ****************** DEMO WARNING ******************<br>
	demonstration version limited to:<br>
	no more than 50 records are returned from any query<br>
	methods affected are<br>
		nextRow: returns false when 50th record is reached<br>
		afterLastRow: position cursor on 50th record<br>
		resultAsDictionary: only the first 50 rows are returned<br>
<br><br>
	Be careful: rowsAffected method returns the REAL result count from the query<br>
<br><br>		
	Methods not implemented:<br>
	beginTransaction, endTransaction, rollbackTransaction<br>
 ****************** END DEMO WARNING ******************<br>
 <br>
	PostgreSQL limitation:<br>
		resultTableName is not implemented (returns <code>"NOT_SUPPORTED_BY_POSTGRESQL"</code>)<br>
		I don't have found yet a way to returns the name of the originating table of a query's resulting column<br> 
*/

//	we copy paste part of geo_decls because including the file itself causes too much trouble
//	we only need the struct definitions

@class	PgASyncThread	;

typedef struct
{
	int		size	;
	BOOL	*bits	;
}
	BOOLARRAY	;
	
typedef struct
{
	int				size	;
	unsigned char	*bytes	;
}
	BSTRING	;
	
typedef struct
{
	int		size	;
	char	*text	;
}
	TEXT	;
	
typedef int int32	;
#define VARHDRSZ                ((int32) sizeof(int32))

typedef	struct
{
	double	x, y	;
}
	POINT	;
	

/*---------------------------------------------------------------------
 * LSEG - A straight line, specified by endpoints.
 *-------------------------------------------------------------------*/
typedef struct
{
	POINT		p[2];

	double		m;				/* precomputed to save time, not in tuple */
} LSEG;


/*---------------------------------------------------------------------
 * PATH - Specified by vertex points.
 *-------------------------------------------------------------------*/
typedef struct
{
	int32		size;			/* XXX varlena */
	int32		npts;
	int32		closed;			/* is this a closed polygon? */
	int32		dummy;			/* padding to make it double align */
	POINT		p[1];			/* variable length array of POINTs */
} PATH;


/*---------------------------------------------------------------------
 * LINE - Specified by its general equation (Ax+By+C=0).
 *		If there is a y-intercept, it is C, which
 *		 incidentally gives a freebie point on the line
 *		 (if B=0, then C is the x-intercept).
 *		Slope m is precalculated to save time; if
 *		 the line is not vertical, m == A.
 *-------------------------------------------------------------------*/
typedef struct
{
	double		A,
				B,
				C;

	double		m;
} LINE;


/*---------------------------------------------------------------------
 * BOX	- Specified by two corner points, which are
 *		 sorted to save calculation time later.
 *-------------------------------------------------------------------*/
typedef struct
{
	POINT		high,
				low;			/* corner POINTs */
} BOX;

/*---------------------------------------------------------------------
 * POLYGON - Specified by an array of doubles defining the points,
 *		keeping the number of points and the bounding box for
 *		speed purposes.
 *-------------------------------------------------------------------*/
typedef struct
{
	int32		size;			/* XXX varlena */
	int32		npts;
	BOX			boundbox;
	POINT		p[1];			/* variable length array of POINTs */
} POLYGON;

/*---------------------------------------------------------------------
 * CIRCLE - Specified by a center point and radius.
 *-------------------------------------------------------------------*/
typedef struct
{
	POINT		center;
	double		radius;
} CIRCLE;


/*!
@const	PgSQLColumnsKeyField
@discussion key for columns array
*/
extern	NSString	*PgSQLColumnsKeyField	;
/*!
@const	PgSQLNameKeyField key for column's name field (string)<br>
*/
extern	NSString	*PgSQLNameKeyField		;
/*!
@const	PgSQLTypeKeyField key for column's type field (integer)<br>
*/
extern	NSString	*PgSQLTypeKeyField		;
/*!
@const	PgSQLTypeNameKeyField key for column's type field (integer)<br>
*/
extern	NSString	*PgSQLTypeNameKeyField	;
/*!
@const	PgSQLRowsKeyField key for fields array
 */
extern	NSString	*PgSQLRowsKeyField	;
/*!
@const	PostgreSQLNotification
 */
extern	NSString *PostgreSQLNotification	;
/*!
@const	PostgreSQLResultAvailable
 */
extern	NSString *PostgreSQLResultAvailable	;

/*!
@const	PgSQLuserInfoNotificationField
 */
extern	NSString *PgSQLuserInfoNotificationField	;
/*!
@const	PgSQLconditionNotificationField
 */
extern	NSString *PgSQLconditionNotificationField	;
/*!
@const	PgSQLconnectionNotificationField
 */
extern	NSString *PgSQLconnectionNotificationField	;

@protocol PostgreSQLProtocol

/*! 
   @method setDebug
   @abstract turns on|off global debugging. 
   @discussion Newly allocated PostgreSQL objects have their instance debug flag set the the global one.
   @param yn  as you expect.
*/
+ (void)setDebug:(BOOL)yn;

/*! 
   @method setDebugMode
   @abstract turns on|off instance debugging. 
   @param turnOn  as you expect.
*/
- (void)setDebugMode:(BOOL)turnOn;

/*! 
   @method startTracing
   @abstract start tracing to file traceFileName. 
   @param traceFileName full path name of file to dump tracing information.
   @result YES if successfull
*/
-(BOOL)startTracing:(const char *)traceFileName;

/*! 
   @method stopTracing
   @abstract turns off tracing. 
*/
-(void)stopTracing;

/*! 
   @method connectToDatabase
   @abstract specify parameters to start PostgreSQL database connection 
   @discussion The wrapper is disconnected from any opened database before establishing a new one.<br>
   Passing nil or empty string to any of the parameters will fail.<br>
   Use <code>(BOOL)connectToDatabase:(const char *)connectInfo return:(int *)returnCode</code> if you need to pass different arguments like options or tty or if you want to use less arguments and let PostgreSQL use default ones.
   @param dbName	databasename as defined in PostgreSQL.
   @param hostName	network name of host computer
   @param loginName user name for login
   @param password user password for login
   @param returnCode where to return int result code for connection
   @result YES if connection successful No otherwise
*/
- (BOOL)connectToDatabase:(const char *)dbName onHost:(const char *)hostName login:(const char *)loginName password:(const char *)password return:(int *)returnCode;


/*! 
	@method connectToDatabase
	@abstract specify parameters string to start PostgreSQL database connection. See PostgreSQL doc for more details on param string.
	@param connectInfo character buffer holding the conenction string to be used.
	@param return pointer an integer buffer o hold result code from backend.
	@result YES if connection successfull, NO otherwise. 
*/
- (BOOL)connectToDatabase:(const char *)connectInfo return:(int *)returnCode;
/*! 
   @method connectAskingUser
   @abstract present a dialog to let the user fill in fields to connect to a databse.
   @param pointer an integer buffer o hold result code from backend.
   @result YES if connection successfull, NO otherwise. 
*/
- (BOOL)connectAskingUser:(int *)returnCode;

/*!
   @method uniqueID
   @abstract if you need to insert the PostgreSQL object in a NSDictionary, <br>
	this is a convenience method that is guaranted to return a unique string usable as key object.
	Uniqueness is guaranted for the lifetime of the program duration.
   @result	unique id string.
*/
-(NSString *)uniqueID;

/*! 
   @method postgreSQLVersion
   @abstract returns version of the running postmaster. 
   @discussion returns reulst of executing the <code>SELECT version()</code> query.
   @result	NSString containing the postgreSQL version.
*/
- (NSString *)postgreSQLVersion;

/*! 
   @method frameworkVersion
   @abstract returns version of the framework. 
   @result	NSString describing the framework version.
*/
- (NSString *)frameworkVersion;
/*! 
   @method databaseName
   @abstract returns database name of the connection. 
   @result	pointer on database name string buffer.
*/
-(const char *)databaseName;
/*! 
   @method hostName
   @abstract returns host name of the connection. 
   @result	pointer on host name string buffer.
*/
-(const char *)hostName;
/*! 
   @method loginName
   @abstract returns user name of the connection. 
   @result	pointer on user name string buffer.
*/
-(const char *)loginName;
/*! 
   @method password
   @abstract returns password of the connection. 
   @result	pointer on password string buffer.
*/
-(const char *)password;
/*! 
   @method port
   @abstract returns port of the connection. 
   @result	pointer on port string buffer.
*/
-(const char *)port;
/*! 
   @method tty
   @abstract returns tty of the connection. 
   @result	pointer on tty string buffer.
*/
-(const char *)tty;
/*! 
   @method options
   @abstract returns options of the connection. 
   @result	pointer on options string buffer.
*/
-(const char *)options;
/*! 
   @method socket
   @abstract returns socket of the database server. 
   @result	socket number.
*/
-(int)socket;
/*! 
   @method backendPID
   @abstract returns backend process id of the database server. 
   @result	backend process id.
*/
-(int)backendPID;
/*! 
   @method getPGconn
   @abstract returns pointer on PGconn record. 
   @discussion ALWAYS returns NIL in DEMO version. See PostgreSQL for advanced use.<br>
   You should never close yourselve a connection by calling PQfinish([pgconn getPGconn]).
   @result	pointer on PGconn record.
*/
-(PGconn *)getPGconn;

/*! 
   @method delegate
*/
-(id) delegate;

/*! 
   @method setDelegate
*/
-(void)setDelegate:(id)newDelegate;

/*! 
   @method disconnect
   @abstract disconnect cuurently opened connection. 
   @discussion Not strictly necessary in the API since connectTodatabase disconnects automatically from any previously opened connection.
*/
-(void)disconnect;

/*! 
   @method clientEncoding
   @result	client encoding id.
*/
-(int)clientEncoding;
/*! 
   @method setClientEncoding
   @param encoding name
*/
-(int)setClientEncoding:(const char *)encoding;

/*! 
   @method bindDouble
   @abstract bindDouble binds the double variable <code>var</code> to the next result column.
*/
-(void)bindDouble:(double *)var;
/*! 
   @method bindDouble
   @abstract bindDouble binds the double variable <code>var</code> to the specific column <code>col</code>. (column count starts at 0)
   @discussion In case you skip columns in binding, they will be binded internally to a dummy item in order to avoid crashes or erroneous behavior.
*/
-(void)bindDouble:(double *)var column:(int)col;
/*! 
   @method bindFloat
*/
-(void)bindFloat:(float *)var;
/*! 
   @method bindFloat
*/
-(void)bindFloat:(float *)var column:(int)col;
/*! 
   @method bindShort
*/
-(void)bindShort:(short *)var;
/*! 
   @method bindShort
*/
-(void)bindShort:(short *)var column:(int)col;
/*! 
   @method bindInteger
*/
-(void)bindInteger:(int *)var;
/*! 
   @method bindInteger
*/
-(void)bindInteger:(int *)var column:(int)col;
/*! 
   @method bindBoolean
   @discussion Y,y,t,T,1 are converted to YES, any other character to NO
*/
-(void)bindBoolean:(BOOL *)var;
/*! 
   @method bindBoolean
*/
-(void)bindBoolean:(BOOL *)var column:(int)col;
/*! 
   @method bindLong
*/
-(void)bindLong:(long *)var;
/*! 
   @method bindLong
*/
-(void)bindLong:(long *)var column:(int)col;
/*! 
   @method bindLongLong
*/
-(void)bindLongLong:(long long *)var;
/*! 
   @method bindLongLong
*/
-(void)bindLongLong:(long long *)var column:(int)col;
/*! 
   @method bindString
*/
-(void)bindString:(char *)var;
/*! 
   @method bindString
*/
-(void)bindString:(char *)var column:(int)col;
/*! 
   @method bindBinaryString
*/
-(void)bindBinaryString:(BSTRING *)var;
/*! 
   @method bindBinaryString
*/
-(void)bindBinaryString:(BSTRING *)var column:(int)col;
/*! 
   @method bindText
*/
-(void)bindText:(TEXT *)var;
/*! 
   @method bindText
*/
-(void)bindText:(TEXT *)var column:(int)col;
/*! 
   @method bindChar
*/
-(void)bindChar:(char *)var;
/*! 
   @method bindChar
*/
-(void)bindChar:(char *)var column:(int)col;
/*! 
   @method bindBinary
*/
- (void)bindBinary:(Oid *)var;
/*! 
   @method bindBinary
*/
- (void)bindBinary:(Oid *)var column:(int)col;
/*! 
   @method bindPoint
   @discussion	*var should be deallocated by free(*var) to avoid memeory leaks
*/
- (void)bindPoint:(POINT *)var;
/*! 
   @method bindPoint
*/
- (void)bindPoint:(POINT *)var column:(int)col;
/*! 
   @method bindPolygon
*/
- (void)bindPolygon:(POLYGON **)var;
/*! 
   @method bindPolygon
*/
- (void)bindPolygon:(POLYGON **)var column:(int)col;
/*! 
   @method bindLSeg
   @discussion	*var should be deallocated by free(*var) to avoid memory leaks
*/
- (void)bindLSeg:(LSEG *)var;
/*! 
   @method bindLSeg
*/
- (void)bindLSeg:(LSEG *)var column:(int)col;
/*! 
   @method bindPath
*/
- (void)bindPath:(PATH **)var;
/*! 
   @method bindPath
*/
- (void)bindPath:(PATH **)var column:(int)col;
/*! 
   @method bindLine
   @discussion	*var should be deallocated by free(*var) to avoid memory leaks
*/
- (void)bindLine:(LINE *)var;
/*! 
   @method bindLine
*/
- (void)bindLine:(LINE *)var column:(int)col;
/*! 
   @method bindBox
*/
- (void)bindBox:(BOX *)var;
/*! 
   @method bindBox
*/
- (void)bindBox:(BOX *)var column:(int)col;
/*! 
   @method bindCircle
*/
- (void)bindCircle:(CIRCLE *)var;
/*! 
   @method bindCircle
*/
- (void)bindCircle:(CIRCLE *)var column:(int)col;
/*! 
   @method bindCIdr
   @discussion	var should point on 12 bytes memory area
*/
- (void)bindCIdr:(unsigned char *)var;
/*! 
   @method bindCIdr
   @discussion	var should point on 12 bytes memory area
*/
- (void)bindCIdr:(unsigned char *)var column:(int)col;
/*! 
   @method bindInet
   @discussion	var should point on 12 bytes memory area
*/
- (void)bindInet:(unsigned char *)var;
/*! 
   @method bindInet
   @discussion	var should point on 12 bytes memory area
*/
- (void)bindInet:(unsigned char *)var column:(int)col;
/*! 
   @method bindMACaddr
   @discussion	var should point on 6 bytes memory area
*/
- (void)bindMACaddr:(unsigned char *)var;
/*! 
   @method bindMACaddr
   @discussion	var should point on 6 bytes memory area
*/
- (void)bindMACaddr:(unsigned char *)var column:(int)col;
/*! 
   @method bindBIT
   @discussion	var should point on 6 bytes memory area
*/
- (void)bindBIT:(BOOLARRAY *)var;
/*! 
   @method bindBIT
   @discussion	var should point on 6 bytes memory area
*/
- (void)bindBIT:(BOOLARRAY *)var column:(int)col;

/*!
	@method escapeString
	@result	nil if out of memory<br>
			autorelase NSString containing the escaped original
*/
-(NSString *)escapeString:(const char *)inSource;
/*!
	@method escapeBinary
	@result	nil if out of memory<br>
			autorelase NSData containing the escaped original
*/
-(NSData *)escapeBinary:(const unsigned char *)inSource length:(size_t)len;
#if	SUPPORT_73
/*!
	@method unescapeBinary
	@abstract	
	@discussion supported in 7.3
	@result	nil if out of memory<br>
			autorelase NSData containing the unescaped original
*/
-(NSData *)unescapeBinary:(const unsigned char *)inSource length:(size_t)len;
#endif

/*! 
   @method	serverMessage
   @result	PQerrorMessage()
*/
-(const char *)serverMessage;
/*! 
   @method connectErrorMessage
   @param errorCode to be converted into string.
   @result PQresStatus(errorCode)
*/
-(const char *)connectErrorMessage:(int)errorCode;

/*! 
   @method beginTransaction
   @abstract self explanatory
   @discussion DOES NOTHING (and returns YES) IN DEMO VERSION.
   @result YES if successful
*/
-(BOOL)beginTransaction;
/*! 
   @method endTransaction
   @abstract self explanatory
   @discussion DOES NOTHING (and returns YES) IN DEMO VERSION.
   @result YES if successful
*/
-(BOOL)endTransaction;
/*! 
   @method rollbackTransaction
   @abstract self explanatory
   @discussion DOES NOTHING (and returns YES) IN DEMO VERSION.
   @result YES if successful
*/
-(BOOL)rollbackTransaction;
/*! 
   @method transactionInProgress
   @abstract self explanatory
   @result YES if a transaction is in progress (beginTransaction has been called)
*/
- (BOOL)transactionInProgress;

/*! 
   @method clearCommands
   @abstract clears the command buffer
*/
-(void)clearCommands;
/*! 
   @method commandBuffer
   @result pointer on command buffer 
*/
-(const char *)commandBuffer;
/*! 
   @method makeCommand
   @abstract concat <code>cmd</code> to current command buffer
   @discussion makeCommand family methods don't escape automatically their arguments<br>
		Use escapeString if you need to.
*/
-(void)makeCommand:(char *)cmd;
/*! 
   @method makeCommandf
   @abstract same as makeCommand but with a format and variable arguments list (printf-like)
*/
-(void)makeCommandf:(const char *)format, ...;
/*! 
   @method executeCommand
   @abstract pass the current command buffer to the PostgreSQL server for execution
   @result	YES if successful
*/
-(BOOL)executeCommand;
/*! 
   @method executeCommandWithCursor
   @abstract pass the current command buffer to the PostgreSQL server for execution inside a cursor declaration
   @param cursorName the name of the cursor
   @param binary if cursor should be declared BINARY
   @result	YES if successful
*/
-(BOOL)executeCommandWithCursor:(const char *)cursorName binary:(BOOL)binary;
/*! 
   @method executeAsyncCommand
   @abstract pass the current command buffer to the PostgreSQL server for asynchronous execution
   @result	YES if successful
*/
-(BOOL)executeAsyncCommand;
/*! 
   @method executeAsyncCommandWithCursor
   @abstract pass the current command buffer to the PostgreSQL server for asynchronous execution inside a cursor declaration
   @param cursorName the name of the cursor
   @param binary if cursor should be declared BINARY
   @result	YES if successful
*/
-(BOOL)executeAsyncCommandWithCursor:(const char *)cursorName binary:(BOOL)binary;

/*!
	@method	cancelRequest
*/
-(BOOL)cancelRequest;

/*! 
   @method closeCursor
   @abstract close the cursor opened with executeCommandWithCursor
   @param cursorName
   @result YES if successful. 
*/
-(BOOL)closeCursor:(const char *)cursorName;

/*! 
   @method bufferHasCommands
   @result YES if command buffer contains not yet executed string. 
*/
-(BOOL)bufferHasCommands;

/*! 
   @method resultColumnCount
   @result the number of columns of latest executed command.
*/
-(int)resultColumnCount;
/*! 
   @method resultColumnName
   @result the name of the column at index col.
*/
-(const char *)resultColumnName:(int)col;
/*! 
   @method resultColumnTypeName
   @result the name of the type of the column at index col.
*/
-(const char *)resultColumnTypeName:(int)col;
/*! 
   @method resultColumnType
   @result the type oid of the column at index col.
*/
-(Oid)resultColumnType:(int)col;
/*! 
   @method resultReturned
   @result YES if latest command executed returns rows.
*/
-(BOOL)resultReturned;
/*! 
   @method resultTableName
   @abstract should return the name of the table of the column at index <code>col</code> in latest result.<br>
   Not yet supported byPostgreSQL.
   @result "NOT_SUPPORTED_BY_POSTGRESQL"
*/
-(const char *)resultTableName:(int)col;
/*! 
   @method resultAsDictionary
   @abstract returns a dictionnary [autorelease] build from result of last query
	@discussion <br>
	resultAsDictionnary	returns a dictionary structured as follows<br>
	<pre><code>&lt;dict&gt;
			&lt;key&gt;columns/key&gt;
			&lt;array&gt;
				&lt;dict&gt;
					&lt;key&gt;name&lt;/key&gt;
					&lt;string&gt;columnTitle&lt;/string&gt;
					&lt;key&gt;typeName&lt;/key&gt;
					&lt;string&gt;columnTypeName&lt;/string&gt;
					&lt;key&gt;typeOid&lt;/key&gt;
					&lt;integer&gt;columnTypeOid&lt;/integer&gt;
				&lt;/dict&gt;
				...
					
			&lt;/array&gt;
			&lt;key&gt;fields/key&gt;
			&lt;array&gt;
				&lt;string&gt;columnFieldValue&lt;/string&gt;
				...

			&lt;/array&gt;
		</code></pre>
			Since PostgreSQL allows creation of new types, we don't provide yet
			an automatic conversion from the PostgreSQL type to the plist one.

			NB<br>
				resultAsDictionary preserves the <code>curRowIndex</code> used by <code>nextRow</code> and <code>previousRow</code> methods<br>
				This means it can be used inside a<br>
				<code>while ([psql nextRow])	{<br>
					...<br>
				}</code><br>
				loop as a debugging tool (for example if an error condition occurs dump the result as a dict in a file)
				without changing the behavior of the loop.
*/
-(NSDictionary *)resultAsDictionary;

/*! 
   @method rowsAffected
*/
-(int)rowsAffected;
/*! 
   @method nextRow
*/
-(BOOL)nextRow;
/*! 
   @method previousRow
*/
-(BOOL)previousRow;
/*! 
   @method afterLastRow
*/
-(void)afterLastRow;
/*! 
   @method isColumnNULL
   @result YES if column at <code>fieldIndex</code> is NULL
*/
-(BOOL)isColumnNULL:(int)fieldIndex;

/*! 
   @method uniqueRowIdForTable
   @abstract uniqueRowIdForTable: requires you have a <br>
							<code>_rowid bigserial NOT NULL PRIMARY KEY</code> <br>
						field in the table
	@result the column value as a string.
*/
-(const char *)uniqueRowIdForTable:(const char *)tableName;
/*! 
   @method uniqueRowIdForTable
   @abstract let you specify the column name to be used for generating the unqiue "rowid".<br>
   Next rowid is generated by executing the following SQL statement:<br><code>
   SELECT max(colName)+1 FROM tableName</code><br>
	@result the column value as a string.
*/
-(const char *)uniqueRowIdForTable:(const char *)tableName column:(const char *)colName;

/*! 
   @method startNotificationFor
   @abstract
   @result
*/
-(PgSQLListener *)startNotificationFor:(const char *)inTableName delegate:(id)notificatonDelegate userInfo:(id)userInfo;
/*! 
   @method removeNotificationFor
   @abstract
*/
-(void)removeNotificationFor:(PgSQLListener *)listener;

/*! 
   @method retrieveBinary
   @abstract	retrieve a BLOB into an newly created NSData object (autorelease)
   @param the Oid of the BLOB to be retrieved.
   @result	NSData object or nil.
*/
-(NSData *)retrieveBinary:(Oid)inOid;
/*! 
	@method retrieveBinaryFromStartLength
	@abstract	retrieve part of a BLOB into an newly created NSData object (autorelease)
    @param the Oid of the BLOB to be retrieved.
	@param start position from which to retrieve data
	@param length number of bytes to retrieve.
	@result	NSData object or nil.
*/
-(NSData *)retrieveBinaryFromStartLength:(Oid)inOid start:(int)inStart length:(int)length;
/*! 
   @method insertBinary
   @abstract insert a BLOB made from a memory buffer.
   @param inData pointer on the memory buffer
   @param size of the memory buffer
   @result	the Oid of the created BLOB, 0 if an error occurred.
*/
-(Oid)insertBinary:(unsigned char *)inData size:(int)size;
/*! 
	@method insertBinaryFromFile
    @abstract insert a BLOB made from NSFileHandle
	@param  the NSFileHandle on the file.
	@result	the Oid of the created BLOB, 0 if an error occurred.
*/
-(Oid)insertBinaryFromFile:(NSFileHandle *)inFile;
/*! 
   @method insertBinaryFromData
   @abstract insert a BLOB made from NSData
   @result	the Oid of the created BLOB, 0 if an error occurred.
*/
-(Oid)insertBinaryFromData:(NSData *)inData;
/*! 
   @method overWriteBinaryWithDataFromStart
   @abstract	overwrite part a of BLOB with bytes from a NSData object from a specified start point.
   @result	YES if successful (number of bytes written == size of NSData buffer).
*/
-(BOOL)overWriteBinaryWithDataFromStart:(Oid)inOid data:(NSData *)inData start:(int)inStart;
/*! 
   @method exportBinaryToFile
   @abstract	Export the BLOB to the file system.
   @result	the lo_export returned code.
*/
-(int)exportBinaryToFile:(Oid)inOid pathName:(const char *)inPathName;
/*! 
	@method importBinaryFromFile
	@abstract insert a BLOB made from a file specified with its pathname.
	@result	the Oid of the created BLOB, 0 if an error occurred.
*/
-(Oid)importBinaryFromFile:(const char *)inPathName;
/*! 
	@method unlinkBinary
	@abstract	delete a large object
	@result 	result code from lo_unlink
*/
-(int)unlinkBinary:(Oid)inOid;
/*! 
   @method asyncResultAvailable
   @abstract	use to check if an asynchronous query has returned results yet.
   @result	YES if data is available ([conn nextRow] will not block).
*/
-(BOOL)asyncResultAvailable;

@end
