/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbsequenceset.h"
#include "../dbconnection.h"
#include "../../utils/debugger.h"

	/**
 	 * Constructor 	
   */		
  DBSequenceSet::DBSequenceSet()
  {
  	// do nothing
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBSequenceSet::~DBSequenceSet()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to indicate if the record set is empty.
	 * @return	true if record set is empty, false otherwise
	 */
	bool DBSequenceSet::isEmpty()
	{
		return m_oSequenceList.isEmpty();
	} // end isEmpty

	/**
	 * Used to get the current index location.
	 * @return	the current index
	 */
	int DBSequenceSet::getCurrentIndex()
	{
		return m_oSequenceList.getCurrentIndex();
	} // end getCurrentIndex
		
	/**
	 * Used to move to the next record in the record set. This function must be called before
	 * the first result can be retrieved. When a record set is initially set, the internal
	 * pointer points to a location before the first record. The reason for this is so that
	 * the record set may sit in a while loop with calls to next indicating if more records
	 * are to come. The function returns true when more records exist and false when no more
	 * records are to come.
	 *
	 * @return	true when more records to come, false when no more records
	 */
	bool DBSequenceSet::next()
	{
		return m_oSequenceList.next();	
	} // end next

	/**
	 * Used to move to the previous record in the record set. The function returns true when
	 * a previous record exists, it will return false when the begining of the set is reached.
	 *
	 * @return	true when previous record exists, false when first record reached (or when empty record set)
	 */
	bool DBSequenceSet::previous()
	{
		return m_oSequenceList.previous();
	} // end previous

	/**
	 * Used to move the cursor back to a position before the first record. This is used when
	 * the recordset needs to be used but then reset, so that next() function can be used
	 * correctly.
	 */
	void DBSequenceSet::reset()
	{
		m_oSequenceList.reset();
	} // end reset
	
	/**
	 * Used to move to the first record.
	 * @exception	throws NoRecordException when empty record
	 */
	void DBSequenceSet::first() throw (NoRecordException)
	{
		return m_oSequenceList.first();
	} // end first

	/**
	 * Used to move to the last record.
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBSequenceSet::last() throw (NoRecordException)
	{
		return m_oSequenceList.last();	
	} // end last
				
	/**
	 * Used to get the number of records in this record set.
	 * @return 	the number of records
	 */
	int DBSequenceSet::getRecordCount()
	{
		return m_oSequenceList.getRecordCount();		
	} // end getRecordCount

	/**
	 * Used to get the current Sequence.
	 * @param	roDBSequence the a DBSequence object used to return output for current record
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBSequenceSet::getDBSequence(DBSequence & roDBSequence) throw (NoRecordException)
	{
		string strMethodName = "DBSequenceSet::getDBSequence";
		
		Debugger::entered(strMethodName);
		
		if (isEmpty())
		{
			throw NoRecordException("Cannot retrieve DBSequence object, no records exist.", "DBSequenceSet", "getDBSequence");
		} // end if empty
		try
		{
			roDBSequence.setSequenceName(m_oSequenceList.getFieldValue("seqname"));
			roDBSequence.setSequenceComment(m_oSequenceList.getFieldValue("comment"));			
			roDBSequence.setSequenceOwner(m_oSequenceList.getFieldValue("seqowner"));						
		} // end try
		catch (VPException e)
		{
			Debugger::logException(strMethodName, e);
		} // end catch	
		
		Debugger::exited(strMethodName);
		
	} // end getDBSequence

