/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#ifndef DBCONNECTION_H
#define DBCONNECTION_H

#include <string>
#include <libpq-fe.h>

#include "../exceptions/dbconnectionexception.h"
#include "../exceptions/sqlexception.h"
#include "dbrecordset.h"

/**
 * This class is used to encapsulate the details of a database connection to postgresql.
 * It is essentially a wrapper for the libpq library provided by postgresql.
 */
class DBConnection
{
public:
	const static string DB_TRUE;
	const static string DB_FALSE;
	
protected:
	string m_strHostName;
	string m_strDatabase;	
	int m_nPortNumber;
	string m_strUserName;
	string m_strPassword;
	PGconn *m_poConnection;			
				
public:
	
	/**
 	 * Constructor
   */		
	DBConnection();
	
	/**
 	 * Destructor
   */		
	~DBConnection();

	/**
 	 * Assignment operator
 	 * This assignment operator simply copies all the connection parameters.
 	 * It does not copy the actual connection.
 	 *
   */		
	DBConnection & operator=(const DBConnection &roDBConnection);
	
	/**
	 * Used to set the database for this database connection.
	 * @param	rstrDatabase	the database for this connection
	 */
	void setDatabase(const string & rstrDatabase);

	/**
	 * Used to return the database for this database connection.
	 * @return	the database set for this connection
	 */
	const string & getDatabase() const;
	
	/**
	 * Used to set the username for this database connection.
	 * @param	rstrUserName	the username for this connection
	 */
	void setUserName(const string & rstrUserName);

	/**
	 * Used to return the username for this database connection.
	 * @return	the username set for this connection
	 */
	const string & getUserName() const;

	/**
	 * Used to set the password for this database connection.
	 * @param	rstrPassword	the password for this connection
	 */
	void setPassword(const string & rstrPassword);

	/**
	 * Used to return the password for this database connection.
	 * @return	the password set for this connection
	 */
	const string & getPassword() const;

	/**
	 * Used to set the hostname for this database connection.
	 * @param	rstrHostName	the hostname for this connection
	 */
	void setHostName(const string & rstrHostName);

	/**
	 * Used to return the hostname for this database connection.
	 * @return	the hostname set for this connection
	 */
	const string & getHostName() const;

	/**
	 * Used to set the port number for this database connection.
	 * @param	nPortNumber	the port number for this connection
	 */
	void setPortNumber(int nPortNumber);

	/**
	 * Used to return the port number for this database connection.
	 * @return	the port number set for this connection
	 */
	int getPortNumber() const;

	/**
 	 * Used to establish a database connection. If the database connection parameters have not
	 * been set, then all default values will be used. The default values are the postgres user
	 * and a connection to a local server on the standard postgres port 5432.
 	 *
 	 * @exception DBConnectionException is thrown if unable to establish connection
   */			    	
	void connect() throw (DBConnectionException);
	
	/**
 	 * Used to establish a database connection. If the server is not using password authentication
 	 * then enter a blank string for password.
 	 *
 	 * @param	rstrUserName	the username to establish the connection (if not specified, postgres is used)
 	 * @param	rstrPassword	the pass associated with the username (if not specified, no password is used)
 	 * @param rstrDatabase	the name of the database
 	 * @param rstrHostName	the name of the server (or IP address) to connect to (if not specified, tries to connect locally)
 	 * @param nPortNumber	the port used for the server connection
 	 *
 	 * @exception DBConnectionException is thrown if unable to establish connection
   */			    	
	void connect(const string & rstrUserName, const string & rstrPassword, const string & rstrDatabase,
				const string & rstrHostName, int nPortNumber) throw (DBConnectionException);
																																			
	/**
 	 * Used to execute a query statement to the database. If a connection has not been established
	 * yet it calls the connect() method first with the current connection settings. The results of
	 * of the query are returned via the record set. It is safe to reuse RecordSet objects since if
	 * new records are assigned to an already used record set, all dynamic data for those records
	 * are freed before assigning the new data. If the statement was a select query then the method
	 * will return true, otherwise if the sql statement was an insert/update/delete then false is
	 * returned. If a select statement was called and returns no data, the method will still return
	 * true with an empty record set.
	 *
	 * @param			strSQLQuery	a SQL query statement
	 * @param			oRecordSet 	a record set containing the results of the query
	 *
	 * @return		true if sql select statement called, false otherwise
	 * @exception	SQLException is thrown if the sql query fails
 	 * @exception DBConnectionException is thrown if unable to establish connection		
   */			    			
	bool executeQuery(const string & rstrSQLQuery, DBRecordSet & oRecordSet) throw (SQLException, DBConnectionException);

	/**
 	 * Used to execute a non-query statement to the database. If a connection has not been established
	 * yet it calls the connect() method first with the current connection settings. If the query is
	 * a select query then the results cannot be retrieved with this method. Use the executeQuery method
	 * if you may need to retrieve records back.
	 *
	 * @param			strSQLStatement		a SQL non-query statement
	 *
	 * @exception	SQLException is thrown if the sql statement call fails
 	 * @exception DBConnectionException is thrown if unable to establish connection		
   */			    			
	void execute(const string & rstrSQLStatement) throw (SQLException, DBConnectionException);

	/**
 	 * Used to free the connection to the postgresql backend.
   */			    			
	void close();
			
};

#endif

