/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbindexmanager.h"
#include "../../../utils/stringutils.h"

	/**
 	 * Constructor
	 */
  DBIndexManager::DBIndexManager()
  	: DBBaseManager()
  {
  } // end constructor
	
	/**
 	 * Constructor
 	 * It is assumed that the database connection object will remain alive during
 	 * the life of this object. Be very careful to ensure that no methods are called
 	 * on this object if the connection object no longer exists. If the connection
 	 * object has already been destroyed then unpredictable results will be returned. 	
   */		
  DBIndexManager::DBIndexManager(DBConnection *poDBConn)
  	: DBBaseManager(poDBConn)
  {
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBIndexManager::~DBIndexManager()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to retrieve the list of indicies for the specified table.
	 * @return	a DBIndexSet object that contains the index details
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if not connected
	 */
	void DBIndexManager::retrieveListOfIndicies(const string & rstrTableName, DBIndexSet & roDBIndexSet, bool bOnlyPrimary)
					throw (SQLException, DBConnectionException)
	{
		// sql statement used to retrieve all the indicies details
		string strSQL = "SELECT c.relname as tablename, i.relname as indexname, "
													 "x.indisunique as isunique, x.indisprimary as isprimary, "		
													 "d.description as comment "
										"FROM pg_index x, pg_class c, pg_class i, pg_description d "
										"WHERE c.oid = x.indrelid "
										"AND i.oid = x.indexrelid "
										"AND c.relname = '" + rstrTableName + "' "
										"AND d.objoid = i.oid ";

		// check to see if only want primary index
		if (bOnlyPrimary == true)
		{
			strSQL += "AND x.indisprimary = 't' ";
		} // end if only primary
																				
		strSQL +=				"UNION ALL "										
										"SELECT c.relname as tablename, i.relname as indexname, "
													 "x.indisunique as isunique, x.indisprimary as isprimary, "		
													 "NULL as comment "
										"FROM pg_index x, pg_class c, pg_class i "
										"WHERE c.oid = x.indrelid "
										"AND i.oid = x.indexrelid "
										"AND c.relname = '" + rstrTableName + "' "
										"AND i.oid not in (select objoid from pg_description) ";
										
		// check to see if only want primary index
		if (bOnlyPrimary == true)
		{
			strSQL += "AND x.indisprimary = 't' ";
		} // end if only primary
		
		strSQL += "ORDER BY indexname";												
										
		// execute query										
		m_poDBConn->executeQuery(strSQL, roDBIndexSet.m_oIndexList);										
		
		// create array
		if (roDBIndexSet.getRecordCount() > 0)
		{
			roDBIndexSet.m_aoIndexedColumns = new DBRecordSet[roDBIndexSet.getRecordCount()];
		} // end if index exist
		
		// now retrieve column details		
		while (roDBIndexSet.next())
		{
			DBIndex oDBIndex;
			roDBIndexSet.getDBIndex(oDBIndex);
			strSQL = "SELECT a.attnum, a.attname as column "
							 "FROM pg_attribute a, pg_class ci, pg_index i "
							 "WHERE a.attrelid = i.indrelid "
							 "AND ci.oid = i.indexrelid "
							 "AND ci.relname = '" + oDBIndex.getIndexName() + "' "
							 "AND a.attnum in (i.indkey[7], i.indkey[6], "
							 "i.indkey[5], i.indkey[4], "
							 "i.indkey[3], i.indkey[2], "
							 "i.indkey[1], i.indkey[0])";							
			// execute query										
			m_poDBConn->executeQuery(strSQL, roDBIndexSet.m_aoIndexedColumns[roDBIndexSet.getCurrentIndex()]);																	
		} // end while more columns							

		// set recordset back to start
		roDBIndexSet.reset();										
	} // end retrieveListOfIndicies
		
	/**
	 * Used to modify the index comment.
	 * @param			rstrIndexName		the name of the index
	 * @param			rstrComment			the comment to set for the index
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBIndexManager::modifyIndexComment(const string & rstrIndexName, const string & rstrComment)
				throw (SQLException, DBConnectionException)
	{
		string strSQLComment;
 		if (rstrComment == "")
 		{
 			strSQLComment = "COMMENT ON INDEX " + rstrIndexName + " IS NULL";
 		} // end if need to drop comment
 		else
 		{
 			strSQLComment = "COMMENT ON INDEX " + rstrIndexName +
 												" IS '" + StringUtils::databasestr(rstrComment) + "'";			
 		} // end else need to set new comment
		// set comment
		m_poDBConn->execute(strSQLComment);												
	
	} // end modifyIndexComment
