/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#ifndef DBTABLECOLUMN_H
#define DBTABLECOLUMN_H

#include <string>

/**
 * This class is used to represent a column in a database table.
 */
class DBTableColumn
{
public:
	const static string NUMERIC_FIELD_NAME;
			
protected:
	string m_strFieldName;
	string m_strFieldType;
	int m_nFieldSize;	
	int m_nNumericPrecision;		
	int m_nNumericDecimal;
	bool m_bPrimaryKey;				
	bool m_bFieldNullable;
	string m_strFieldDefault;
	string m_strFieldComment;
								
public:
	
	/**
 	 * Constructor
   */		
	DBTableColumn();
	
	/**
 	 * Destructor
   */		
	~DBTableColumn();

	/**
	 * This function will clear out variables in this object, so that it can be used again.
	 */
	void clear();
	
	/**
 	 * Copy Constructor
 	 * Implemented for the vector class.
   */		
	DBTableColumn(const DBTableColumn &roDBTableColumn);

	/**
 	 * Assignment operator
 	 * Implemented for the vector class.
   */		
	DBTableColumn & operator=(const DBTableColumn &roDBTableColumn);
	
	/**
	 * Equivalence operator
	 * Implemented for the vector class.
	 * If both table columns have the same column name, then they are equivalent.
	 */
	bool operator==(DBTableColumn &roDBTableCol);

	/**
	 * Less than operator
	 * Implemented for the vector class.
	 * This makes a lexical comparison on the column name. If the left column
	 * has a column name that is lexically less than the right column, it will
	 * return true, otherwise false.
	 */
	bool operator<(DBTableColumn &roDBTableCol);
				
	/**
 	 * Used to set field details. For numeric types use the other method.
   */			    						
	void setFieldDetails(const string &rstrFieldName, const string &rstrFieldType,
												int nFieldSize, bool bPrimaryKey, bool bFieldNullable, const string &rstrFieldDefault,
												const string &rstrFieldComment);

	/**
 	 * Used to set numerice field details. This method should be used when the field being added
 	 * is of the numeric type.
   */			    						
	void setNumericFieldDetails(const string &rstrFieldName, int nNumericSize,
												int nDecimalSize,  bool bPrimaryKey, bool bFieldNullable, const string &rstrFieldDefault,
												const string &rstrFieldComment);
												
	/**
 	 * Used to retrieve the field name.
   */			    			
	const string & getFieldName() const;			

	/**
 	 * Used to retrieve the field type.
   */			    			
	const string & getFieldType() const;			

	/**
 	 * Used to retrieve the field size.
   */			    			
	int getFieldSize() const;			

	/**
	 * Used to check if the field type is numeric.
	 */
	bool isNumeric() const;
	
	/**
 	 * Used to retrieve the precision size of a numeric type.
   */			    			
	int getNumericPrecision() const;			

	/**
 	 * Used to retrieve the decimal size of a numeric type.
   */			    			
	int getNumericDecimal() const;			
		
	/**
 	 * Used to identify if a field is a primary key
   */			    			
	bool isPrimaryKey() const;			
	
	/**
 	 * Used to retrieve if a field is nullable.
   */			    			
	bool isFieldNullable() const;			

	/**
 	 * Used to retrieve a field default.
   */			    			
	const string & getFieldDefault() const;			

	/**
 	 * Used to retrieve a field comment.
   */			    			
	const string & getFieldComment() const;			
	 						
};

#endif

