/*  Prolog Interface to Postgresql
    Copyright (C) 1999-2003  Alexander Diemand
  
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <gprolog.h>
#include <postgresql/libpq-fe.h>


/* stolen from the python interface, should be defined in a pgsql header instead!! */

#define INT2OID         21
#define INT4OID         23
#define OIDOID          26
#define FLOAT4OID       700
#define FLOAT8OID       701
#define CASHOID         790


/* Prototypes */

Bool gp_pgsql_connect(char *host, long port, char *user, char *passwd, char *dbname, FIOArg *res_pgsql);
Bool gp_pgsql_connect1(FIOArg *res_pgsql);
Bool gp_pgsql_disconnect (FIOArg *my_pgsql);

Bool gp_pgsql_query1 (FIOArg *my_pgsql, char *query);

Bool gp_pgsql_query2 (FIOArg *my_pgsql, char *query, FIOArg *my_result);


/* Implementation */

Bool gp_pgsql_connect1(FIOArg *res_pgsql)
{
	char       errmsg[256];
	char       *tenv;
	char       *hostname;
	char       *dbname;
	int        port;

        if (! res_pgsql->is_var) {
                Pl_Err_Instantiation();
        }

        res_pgsql->value.l = 0L;

	/* get PGHOST */
	tenv = getenv("PGHOST");
	if (!tenv)
	{
                sprintf (errmsg, "PGSQL: pl_pgsql_connect/1 failed.\nNo environment variable $PGHOST\n");
                Pl_Err_System(Create_Atom(errmsg));
                return FALSE;

	}
	hostname = tenv;
	
	/* get PGPORT */
	tenv = getenv("PGPORT");
	if (tenv)
	{
		port = atol(tenv);
	} else {
		port = 5432;
	}

	/* get PGDATABASE */
	tenv = getenv("PGDATABASE");
	if (!tenv)
	{
                sprintf (errmsg, "PGSQL: pl_pgsql_connect/1 failed.\nNo environment variable $PGDATABASE\n");
                Pl_Err_System(Create_Atom(errmsg));
                return FALSE;

	}
	dbname = tenv;
	
	return gp_pgsql_connect(hostname,port,NULL,NULL,dbname,res_pgsql);
}

Bool gp_pgsql_connect(char *host, long port, char *user, char *passwd, char *dbname, FIOArg *res_pgsql) {
        PGconn     *conn;
        char       my_port[128];
        char       errmsg[1024];

        if (! res_pgsql->is_var) {
                Pl_Err_Instantiation();
        }

        sprintf(my_port, "%ld", port);
        conn = PQsetdb(host, my_port, NULL, NULL, dbname);

        /*
        * check to see that the backend connection was successfully made
        */
        if (PQstatus(conn) == CONNECTION_BAD)
        {
                /* printf ("PGSQL: Connection to database '%s' failed.\n%s", dbname, PQerrorMessage(conn)); */
                sprintf (errmsg, "PGSQL: Connection to database %s failed.\n%s", dbname, PQerrorMessage(conn));
                PQfinish(conn);
                Pl_Err_System(Create_Atom(errmsg));
                return FALSE;
        /* } else {
                printf ("PGSQL: connection ok.\n");
        */
        }

	#if defined(M_mips_irix) || defined(M_powerpc_linux)
        res_pgsql->value.l = (long)conn>>1;
	#else
        res_pgsql->value.l = (long)conn;
	#endif
        res_pgsql->unify = TRUE;

	/* fprintf(stderr,"conn is %lx\n",res_pgsql->value.l); */

        return TRUE;
}

Bool gp_pgsql_disconnect (FIOArg *my_pgsql) {
        PGconn     *conn;
        if (!my_pgsql->value.l)
                return FALSE;
	#if defined(M_mips_irix) || defined(M_powerpc_linux)
		conn = (PGconn*)((long)(my_pgsql->value.l)<<1);
	#else
		conn = (PGconn*)my_pgsql->value.l;
	#endif
        
        if (PQstatus(conn) == CONNECTION_OK)
	{
		PQfinish(conn);
	} else {
		char *typereason;
		char errmsg[1024];
		ConnStatusType connstat = PQstatus(conn);
		switch (connstat) {
			case CONNECTION_SETENV:
				typereason="CONNECTION_SETENV";
				break;
			case CONNECTION_AUTH_OK:
				typereason="CONNECTION_AUTH_OK";
				break;
			case CONNECTION_AWAITING_RESPONSE:
				typereason="CONNECTION_AWAITING_RESPONSE";
				break;
			case CONNECTION_MADE:
				typereason="CONNECTION_MADE";
				break;
			case CONNECTION_STARTED:
				typereason="CONNECTION_STARTED";
				break;
			case CONNECTION_BAD:
				typereason="CONNECTION_BAD";
				break;
			case CONNECTION_OK:
				typereason="CONNECTION_OK";
				break;
			default:
				typereason="unknown Connection type reason.";
				break;
		};
		No_More_Choice();
		strcpy (errmsg, "PGSQL: connection type: ");
		strcat (errmsg, typereason);
		Pl_Err_System(Create_Atom(errmsg));
		return FALSE;
	}
        my_pgsql->value.l = 0;
        return TRUE;
}
                
Bool gp_pgsql_query1 (FIOArg *my_pgsql, char *query) {
        PGconn     *conn;
        PGresult   *result;
        
        if (!my_pgsql->value.l)
                return FALSE;

	#if defined(M_mips_irix) || defined(M_powerpc_linux)
		conn = (PGconn*)((long)(my_pgsql->value.l)<<1);
	#else
		conn = (PGconn*)(my_pgsql->value.l);
	#endif

        /* fprintf(stderr, "Query: %s\n", query); */

        result = PQexec (conn, query);
        if (PQresultStatus(result) != PGRES_COMMAND_OK) {
                
                PQclear(result);
                return FALSE;
        }

        PQclear(result);
        return TRUE;
}



Bool gp_pgsql_query2 (FIOArg *my_pgsql, char *query, FIOArg *my_res) {
        PGconn     *conn;
        PGresult   **result;
        int nfields, nrows, fnum, choice;
        PlTerm *newargs;
        char errmsg[1024];
        
        if (!my_pgsql->value.l)
                return FALSE;

        /* fprintf(stderr, "Conn: %lx Query: %s\n", my_pgsql->value.l,query); */

	#if defined(M_mips_irix) || defined(M_powerpc_linux)
		conn = (PGconn*)((long)(my_pgsql->value.l)<<1);
	#else
		conn = (PGconn*)(my_pgsql->value.l);
	#endif

        result = Get_Choice_Buffer(PGresult**);

        choice = Get_Choice_Counter();
        /* printf ("CHoice is %d\n", choice); */

        if (choice == 0) { /* first call */

                if (!(*result = PQexec (conn, query))) {
                        No_More_Choice();
                        strcpy (errmsg, "PGSQL: ");
                        strcat (errmsg, PQerrorMessage(conn));
                        PQfinish(conn);
                        Pl_Err_System(Create_Atom(errmsg));
                        return FALSE;
                }

        } /* first call */

        nrows =  PQntuples(*result);
        nfields = PQnfields(*result);
        /* fprintf(stderr, "Result rows: %d fields: %d\n", nrows, nfields); */

        if (nrows < 1) {
                No_More_Choice();
                PQclear(*result);
                return FALSE;
        }

        newargs = (PlTerm*)malloc(sizeof(PlTerm)*(nfields));
        fnum = 0;
        while (fnum < nfields) {
               char *tchar;
               tchar=PQgetvalue(*result,choice,fnum);
               /* printf ("got: type %d  value %s\n", PQftype((*result),fnum), tchar);  */
               switch (PQftype((*result), fnum)) {
                        case INT2OID:
                        case INT4OID:
                        case OIDOID:
                                newargs[fnum] = Mk_Integer(strtol(tchar,NULL,10));
                                break;
                        case FLOAT4OID:
                        case FLOAT8OID:
                        case CASHOID:
                                newargs[fnum] = Mk_Float(strtod(tchar,NULL));
                                break;
                        default:
                                if (strcmp(tchar,"NULL") == 0) {
                                        newargs[fnum] = Mk_Codes("");
                                } else {
                                        newargs[fnum] = Mk_Codes(tchar);
                                }
                                break;
                } /* switch field type */
                fnum++;
        } /* for every field in the row */

        my_res->value.l = Mk_Proper_List(nfields,newargs);
	/* fprintf(stderr,"res is %lx\n",my_res->value.l); */


        free(newargs);

        /* check if this is the last row to fetch */
        if (choice+1 >= nrows) {
                /* don't allow backtracking into this anymore */
                No_More_Choice();
                PQclear(*result);
        }

        return TRUE;
}

