/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <kdebug.h>
#include <kurl.h>

// Qt includes
#include <qclipboard.h>
#include <qdom.h>
#include <qfile.h>

// WorKflow includes
#include "document.h"
#include "document_p.h"
#include "command.h"
#include "commanddrag.h"
#include "commandfactory.h"
#include "commanddescription.h"
#include "commandmanager.h"
#include "parameter.h"
#include "result.h"
#include "datatype.h"
#include "applicationmanager.h"

using namespace WorKflow;

// CompositeAction implementation
CompositeAction::~CompositeAction()
{
  for (ActionList::Iterator i = m_actions.begin(); i != m_actions.end(); ++i)
    delete *i;
}

bool CompositeAction::execute()
{
  for (ActionList::Iterator i = m_actions.begin(); i != m_actions.end(); ++i)
    if (!(*i)->execute())
      return false;
  return true;
}

bool CompositeAction::unexecute()
{
  // unexecute in reverse order
  ActionList::Iterator i = m_actions.end();
  while (i != m_actions.begin()) {
    --i;
    if (!(*i)->unexecute())
      return false;
  }

  return true;
}

void CompositeAction::addAction(Action* action)
{
  m_actions.append(action);
}

// InsertCommandAction implementation
InsertCommandAction::InsertCommandAction(Document* document, int row, Command* cmd)
  : Action(document), m_row(row), m_command(cmd), m_ownsCommand(true)
{
}

InsertCommandAction::~InsertCommandAction()
{
  // by default, this action takes ownership of the command
  if (m_ownsCommand)
    delete m_command;
}

QString InsertCommandAction::name()
{
  return i18n("Insert command \"%1\"").arg(m_command->title());
}

bool InsertCommandAction::execute()
{
  document()->doInsertCommand(m_row, m_command);

  return true;
}

bool InsertCommandAction::unexecute()
{
  document()->doRemoveCommand(m_row);

  return true;
}

// RemoveCommandAction implementation
RemoveCommandAction::RemoveCommandAction(Document* document, int row)
  : Action(document), m_row(row)
{
  m_command = this->document()->commandAt(row);
}

QString RemoveCommandAction::name()
{
  return i18n("Remove command \"%1\"").arg(m_command->title());
}

bool RemoveCommandAction::execute()
{
  document()->doRemoveCommand(m_row);

  return true;
}

bool RemoveCommandAction::unexecute()
{
  document()->doInsertCommand(m_row, m_command);

  return true;
}

// MoveCommandAction implementation
MoveCommandAction::MoveCommandAction(Document* document, int sourceRow, int destRow)
  : CompositeAction(document), m_command(document->commandAt(sourceRow))
{
  addAction(new RemoveCommandAction(document, sourceRow));
  InsertCommandAction* insert = new InsertCommandAction(document, destRow, m_command);
  insert->setOwnsCommand(false);
  addAction(insert);
}

QString MoveCommandAction::name()
{
  return i18n("Move command \"%1\"").arg(m_command->title());
}

// RemoveRangeAction implementation
RemoveRangeAction::RemoveRangeAction(Document* document, int startRow, int endRow)
  : CompositeAction(document)
{
  ::ActionList actions;

  // for each command in the range, create a remove action
  Document::Iterator i = document->iterator(startRow),
                           end = document->iterator(endRow);
  for (; i != end; ++i) {
    actions.append(new RemoveCommandAction(document, i.row()));
  }

  // add actions in reverse order
  ::ActionList::ConstIterator j = actions.end();
  while (j != actions.begin()) {
    --j;
    addAction(*j);
  }
}

QString RemoveRangeAction::name()
{
  return i18n("Remove commands");
}

// ModifyCommandAction implementation
ModifyCommandAction::ModifyCommandAction(Document* document, Command* command, const CommandState& oldState, const CommandState& newState)
  : Action(document), m_command(command), m_oldState(oldState), m_newState(newState), m_dontReallyExecute(true)
{
}

QString ModifyCommandAction::name()
{
  return i18n("Modify command \"%1\"").arg(m_command->title());
}

bool ModifyCommandAction::execute()
{
  if (m_dontReallyExecute) {
    // don't execute the first time, because the command's state is up-to-date
    m_dontReallyExecute = false;
    return true;
  }

  m_command->loadState(m_newState);

  return true;
}

bool ModifyCommandAction::unexecute()
{
  m_command->loadState(m_oldState);

  return true;
}

// CutRangeAction implementation
CutRangeAction::CutRangeAction(Document* document, int startRow, int endRow)
  : RemoveRangeAction(document, startRow, endRow)
{
}

QString CutRangeAction::name()
{
  return i18n("Cut commands");
}

// PasteCommandAction implementation
PasteCommandAction::PasteCommandAction(Document* document, int row, const CommandInfoList& commands)
  : Action(document), m_row(row), m_commands(commands)
{
}

PasteCommandAction::~PasteCommandAction()
{
  for (QValueList<CommandInfo>::Iterator i = m_commands.begin(); i != m_commands.end(); ++i) {
    if ((*i).command)
      delete (*i).command;
  }
}

QString PasteCommandAction::name()
{
  return i18n("Paste commands");
}

bool PasteCommandAction::execute()
{
  int row = m_row;

  for (QValueList<CommandInfo>::Iterator i = m_commands.begin(); i != m_commands.end(); ++i) {
    CommandInfo& info = *i;
    if (!info.command) {
      CommandFactoryBase* factory = CommandManager::self()->commandDescription((*i).commandId)->factory();
      if (factory) {
        Command* cmd = factory->create(document());
        info.command = cmd;

        cmd->loadState(info.commandState);
      } else {
        kdWarning() << "attempted to paste command with unknown id \"" << (*i).commandId << "\"!" << endl;
        return false;
      }
    }

    document()->doInsertCommand(row, info.command);
  }

  return true;
}

bool PasteCommandAction::unexecute()
{
  QValueList<CommandInfo>::Iterator i = m_commands.end();

  while (i != m_commands.begin()) {
    --i;
    Command* cmd = (*i).command;
    if (cmd) {
      document()->doRemoveCommand(cmd->row());
    } else {
      kdWarning() << "command pointer unset in " << k_funcinfo << endl;
      return false;
    }
  }

  return true;
}

// Document::Private implementation
Document::Private::Private(Document* doc)
  : modified(false), executing(false), modifiedCommand(0), undoing(false), p(doc)
{
  connect(&modificationTimer, SIGNAL(timeout()), this, SLOT(modificationTimeout()));
}

void Document::Private::execAction(Action* action)
{
  undoList.append(action);
  action->execute();

  for (ActionList::Iterator i = redoList.begin(); i != redoList.end(); ++i) {
    delete *i;
  }
  redoList.clear();

  modified = true;
  p->emitUndoSignals();
}

void Document::Private::updateCommandState(int row)
{
  Command* cmd = p->commandAt(row);
  if (cmd) {
    commandStates[row] = cmd->saveState();
  }
}

void Document::Private::slotCommandModified(Command* command)
{
  if (undoing)
    return;

  kdDebug() << k_funcinfo << endl;
  // if another command was modified
  if (modifiedCommand && modifiedCommand != command) {
    // stop the timer and generate undo object immediately
    modificationTimeout();
  }

  modifiedCommand = command;
  // one second inactivity creates an undo command
  modificationTimer.start(1000);
}

void Document::Private::modificationTimeout()
{
  kdDebug() << k_funcinfo << endl;
  if (modifiedCommand) {
    int row = modifiedCommand->row();
    CommandState oldState = commandStates[row];

    updateCommandState(row);
    CommandState newState = commandStates[row];

    execAction(new ModifyCommandAction(p, modifiedCommand, oldState, newState));
  }

  modificationTimer.stop();
  modifiedCommand = 0;
}

void Document::Private::slotProgress(double prog, const QString& task)
{
  double total = (double(currentCommand->row()) + prog) / double(p->numRows());
  emit progress(total, prog, currentCommand->title(), task);
}

void Document::Private::abort()
{
  executing = false;
}

// Document::Iterator implementation
Document::Iterator::Iterator(Document* document, int row)
  : m_document(document), m_row(row)
{
}

Command* Document::Iterator::operator*()
{
  return m_document->commandAt(m_row);
}

Document::Iterator& Document::Iterator::operator++()
{
  m_row++;
  return *this;
}

bool Document::Iterator::operator==(const Iterator& it)
{
  return m_row == it.m_row;
}

bool Document::Iterator::operator!=(const Iterator& it)
{
  return !operator==(it);
}

// Document implementation
Document::Document(QObject* parent, const char* name)
  : QObject(parent, name)
{
  d = new Private(this);

  connect(d, SIGNAL(progress(double, double, const QString&, const QString&)),
          this, SIGNAL(progress(double, double, const QString&, const QString&)));
}

Document::~Document()
{
  clear();

  delete d;
}

void Document::execute()
{
  d->executing = true;
  d->variables.clear();

  Result* res = 0;
  for (Iterator i = begin(); i != end(); ++i) {
    if (!d->executing) {
      return;
    }

    d->currentCommand = *i;

    if (d->currentCommand->input()) {
      if (!res /*&& !cmd->input()->isOptional()*/)
        kdWarning() << "command " << d->currentCommand->id() << " needs input but didn't get any" << endl;
      else {
        // convert value
        Value val = res->value().convertTo(d->currentCommand->input()->type());
        // set parameter
        d->currentCommand->input()->setStaticValue(val);
      }
    }

    connect(d->currentCommand, SIGNAL(aborted()), d, SLOT(abort()));
    connect(d->currentCommand, SIGNAL(progress(double, const QString&)), d, SLOT(slotProgress(double, const QString&)));

    // in case the command does not report progress:
    d->slotProgress(0, "");

    d->currentCommand->execute();
    disconnect(d->currentCommand, SIGNAL(aborted()), d, SLOT(abort()));
    disconnect(d->currentCommand, SIGNAL(progress(double, const QString&)), d, SLOT(slotProgress(double, const QString&)));

    res = d->currentCommand->output();
  }

  d->executing = false;
}

int Document::numRows()
{
  return d->commands.count();
}

Command* Document::commandAt(int row)
{
  if (row >= d->commands.count())
    return 0;

  return d->commands[row];
}

Document::Iterator Document::begin()
{
  return Iterator(this, 0);
}

Document::Iterator Document::end()
{
  return Iterator(this, numRows());
}

Document::Iterator Document::iterator(int row)
{
  return Iterator(this, row);
}

void Document::abort()
{
  d->executing = false;
}

void Document::reorderCommands()
{
  bool change = false;

  for (Iterator i = begin(); i != end(); ++i) {
    Command* cmd = *i;
    cmd->setRow(i.row());
  }

  if (change)
    emit orderChanged();
}

void Document::insertCommand(int row, Command* command)
{
  d->modificationTimeout();
  d->execAction(new InsertCommandAction(this, row, command));
}

void Document::removeCommand(int row)
{
  d->modificationTimeout();
  d->execAction(new RemoveCommandAction(this, row));
}

void Document::removeRange(int startRow, int endRow)
{
  d->modificationTimeout();
  d->execAction(new RemoveRangeAction(this, startRow, endRow));
}

void Document::moveCommand(int sourceRow, int destRow)
{
  d->modificationTimeout();
  d->execAction(new MoveCommandAction(this, sourceRow, destRow));
}

void Document::clear()
{
  while (numRows()) {
    Command* cmd = commandAt(0);
    doRemoveCommand(0);
    delete cmd;
  }

  d->undoList.clear();
  d->redoList.clear();
}

void Document::undo()
{
  d->modificationTimeout();

  d->undoing = true;
  if (!d->undoList.isEmpty()) {
    Action* action = d->undoList.last();
    if (action->unexecute()) {
      d->undoList.pop_back();
      d->redoList.append(action);

      d->modified = !d->undoList.isEmpty();
    } else {
      kdWarning() << "could not undo action \"" << action->name() << "\"!" << endl;
    }

    emitUndoSignals();
  }
  d->undoing = false;
}

void Document::redo()
{
  d->modificationTimeout();

  d->undoing = true;
  if (!d->redoList.isEmpty()) {
    Action* action = d->redoList.last();
    if (action->execute()) {
      d->redoList.pop_back();
      d->undoList.append(action);

      d->modified = !d->undoList.isEmpty();
    } else {
      kdWarning() << "could not redo action \"" << action->name() << "\"!" << endl;
    }

    emitUndoSignals();
  }
  d->undoing = false;
}

int Document::undoCount()
{
  return d->undoList.count();
}

int Document::redoCount()
{
  return d->redoList.count();
}

Parameter* Document::search(int startRow, const QString& text)
{
  d->searchText = text;
  d->searchCommand = commandAt(startRow);
  d->searchParameter = 0;
  return findNext();
}

Parameter* Document::findNext()
{
/*  typedef Command::ParameterList Params;

  int row = d->searchCommand->row();

  while (row != numRows()) {
    Params params = d->searchCommand->parameterList();

    if (!d->searchParameter) {
      // search title
      if (d->searchCommand->title().contains(d->searchText))
        

      // then first parameter
      d->searchParameter = params.first();

    } else {
      Params::ConstIterator i = params.find(d->searchParameter);
      if (i != params.end()) {

      }
    }
  }*/
}

Parameter* Document::findPrev()
{
}

// bool Document::openFile(KIO::Job* job)
// {
// }

void Document::cut(int startRow, int endRow)
{
  copy(startRow, endRow);
  d->execAction(new CutRangeAction(this, startRow, endRow));
}

void Document::copy(int startRow, int endRow)
{
  d->modificationTimeout();

  CommandInfoList list;

  Iterator end = iterator(endRow);
  for (Iterator i = iterator(startRow); i != end; ++i) {
    CommandInfo info;
    info.commandId = (*i)->id();
    info.commandState = d->commandStates[i.row()];

    list.append(info);
  }

  KApplication::clipboard()->setData(new CommandDrag(CommandDrag::Info(list), 0), QClipboard::Clipboard);
}

void Document::paste(int row)
{
  d->modificationTimeout();
  QMimeSource* data = KApplication::clipboard()->data(QClipboard::Clipboard);

  if (CommandDrag::canDecode(data)) {
    CommandDrag::Info info;
    if (CommandDrag::decode(data, info)) {
      if (info.kind() == CommandDrag::Info::CopiedCommands) {
        d->execAction(new PasteCommandAction(this, row, info.copiedCommands()));
      } else {
        kdWarning() << "clipboard contains non-copied commands" << endl;
      }
    }
  }
}

QString Document::fileName()
{
  if (d->url.isEmpty())
    return i18n("unnamed");
  else
    return d->url.fileName();
}

void Document::setUrl(const KURL& url)
{
  d->url = url;
}

KURL Document::url()
{
  return d->url;
}

bool Document::isModified()
{
  return d->modified;
}

void Document::doInsertCommand(int row, Command* command)
{
  if (row > d->commands.count()) {
    kdWarning() << k_funcinfo << "numRows() < row (" << row << ")" << endl;
    return;
  }

  Commands::Iterator it = d->commands.begin() + row;
  d->commands.insert(it, command);

  // command state
  d->commandStates.insert(d->commandStates.begin() + row, CommandState());
  d->updateCommandState(row);

  connect(command, SIGNAL(modified(Command*)), d, SLOT(slotCommandModified(Command*)));
  connect(command, SIGNAL(typeCheckNeeded()), this, SLOT(typeCheck()));
  reorderCommands();
  emit commandAdded(command);

  typeCheck();
}

void Document::doRemoveCommand(int row)
{
  if (row >= d->commands.count()) {
    kdWarning() << k_funcinfo << "numRows() < row (" << row << ")" << endl;
    return;
  }

  Commands::Iterator j = d->commands.begin() + row;
  Command* cmd = d->commands[row];
  d->commands.erase(j);

  // command state
  d->commandStates.erase(d->commandStates.begin() + row);

  disconnect(cmd, SIGNAL(modified(Command*)), d, SLOT(slotCommandModified(Command*)));
  emit commandRemoved(cmd);
  reorderCommands();

  typeCheck();
}

void Document::emitUndoSignals()
{
  emit undoChanged();
  emit modified();
}

bool Document::isExecuting()
{
  return d->executing;
}

Value Document::variable(const QString& name) const
{
  return d->variables[name];
}

void Document::setVariable(const QString& name, const Value& value)
{
  d->variables[name] = value;
}

void Document::typeCheck()
{
  if (numRows() == 0)
    return;

  kdDebug() << k_funcinfo << endl;

  Result* prev = 0;
  for (Iterator i = begin(); i != end(); ++i) {
    (*i)->typeCheck(prev);
    prev = (*i)->output();
  }
}

bool Document::load(const QString& filename)
{
  QDomDocument doc("");
  QFile file(filename);
  if (!file.open(IO_ReadOnly)) {
    kdWarning() << "Could not read XML file \"" << filename << "\"!" << endl;
    return false;
  }
  if (!doc.setContent(&file)) {
    kdWarning() << "Could not parse XML file \"" << filename << "\"!" << endl;
    file.close();
    return false;
  }
  file.close();

  QDomElement docElem = doc.documentElement();

  readXML(docElem);

  d->modified = false;

  return true;
}

bool Document::save(const QString& filename)
{
  QFile file(filename);
  if (!file.open(IO_WriteOnly)) {
    kdWarning() << "Could not write XML file \"" << filename << "\"!" << endl;
    return false;
  }

  QDomDocument doc("workflow");
  QDomElement docElem = doc.createElement("workflow");

  writeXML(doc, docElem);

  doc.appendChild(docElem);

  QTextStream stream(&file);
  doc.save(stream, 4);

  d->modified = false;

  return true;
}

void Document::readXML(const QDomElement& e)
{
  clear();

  // small hacklett to prevent lots of undo actions to appear
  d->undoing = true;

  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement e = n.toElement();
    if (!e.isNull()) {
      if (e.tagName() == "command") {
        QString id = e.attribute("id");
        if (!id.isNull()) {
          CommandFactoryBase* factory = CommandManager::self()->commandDescription(id)->factory();
          if (factory) {
            Command* cmd = factory->create(this);
            doInsertCommand(numRows(), cmd);
            // Allow command widget to connect to parameters
            kapp->processEvents();

            cmd->readXML(e);
          }
        }
      }
    }

    n = n.nextSibling();
  }

  d->undoing = false;

  emitUndoSignals();
}

void Document::writeXML(QDomDocument& doc, QDomElement& e)
{
  for (Iterator i = begin(); i != end(); ++i) {
    (*i)->writeXML(doc, e);
  }
}

ApplicationManager * WorKflow::Document::appManager( )
{
  if (!d->appManager) {
    d->appManager = new ApplicationManager(this);
  }
  return d->appManager;
}

#include "document.moc"
#include "document_p.moc"
