/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kapplication.h>
#include <kdebug.h>

// Qt includes
#include <qdom.h>

// WorKflow includes
#include "command.h"
#include "commanddescription.h"
#include "resultdescription.h"
#include "parameter.h"
#include "result.h"
#include "document.h"
#include "service.h"

using namespace WorKflow;

class Command::Private
{
public:
  Document* parent;

  CommandDescription* description;
  QString title;

  ServiceList services;

  ParameterList parameterList;
  ResultList resultList;

  Parameter* input;
  Result* output;

  QWidget* widget;
  int row;
  Document* children;
};

Command::Command(Document* parent, CommandDescription* description)
  : QObject(parent)
{
  d = new Private;
  d->parent = parent;
  d->description = description;
  d->input = 0;
  d->output = 0;
  d->widget = 0;
  d->row = -1;
  d->children = 0;
  d->title = d->description->name();

  typedef CommandDescription::ParamDescList Params;
  typedef CommandDescription::ResultDescList Results;

  Params params = description->parameters();
  for (Params::ConstIterator i = params.begin(); i != params.end(); ++i) {
    Parameter* param = new Parameter(this, *i);
    connect(param, SIGNAL(guiUpdateNeeded()), this, SIGNAL(guiUpdateNeeded()));
  }

  Results results = description->results();
  for (Results::ConstIterator i = results.begin(); i != results.end(); ++i) {
    Result* result = new Result(this, *i);
    connect(result, SIGNAL(guiUpdateNeeded()), this, SIGNAL(guiUpdateNeeded()));
  }

  ParameterList paramlist = connectableParameters();
  if (!paramlist.isEmpty())
    d->input = paramlist.first();

  ResultList resultlist = resultList();
  if (!resultlist.isEmpty())
    d->output = resultlist.first();
}

Command::~Command()
{
  for (ParameterList::Iterator i = d->parameterList.begin(); i != d->parameterList.end(); ++i)
    delete *i;

  for (ResultList::Iterator i = d->resultList.begin(); i != d->resultList.end(); ++i)
    delete *i;

  for (ServiceList::Iterator i = d->services.begin(); i != d->services.end(); ++i)
    delete *i;

  if (d->children)
    delete d->children;

  delete d;
}

QString Command::id()
{
  return d->description->id();
}

QString Command::title()
{
  return d->title;
}

QString Command::description()
{
  return d->description->description();
}

const Command::ParameterList& Command::parameterList()
{
  return d->parameterList;
}

const Command::ResultList& Command::resultList()
{
  return d->resultList;
}

Parameter* Command::parameter(const QString& id)
{
  for (ParameterList::ConstIterator i = d->parameterList.begin(); i != d->parameterList.end(); ++i)
    if ((*i)->id() == id)
      return *i;
  return 0;
}

Result* Command::result(const QString& id)
{
  for (ResultList::ConstIterator i = d->resultList.begin(); i != d->resultList.end(); ++i)
    if ((*i)->id() == id)
      return *i;
  return 0;
}

Value Command::value(const QString& paramId)
{
  return parameter(paramId)->value();
}

void Command::setResult(const QString& resultId, const Value& value)
{
  result(resultId)->setValue(value);
}

void Command::execute()
{
  fillResults();
}

void Command::reset()
{
  for (ParameterList::Iterator i = d->parameterList.begin(); i != d->parameterList.end(); ++i)
    (*i)->reset();
}

QWidget* Command::widget()
{
  return d->widget;
}

void Command::addParameter(Parameter* parameter)
{
  d->parameterList.append(parameter);
}

void Command::addResult(Result* result)
{
  d->resultList.append(result);
}

void Command::setWidget(QWidget* widget)
{
  d->widget = widget;
}

bool Command::canExecute()
{
  return true;
}

void Command::typeCheck(Result* prev)
{
  if (input())
    input()->resetType();
  if (output())
    output()->resetType();

  // check input type
  if (prev && input()) {
    Datatype* outType = prev->type();

    if (!input()->assignType(outType)) {
      // Type mismatch
      input()->setProblem(Slot::TypeMismatch);
      prev->setProblem(Slot::TypeMismatch);
    }
  } else if (!prev && input() && !input()->isOptional()) {
    input()->setProblem(Slot::MissingInput);
  }

  // set output type, if appropriate
  if (input() && output()) {
    if (output()->linkedParameter() == input()->id()) {
      output()->setType(input()->typeId());
    }
  }
}

int Command::row()
{
  return d->row;
}

void Command::setRow(int row)
{
  d->row = row;

  emit orderChanged(row);
}

QString Command::saveState()
{
  QDomDocument doc;
  QDomElement el = doc.createElement("state");
  writeXML(doc, el);
  doc.appendChild(el);

  kdDebug() << "Command State: " << doc.toString(4) << endl;

  return doc.toString(4);
}

void Command::loadState(const QString& state)
{
  QDomDocument doc;
  doc.setContent(state);
  readXML(doc.documentElement().firstChild().toElement());
}

void Command::updateParameters()
{
}

void Command::updateWidget()
{
}

void Command::notifyChange()
{
  updateParameters();
  emit modified(this);
}

bool Command::drop(QMimeSource*)
{
  return false;
}

Command::ParameterList Command::connectableParameters() const
{
  ParameterList res;
  for (ParameterList::ConstIterator i = d->parameterList.begin(); i != d->parameterList.end(); ++i)
    if ((*i)->source() != Parameter::GuiSource)
      res.append(*i);
  return res;
}

void Command::setInput(const QString& parameterId)
{
  d->input = parameter(parameterId);
  emit modified(this);
  emit typeCheckNeeded();
}

void Command::setOutput(const QString& resultId)
{
  d->output = result(resultId);
  emit modified(this);
  emit typeCheckNeeded();
}

Parameter* Command::input() const
{
  return d->input;
}

Result* Command::output() const
{
  return d->output;
}

void Command::setTitle(const QString& title)
{
  d->title = title;
  emit modified(this);
}

void Command::fillResults()
{
  typedef CommandDescription::ResultDescList Results;

  Results results = d->description->results();
  for (Results::ConstIterator i = results.begin(); i != results.end(); ++i) {
    QString paramId = (*i)->parameter();
    if (!paramId.isEmpty() && (*i)->copyValue()) {
      setResult((*i)->id(), value(paramId));
    }
  }
}

Document* Command::document() const
{
  return d->parent;
}

void Command::addService(Service* service)
{
  d->services.append(service);
  emit typeCheckNeeded();
}

void Command::removeService(Service* service)
{
  d->services.erase(d->services.find(service));
  emit typeCheckNeeded();
}

Command::ServiceList Command::services() const
{
  return d->services;
}

void Command::waitFor(QObject* sender, const char* signal)
{
  connect(sender, signal, this, SLOT(doneWaiting()));
  kapp->enter_loop();
}

void Command::doneWaiting()
{
  disconnect(this, SLOT(doneWaiting()));
  kapp->exit_loop();
}

void Command::readXML(const QDomElement& e)
{
  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement e = n.toElement();
    if (!e.isNull()) {
      if (e.tagName() == "title") {
        QString title = e.text();
        if (!title.isNull())
          d->title = title;
      } else if (e.tagName() == "input") {
        QString id = e.attribute("id");
        if (!id.isNull())
          setInput(id);
      } else if (e.tagName() == "output") {
        QString id = e.attribute("id");
        if (!id.isNull())
          setOutput(id);
      } else if (e.tagName() == "param") {
        QString id = e.attribute("id");
        Parameter* param = parameter(id);
        if (param)
          param->readXML(e);
      }
    }

    n = n.nextSibling();
  }

  if (d->children) {
    d->children->readXML(e);
  }

  updateWidget();

  emit guiUpdateNeeded();
}

void Command::writeXML(QDomDocument& doc, QDomElement& e)
{
  QDomElement cmdTag = doc.createElement("command");
  cmdTag.setAttribute("id", id());

  QDomElement titleTag = doc.createElement("title");
  titleTag.appendChild(doc.createTextNode(d->title));
  cmdTag.appendChild(titleTag);

  if (d->input) {
    QDomElement inputTag = doc.createElement("input");
    inputTag.setAttribute("id", d->input->id());
    cmdTag.appendChild(inputTag);
  }

  if (d->output) {
    QDomElement outputTag = doc.createElement("output");
    outputTag.setAttribute("id", d->output->id());
    cmdTag.appendChild(outputTag);
  }

  for (ParameterList::ConstIterator i = d->parameterList.begin(); i != d->parameterList.end(); ++i)
    (*i)->writeXML(doc, cmdTag);

  if (d->children) {
    d->children->writeXML(doc, cmdTag);
  }

  e.appendChild(cmdTag);
}

QString Command::originalTitle()
{
  return d->description->name();
}

bool Command::isContainer()
{
  return d->children != 0;
}

void Command::setContainer(bool container)
{
  if (d->children)
    delete d->children;

  if (container) {
    d->children = new Document(this);
    connect(d->children, SIGNAL(modified()), this, SLOT(notifyChange()));
  } else {
    d->children = 0;
  }
}

Document* Command::childDocument()
{
  return d->children;
}

#include "command.moc"
