/*
	keditmask.cpp - part of KDiskCat the KDE Disk Catalog.

	Copyright (c) 1999 Balzs Ternyi <terenyi@freemail.c3.hu>

	KDiskCat is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
			
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
					
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <qkeycode.h>
#include <kapp.h>
#include "keditmask.h"

KEditMask::KEditMask(QWidget *parent=0, const char *name=0)
	:QLineEdit(parent,name)
{
	mask_definition=0;
}

KEditMask::~KEditMask()
{
}

bool KEditMask::setData(void *ref)
{
	QString init;
	bool ret;

	if (mask_definition->setData(&init,ref))
	{
     	setText(init);
     	setCursorPosition(0);
   	tunePosition(1);
     	deselect(); // This was required, because of a QT bug or feature ;)
     	ret=true;
 	}
 	else
 	{
 		ret=false;
 	}
 	
 	return ret;
}

bool KEditMask::getData(void *ref)
{
	QString str=text();
	
 	return mask_definition->getData(&str,ref);
}

void KEditMask::setMaskDefinition(KEditMaskDefinition *p_def)
{
	if (p_def)
	{
		if (mask_definition)
		{
			disconnect(mask_definition,SIGNAL(definitionChanged()),this,SLOT(slotDefinitionChanged()));
		}
		mask_definition=p_def;
		connect(mask_definition,SIGNAL(definitionChanged()),this,SLOT(slotDefinitionChanged()));
		slotDefinitionChanged();
	}
}

KEditMaskDefinition* KEditMask::maskDefinition()
{
	return mask_definition;
}

void KEditMask::slotDefinitionChanged()
{
	setMaxLength(mask_definition->mask()->length());
}

bool KEditMask::isValidKey(QKeyEvent *e)
{
	bool ret;
	
	if ((e->ascii() < 32 ||
			mask_definition->validChars()->contains((char) e->ascii())) &&
			e->state()!=ControlButton)
	{
		ret=true;
	}
	else
	{
		ret=false;
	}
	
	return ret;
}

void KEditMask::tunePosition(int direction)
{
	int newcursorpos;
	
  	while (!mask_definition->dataFieldChars()->contains(mask_definition->mask()->mid(cursorPosition(),1)) &&
  				cursorPosition() < (int)mask_definition->mask()->length() && cursorPosition() >= 0)
  	{
  		newcursorpos=cursorPosition() + direction;
  		if (newcursorpos<0 || newcursorpos>(int)mask_definition->mask()->length())
  		{
  			direction*= -1;
  		}
  		else
  		{
  			setCursorPosition(newcursorpos);
  		}
  	}
}

void KEditMask::keyPressEvent(QKeyEvent *e)
{
	QString str,temp,before;
	bool processkey=true;
	int newcursorpos;
	int direction=0;
	
	if (isValidKey(e))
	{
		setUpdatesEnabled(false);

		switch(e->key())
		{
			case Key_Left:
				direction= -1;
				break;
			case Key_Right:
				direction= 1;
				break;
			case Key_Home:
     			direction= 1;
				break;
			case Key_End:
     			direction= -1;
				break;
			case Key_Backspace:
				setCursorPosition(cursorPosition() - 1);
				direction= -1;
				processkey=false;
				break;
			default:
        		processkey=false;
				if (mask_definition->dataFieldChars()->contains(mask_definition->mask()->mid(cursorPosition(),1)))
				{
         		if (mask_definition->validChars()->contains((char) e->ascii()))
         		{
         			str=text();
         			before=str;
         			temp.fill((char)e->ascii(),1);
         			str.replace(cursorPosition(),1,(const char*) temp);
         			if (!mask_definition->isValid(&str))
         			{
         				if (mask_definition->doMagic(&str,&str) && before!=str)
         				{
         					newcursorpos=cursorPosition() + 1;
         					setText(str);
        						setCursorPosition(newcursorpos);
         	    			direction= 1;
         				}
         				else
         				{
         					KApplication::beep();
         				}
         			}
         			else
         			{
      					newcursorpos=cursorPosition() + 1;
      					setText(str);
     						setCursorPosition(newcursorpos);
      	    			direction= 1;
         			}
         		}
         	}
         	else
         	{
  					KApplication::beep();
  				}
        		break;
		}
				
  		if (processkey)
  		{
			QLineEdit::keyPressEvent(e);
		}

		if (direction != 0) tunePosition(direction);
		
		setUpdatesEnabled(true);
   	repaint();
	}
}

/*********************************************************************

	KEditMaskDefinition
	
*********************************************************************/
KEditMaskDefinition::KEditMaskDefinition(QObject * parent=0, const char * name=0)
	:QObject(parent,name)
{
}

KEditMaskDefinition::~KEditMaskDefinition()
{
}
	
void KEditMaskDefinition::setValidChars(const QString chars)
{
	qs_valid_chars=chars;
}

void KEditMaskDefinition::setDataFieldChars(const QString chars)
{
	qs_datafield_chars=chars;
}

bool KEditMaskDefinition::setMask(QString p_mask)
{
	qs_mask=p_mask;
	emit definitionChanged();
	return true;
}

const QString* KEditMaskDefinition::mask()
{
	return &qs_mask;
}

const QString* KEditMaskDefinition::validChars()
{
	return &qs_valid_chars;
}

const QString* KEditMaskDefinition::dataFieldChars()
{
	return &qs_datafield_chars;
}

bool KEditMaskDefinition::extractData(const QString* input, char mask_element, QString* ref)
{
	int start_pos;
	int end_pos;
	bool ret;
	QString comp_str;
	QString str;
	
	start_pos=qs_mask.find(mask_element);
	end_pos=qs_mask.findRev(mask_element);
	
	if (start_pos != -1 && end_pos != -1)
	{
   	comp_str.fill(mask_element,end_pos - start_pos + 1);
   	
   	if (qs_mask.mid(start_pos,end_pos - start_pos + 1) == comp_str)
   	{
   		*ref=input->mid(start_pos,end_pos - start_pos + 1);
   		ret=true;
   	}
   	else
   	{
   		ret=false;
			warning("KEditMask::extractData(2): Invalid mask");
   	}
	}
	else
	{
		// This mask element is not present is the mask,
		// the caller has to do something
		ret=false;
	}
	
	return ret;
}

bool KEditMaskDefinition::fillData(const QString* data, char mask_element, QString* ref)
{
	int start_pos;
	int end_pos;
	bool ret=true;
	QString comp_str;
	QString str;
	
	start_pos=qs_mask.find(mask_element);
	end_pos=qs_mask.findRev(mask_element);
	
	if (start_pos != -1 && end_pos != -1)
	{
   	comp_str.fill(mask_element,end_pos - start_pos + 1);
   	
   	if (qs_mask.mid(start_pos,end_pos - start_pos + 1) == comp_str)
   	{
   		if (comp_str.length()==data->length())
   		{
      		ref->replace(start_pos,end_pos - start_pos + 1,(const char*) *data);
   		}
   		else
   		{
   			ret=false;
   			warning("KEditMask::fillData(3): Invalid data");
   		}
   	}
   	else
   	{
   		ret=false;
			warning("KEditMask::fillData(2): Invalid mask");
   	}
	}
	
	return ret;
}

/*********************************************************************

	KEditMaskDefinitionDate
	
*********************************************************************/
KEditMaskDefinitionDate::KEditMaskDefinitionDate(QObject * parent=0, const char * name=0, const char * p_mask)
	:KEditMaskDefinition(parent,name)
{
	setValidChars("0123456789");
	setDataFieldChars("ymd");
	setMask(p_mask);
}

KEditMaskDefinitionDate::~KEditMaskDefinitionDate()
{
}

bool KEditMaskDefinitionDate::setData(QString* data_string, void* data_holder)
{
	QString str;
	bool ret;
	QDate* date_view;
	
	date_view=(QDate*) data_holder;
	*data_string=(const char*) *mask();
	str.setNum(date_view->year());
	ret=fillData(&str,'y',data_string);
	
	if (ret)
	{
		str.setNum(date_view->month());
		str=str.rightJustify(2,'0');
		ret=fillData(&str,'m',data_string);
		if (ret)
		{
			str.setNum(date_view->day());
			str=str.rightJustify(2,'0');
			ret=fillData(&str,'d',data_string);
		}
	}
	
	return ret;
}

bool KEditMaskDefinitionDate::getData(const QString* data_string, void* data_holder)
{
	int y,m,d;
	QString str;
	QDate* date_view;
	
	date_view=(QDate*) data_holder;
	
	if (!extractData(data_string,'y',&str)) str="1900";
	y=str.toInt();
	if (!extractData(data_string,'m',&str)) str="1";
	m=str.toInt();
	if (!extractData(data_string,'d',&str)) str="1";
	d=str.toInt();
	
	return date_view->setYMD(y,m,d);
}

bool KEditMaskDefinitionDate::isValid(const QString* data)
{
	int y,m,d;
	QString str;
	
	if (!extractData(data,'y',&str)) str="1900";
	y=str.toInt();
	if (!extractData(data,'m',&str)) str="1";
	m=str.toInt();
	if (!extractData(data,'d',&str)) str="1";
	d=str.toInt();

	if (y == 0) return false;
		
	return QDate::isValid(y,m,d);
}

bool KEditMaskDefinitionDate::doMagic(const QString* input, QString* ref)
{
	bool ret=true;
	int y,m,d;
	QDate date;
	QString str;
	
	if (!extractData(input,'y',&str)) str="1900";
	y=str.toInt();
	if (!extractData(input,'m',&str)) str="1";
	m=str.toInt();
	if (!extractData(input,'d',&str)) str="1";
	d=str.toInt();
	if (!QDate::isValid(y,1,1) || y==0) // The year is invalid
	{
		if (y<1753)
		{
			y=1753;
		}
		else
		{
			ret=false;
		}
	}
	else
	{
		if (!QDate::isValid(y,m,1)) // The month is invalid
	  	{
	  		if (m > 12)
	  		{
	  			m=12;
	  		}
	  		else // Month is 00
	  		{
	  			m=1;
	  		}
	  	}
	  	else // The day is invalid
	  	{
	  		if (d>15)
	  		{
	  			date.setYMD(y,m,1);
	  			d=date.daysInMonth();
	  		}
	  		else // The day is 00
	  		{
	  			d=1;
	  		}
	  	}
	}
	if (ret)
	{
		ret=QDate::isValid(y,m,d);
		if (ret)
		{
			*ref=(const char *) *mask();
			str.setNum(y);
			fillData(&str,'y',ref);
			str.setNum(m);
			str=str.rightJustify(2,'0');
			fillData(&str,'m',ref);
			str.setNum(d);
			str=str.rightJustify(2,'0');
			fillData(&str,'d',ref);					
		}
	}
	
	return ret;
}

/*********************************************************************

	KEditMaskDefinitionTime
	
*********************************************************************/
KEditMaskDefinitionTime::KEditMaskDefinitionTime(QObject * parent=0, const char * name=0, const char * p_mask)
	:KEditMaskDefinition(parent,name)
{
	setValidChars("0123456789");
	setDataFieldChars("hMsr");
	setMask(p_mask);
}

KEditMaskDefinitionTime::~KEditMaskDefinitionTime()
{
}

bool KEditMaskDefinitionTime::setData(QString* data_string, void* data_holder)
{
	QString str;
	bool ret;
	QTime* time_view;
	
	time_view=(QTime*) data_holder;
	*data_string=(const char*) *mask();
	str.setNum(time_view->hour());
	str=str.rightJustify(2,'0');
	ret=fillData(&str,'h',data_string);
	
	if (ret)
	{
		str.setNum(time_view->minute());
		str=str.rightJustify(2,'0');
		ret=fillData(&str,'M',data_string);
		if (ret)
		{
			str.setNum(time_view->second());
			str=str.rightJustify(2,'0');
			ret=fillData(&str,'s',data_string);
   		if (ret)
   		{
   			str.setNum(time_view->msec());
   			str=str.rightJustify(3,'0');
   			ret=fillData(&str,'r',data_string);
   		}
		}
	}
	
	return ret;
}

bool KEditMaskDefinitionTime::getData(const QString* data_string, void* data_holder)
{
	int h,M,s,r;
	QString str;
	QTime* time_view;
	
	time_view=(QTime*) data_holder;
	
  	if (!extractData(data_string,'h',&str)) str="0";
  	h=str.toInt();
  	if (!extractData(data_string,'M',&str)) str="0";
  	M=str.toInt();
  	if (!extractData(data_string,'s',&str)) str="0";
  	s=str.toInt();
	if (!extractData(data_string,'r',&str)) str="0";
	r=str.toInt();
	return time_view->setHMS(h,M,s,r);
}

bool KEditMaskDefinitionTime::isValid(const QString* data)
{
	int h,M,s,r;
	QString str;

   if (!extractData(data,'h',&str)) str="0";
   h=str.toInt();
   if (!extractData(data,'M',&str)) str="0";
   M=str.toInt();
   if (!extractData(data,'s',&str)) str="0";
   s=str.toInt();
   if (!extractData(data,'r',&str)) str="0";
   r=str.toInt();
   return QTime::isValid(h,M,s,r);
}

bool KEditMaskDefinitionTime::doMagic(const QString* input, QString* ref)
{
	bool ret=true;
	int h,M,s,r;
	QString str;
	
	if (!extractData(input,'h',&str)) str="0";
	h=str.toInt();
	if (!extractData(input,'M',&str)) str="0";
	M=str.toInt();
	if (!extractData(input,'s',&str)) str="0";
	s=str.toInt();
	if (!extractData(input,'r',&str)) str="0";
	r=str.toInt();
	if (h>23)
	{
		h=23;
	}
	ret=QTime::isValid(h,M,s,r);
	if (ret)
	{
		*ref=(const char *) *mask();
		str.setNum(h);
		str=str.rightJustify(2,'0');
  		fillData(&str,'h',ref);
  		str.setNum(M);
  		str=str.rightJustify(2,'0');
  		fillData(&str,'M',ref);
  		str.setNum(s);
  		str=str.rightJustify(2,'0');
  		fillData(&str,'s',ref);
  		str.setNum(r);
  		str=str.rightJustify(3,'0');
  		fillData(&str,'r',ref);
	}

	return ret;
}

/*********************************************************************

	KEditMaskDefinitionDateTime
	
*********************************************************************/
KEditMaskDefinitionDateTime::KEditMaskDefinitionDateTime(QObject * parent=0, const char * name=0, const char * p_mask)
	:KEditMaskDefinition(parent,name)
{
	setValidChars("0123456789");
	setDataFieldChars("ymdhMsr");
	setMask(p_mask);
}

KEditMaskDefinitionDateTime::~KEditMaskDefinitionDateTime()
{
}

bool KEditMaskDefinitionDateTime::setData(QString* data_string, void* data_holder)
{
	QString str;
	bool ret;
	QDateTime* datetime_view;
	
	datetime_view=(QDateTime*) data_holder;
	*data_string=(const char*) *mask();
	str.setNum(datetime_view->date().year());
	ret=fillData(&str,'y',data_string);
	
	if (ret)
	{
		str.setNum(datetime_view->date().month());
		str=str.rightJustify(2,'0');
		ret=fillData(&str,'m',data_string);
		if (ret)
		{
			str.setNum(datetime_view->date().day());
			str=str.rightJustify(2,'0');
			ret=fillData(&str,'d',data_string);
			if (ret)
			{
            str.setNum(datetime_view->time().hour());
            str=str.rightJustify(2,'0');
            ret=fillData(&str,'h',data_string);
         	if (ret)
         	{
         		str.setNum(datetime_view->time().minute());
         		str=str.rightJustify(2,'0');
         		ret=fillData(&str,'M',data_string);
         		if (ret)
         		{
         			str.setNum(datetime_view->time().second());
         			str=str.rightJustify(2,'0');
         			ret=fillData(&str,'s',data_string);
            		if (ret)
            		{
            			str.setNum(datetime_view->time().msec());
            			str=str.rightJustify(3,'0');
            			ret=fillData(&str,'r',data_string);
            		}
         		}
				}
			}
		}
	}
	
	return ret;
}

bool KEditMaskDefinitionDateTime::getData(const QString* data_string, void* data_holder)
{
	int y,m,d,h,M,s,r;
	QString str;
	QDateTime* datetime_view;
	QDate date;
	QTime time;
	bool ret;
	
	datetime_view=(QDateTime*) data_holder;
	
	if (!extractData(data_string,'y',&str)) str="1900";
	y=str.toInt();
	if (!extractData(data_string,'m',&str)) str="1";
	m=str.toInt();
	if (!extractData(data_string,'d',&str)) str="1";
	d=str.toInt();
  	if (!extractData(data_string,'h',&str)) str="0";
  	h=str.toInt();
  	if (!extractData(data_string,'M',&str)) str="0";
  	M=str.toInt();
  	if (!extractData(data_string,'s',&str)) str="0";
  	s=str.toInt();
	if (!extractData(data_string,'r',&str)) str="0";
	r=str.toInt();
	
	ret=date.setYMD(y,m,d);
	if (ret)
	{
		ret=time.setHMS(h,M,s,r);
		if (ret)
		{
			datetime_view->setDate(date);
			datetime_view->setTime(time);
		}
	}
	
	return ret;
}

bool KEditMaskDefinitionDateTime::isValid(const QString* data)
{
	int y,m,d,h,M,s,r;
	QString str;
	
	if (!extractData(data,'y',&str)) str="1900";
	y=str.toInt();
	if (!extractData(data,'m',&str)) str="1";
	m=str.toInt();
	if (!extractData(data,'d',&str)) str="1";
	d=str.toInt();
   if (!extractData(data,'h',&str)) str="0";
   h=str.toInt();
   if (!extractData(data,'M',&str)) str="0";
   M=str.toInt();
   if (!extractData(data,'s',&str)) str="0";
   s=str.toInt();
   if (!extractData(data,'r',&str)) str="0";
   r=str.toInt();

   if (y==0) return false;

	return QDate::isValid(y,m,d) && QTime::isValid(h,M,s,r);
}

bool KEditMaskDefinitionDateTime::doMagic(const QString* input, QString* ref)
{
	bool ret=true;
	int y,m,d,h,M,s,r;
	QDate date;
	QString str;
	
	if (!extractData(input,'y',&str)) str="1900";
	y=str.toInt();
	if (!extractData(input,'m',&str)) str="1";
	m=str.toInt();
	if (!extractData(input,'d',&str)) str="1";
	d=str.toInt();
	if (!extractData(input,'h',&str)) str="0";
	h=str.toInt();
	if (!extractData(input,'M',&str)) str="0";
	M=str.toInt();
	if (!extractData(input,'s',&str)) str="0";
	s=str.toInt();
	if (!extractData(input,'r',&str)) str="0";
	r=str.toInt();
	
	if (!QDate::isValid(y,1,1) || y==0) // The year is invalid
	{
		if (y<1753)
		{
			y=1753;
		}
		else
		{
			ret=false;
		}
	}
	else
	{
		if (!QDate::isValid(y,m,1)) // The month is invalid
	  	{
	  		if (m > 12)
	  		{
	  			m=12;
	  		}
	  		else // Month is 00
	  		{
	  			m=1;
	  		}
	  	}
	  	else
	  	{
			if (!date.isValid(y,m,d)) // The day is invalid
			{
   	  		if (d>15)
   	  		{
   	  			date.setYMD(y,m,1);
   	  			d=date.daysInMonth();
   	  		}
   	  		else // The day is 00
   	  		{
   	  			d=1;
   	  		}
   		}
   		else
   		{
  				if (h>23) // The hour is invalid
  				{
  					h=23;
  				}
   		}
	  	}
	}
	if (ret)
	{
		ret=QDate::isValid(y,m,d) && QTime::isValid(h,M,s,r);
		if (ret)
		{
			*ref=(const char *) *mask();
			str.setNum(y);
			fillData(&str,'y',ref);
			str.setNum(m);
			str=str.rightJustify(2,'0');
			fillData(&str,'m',ref);
			str.setNum(d);
			str=str.rightJustify(2,'0');
			fillData(&str,'d',ref);					
   		str.setNum(h);
   		str=str.rightJustify(2,'0');
     		fillData(&str,'h',ref);
     		str.setNum(M);
     		str=str.rightJustify(2,'0');
     		fillData(&str,'M',ref);
     		str.setNum(s);
     		str=str.rightJustify(2,'0');
     		fillData(&str,'s',ref);
     		str.setNum(r);
     		str=str.rightJustify(3,'0');
     		fillData(&str,'r',ref);
		}
	}
	
	return ret;
}

/*********************************************************************

	KEditMaskDefinitionNumeric
	
*********************************************************************/
KEditMaskDefinitionNumeric::KEditMaskDefinitionNumeric(QObject * parent=0, const char * name=0, const char * p_mask)
	:KEditMaskDefinition(parent,name)
{
	setValidChars("0123456789");
	setDataFieldChars("9");
	setMask(p_mask);
}

KEditMaskDefinitionNumeric::~KEditMaskDefinitionNumeric()
{
}

bool KEditMaskDefinitionNumeric::setData(QString* data_string, void* data_holder)
{
	QString str;
	bool ret;
	QTime* time_view;
	
	time_view=(QTime*) data_holder;
	*data_string=(const char*) *mask();
	str.setNum(time_view->hour());
	str=str.rightJustify(2,'0');
	ret=fillData(&str,'h',data_string);
	
	if (ret)
	{
		str.setNum(time_view->minute());
		str=str.rightJustify(2,'0');
		ret=fillData(&str,'M',data_string);
		if (ret)
		{
			str.setNum(time_view->second());
			str=str.rightJustify(2,'0');
			ret=fillData(&str,'s',data_string);
   		if (ret)
   		{
   			str.setNum(time_view->msec());
   			str=str.rightJustify(3,'0');
   			ret=fillData(&str,'r',data_string);
   		}
		}
	}
	
	return ret;
}

bool KEditMaskDefinitionNumeric::getData(const QString* data_string, void* data_holder)
{
	int h,M,s,r;
	QString str;
	QTime* time_view;
	
	time_view=(QTime*) data_holder;
	
  	if (!extractData(data_string,'h',&str)) str="0";
  	h=str.toInt();
  	if (!extractData(data_string,'M',&str)) str="0";
  	M=str.toInt();
  	if (!extractData(data_string,'s',&str)) str="0";
  	s=str.toInt();
	if (!extractData(data_string,'r',&str)) str="0";
	r=str.toInt();
	return time_view->setHMS(h,M,s,r);
}

bool KEditMaskDefinitionNumeric::isValid(const QString* data)
{
   return true;
}

bool KEditMaskDefinitionNumeric::doMagic(const QString* input, QString* ref)
{
	return true;
}

/*bool KEditMask::isValid(QString data)
{
	bool ret=false;
	int y,m,d;
	int h,M,s,r;
	QString str;
	QDate date;
	QTime time;
	
	switch (style)
	{
		case KEditMask::date:
			if (!extractData(data,'y',&str)) str="1900";
			y=str.toInt();
			if (!extractData(data,'m',&str)) str="1";
			m=str.toInt();
			if (!extractData(data,'d',&str)) str="1";
			d=str.toInt();
			ret=date.isValid(y,m,d);
			break;
		case KEditMask::time:
			if (!extractData(data,'h',&str)) str="0";
			h=str.toInt();
			if (!extractData(data,'M',&str)) str="0";
			M=str.toInt();
			if (!extractData(data,'s',&str)) str="0";
			s=str.toInt();
			if (!extractData(data,'r',&str)) str="0";
			r=str.toInt();
			ret=time.isValid(h,M,s,r);
			break;
		case KEditMask::datetime:
			if (!extractData(data,'y',&str)) str="1900";
			y=str.toInt();
			if (!extractData(data,'m',&str)) str="1";
			m=str.toInt();
			if (!extractData(data,'d',&str)) str="1";
			d=str.toInt();
			if (!extractData(data,'h',&str)) str="0";
			h=str.toInt();
			if (!extractData(data,'M',&str)) str="0";
			M=str.toInt();
			if (!extractData(data,'s',&str)) str="0";
			s=str.toInt();
			if (!extractData(data,'r',&str)) str="0";
			r=str.toInt();
			ret=date.isValid(y,m,d) && time.isValid(h,M,s,r);
			break;
		case KEditMask::numeric:
			ret=true;
			break;
		case KEditMask::userdefined:
			break;
	}
	
	return ret;
}*/

/*bool KEditMask::doMagic(QString input, QString* ref)
{
	bool ret=true;
	int y,m,d;
	int h,M,s,r;
	QString str;
	QDate date;
	QTime time;
	
	switch (style)
	{
		case KEditMask::date:
			if (!extractData(input,'y',&str)) str="1900";
			y=str.toInt();
			if (!extractData(input,'m',&str)) str="1";
			m=str.toInt();
			if (!extractData(input,'d',&str)) str="1";
			d=str.toInt();
			if (!date.isValid(y,1,1)) // The year is invalid
			{
				if (y<1753)
				{
					y=1753;
				}
				else
				{
					ret=false;
				}
			}
			else
			{
				if (!date.isValid(y,m,1)) // The month is invalid
  				{
  					if (m > 12)
  					{
  						m=12;
  					}
  					else // Month is 00
  					{
  						m=1;
  					}
  				}
  				else // The day is invalid
  				{
  					if (d>15)
  					{
  						date.setYMD(y,m,1);
  						d=date.daysInMonth();
  					}
  					else // The day is 00
  					{
  						d=1;
  					}
  				}
			}
			if (ret)
			{
				ret=date.isValid(y,m,d);
				if (ret)
				{
					*ref=mask;
					str.setNum(y);
					fillData(str,'y',ref);
					str.setNum(m);
					str=str.rightJustify(2,'0');
					fillData(str,'m',ref);
					str.setNum(d);
					str=str.rightJustify(2,'0');
					fillData(str,'d',ref);					
				}
			}
			break;
		case KEditMask::time:
			if (!extractData(input,'h',&str)) str="0";
			h=str.toInt();
			if (!extractData(input,'M',&str)) str="0";
			M=str.toInt();
			if (!extractData(input,'s',&str)) str="0";
			s=str.toInt();
			if (!extractData(input,'r',&str)) str="0";
			r=str.toInt();
			if (h>23)
			{
				h=23;
			}
			ret=time.isValid(h,M,s,r);
  			if (ret)
  			{
  				*ref=mask;
  				str.setNum(h);
  				str=str.rightJustify(2,'0');
  				fillData(str,'h',ref);
  				str.setNum(M);
  				str=str.rightJustify(2,'0');
  				fillData(str,'M',ref);
  				str.setNum(s);
  				str=str.rightJustify(2,'0');
  				fillData(str,'s',ref);
  				str.setNum(r);
  				str=str.rightJustify(3,'0');
  				fillData(str,'r',ref);
  			}
			break;
		case KEditMask::datetime:
			if (!extractData(input,'y',&str)) str="1900";
			y=str.toInt();
			if (!extractData(input,'m',&str)) str="1";
			m=str.toInt();
			if (!extractData(input,'d',&str)) str="1";
			d=str.toInt();
			if (!extractData(input,'h',&str)) str="0";
			h=str.toInt();
			if (!extractData(input,'M',&str)) str="0";
			M=str.toInt();
			if (!extractData(input,'s',&str)) str="0";
			s=str.toInt();
			if (!extractData(input,'r',&str)) str="0";
			r=str.toInt();
			if (!date.isValid(y,1,1)) // The year is invalid
			{
				if (y<1753)
				{
					y=1753;
				}
				else
				{
					ret=false;
				}
			}
			else
			{
				if (!date.isValid(y,m,1)) // The month is invalid
  				{
  					if (m > 12)
  					{
  						m=12;
  					}
  					else // Month is 00
  					{
  						m=1;
  					}
  				}
  				else
  				{
  					if (!date.isValid(y,m,d)) // The day is invalid
  					{
     					if (d>15)
     					{
     						date.setYMD(y,m,1);
     						d=date.daysInMonth();
     					}
     					else // The day is 00
     					{
     						d=1;
     					}
     				}
     				else // The problem is in the time part
     				{
     					if (h>23) // The hour is invalid
     					{
     						h=23;
     					}
     				}
  				}
			}
			if (ret)
			{
				ret=date.isValid(y,m,d) && time.isValid(h,M,s,r);
				if (ret)
				{
					*ref=mask;
					str.setNum(y);
					fillData(str,'y',ref);
					str.setNum(m);
					str=str.rightJustify(2,'0');
					fillData(str,'m',ref);
					str.setNum(d);
					str=str.rightJustify(2,'0');
					fillData(str,'d',ref);					
   				str.setNum(h);
   				str=str.rightJustify(2,'0');
   				fillData(str,'h',ref);
   				str.setNum(M);
   				str=str.rightJustify(2,'0');
   				fillData(str,'M',ref);
   				str.setNum(s);
   				str=str.rightJustify(2,'0');
   				fillData(str,'s',ref);					
   				str.setNum(r);
   				str=str.rightJustify(3,'0');
   				fillData(str,'r',ref);					
				}
			}
			break;
		case KEditMask::numeric:
			break;
		case KEditMask::userdefined:
			break;
	}
	
	return ret;
}*/

/*bool KEditMask::getData(QDate* ref)
{
	int y,m,d;
	QString str;
	
	if (style==KEditMask::date)
	{
		if (!extractData(text(),'y',&str)) str="1900";
		y=str.toInt();
		if (!extractData(text(),'m',&str)) str="1";
		m=str.toInt();
		if (!extractData(text(),'d',&str)) str="1";
		d=str.toInt();
		return ref->setYMD(y,m,d);
	}
	else
	{
		return false;
	}
}

bool KEditMask::getData(QTime* ref)
{
	int h,M,s,r;
	QString str;
	
	if (style==KEditMask::time)
	{
  		if (!extractData(text(),'h',&str)) str="0";
  		h=str.toInt();
  		if (!extractData(text(),'M',&str)) str="0";
  		M=str.toInt();
  		if (!extractData(text(),'s',&str)) str="0";
  		s=str.toInt();
  		if (!extractData(text(),'r',&str)) str="0";
  		r=str.toInt();
		return ref->setHMS(h,M,s,r);
	}
	else
	{
		return false;
	}
}

bool KEditMask::getData(QDateTime* ref)
{
	int y,m,d;
	int h,M,s,r;
	QDate date;
	QTime time;
	QString str;
	bool ret;
	
	if (style==KEditMask::datetime)
	{
		if (!extractData(text(),'y',&str)) str="1900";
		y=str.toInt();
		if (!extractData(text(),'m',&str)) str="1";
		m=str.toInt();
		if (!extractData(text(),'d',&str)) str="1";
		d=str.toInt();
		ret=date.setYMD(y,m,d);
		if (ret)
		{
     		if (!extractData(text(),'h',&str)) str="0";
     		h=str.toInt();
     		if (!extractData(text(),'M',&str)) str="0";
     		M=str.toInt();
     		if (!extractData(text(),'s',&str)) str="0";
     		s=str.toInt();
     		if (!extractData(text(),'r',&str)) str="0";
     		r=str.toInt();
   		ret=time.setHMS(h,M,s,r);
   		if (ret)
   		{
   			ref->setDate(date);
  				ref->setTime(time);
   		}
  		}
	}
	else
	{
		ret=false;
	}
	
	return ret;
}*/