/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <kapp.h>

#include "pcidevice.h"


PCIDevice::PCIDevice(uint bus, uint slot, uint func, uint vendorid, 
		     uint deviceid, baseclass bclass, 
		     pciclass pclass) : _vendorname(""),
		     			_devicename(""),
		     			_bus(bus),
		     			_slot(slot),
					_func(func),
					_vendorid(vendorid),
					_deviceid(deviceid)
{
  _pciclass = (pciclass) ((((uint) bclass) << 8)+((uint) pclass));
  
  lookupData();

  // On the long term PCIBaseClasses have to disappear in Device

  if (_pciclass == PCIClassStorageSCSI) _baseclass = BaseClassSCSIController;
  else                                  _baseclass = bclass;
}    


QString  PCIDevice::deviceClassName()
{
  switch (_pciclass)
  {
    case PCIClassUnclassifiedNonVGA    : return QString("Non-VGA unclassified device");
    case PCIClassUnclassifiedVGA       : return QString("VGA compatible unclassified device");

    case PCIClassStorageSCSI           : return QString("SCSI storage controller");
    case PCIClassStorageIDE            : return QString("IDE interface");
    case PCIClassStorageFloppy         : return QString("Floppy disk controller");	  
    case PCIClassStorageIPI            : return QString("IPI bus controller");
    case PCIClassStorageRAID           : return QString("RAID bus controller");
    case PCIClassStorageOther          : return QString("Unknown mass storage controller");

    case PCIClassNetworkEthernet       : return QString("Ethernet controller");
    case PCIClassNetworkTokenring      : return QString("Token ring network controller");
    case PCIClassNetworkFDDI 	       : return QString("FDDI network controller");
    case PCIClassNetworkATM  	       : return QString("ATM network controller");
    case PCIClassNetworkOther	       : return QString("Network controller");

    case PCIClassDisplayVGA  	       : return QString("VGA compatible controller");
    case PCIClassDisplayXGA  	       : return QString("XGA compatible controller");
    case PCIClassDisplayOther	       : return QString("Display controller");

    case PCIClassMultimediaVideo       : return QString("Multimedia video controller");
    case PCIClassMultimediaAudio       : return QString("Multimedia audio controller");
    case PCIClassMultimediaOther       : return QString("Multimedia controller"); 

    case PCIClassMemoryRAM	       : return QString("RAM memory");
    case PCIClassMemoryFlash 	       : return QString("FLASH memory");
    case PCIClassMemoryOther 	       : return QString("Memory");

    case PCIClassBridgeHost  	       : return QString("Host bridge");
    case PCIClassBridgeISA	       : return QString("ISA bridge");
    case PCIClassBridgeEISA  	       : return QString("EISA bridge");
    case PCIClassBridgeMC	       : return QString("MicroChannel bridge");
    case PCIClassBridgePCI	       : return QString("PCI bridge");
    case PCIClassBridgePCMCIA	       : return QString("PCMCIA bridge");
    case PCIClassBridgeNuBus 	       : return QString("NuBus bridge");
    case PCIClassBridgeCardBus	       : return QString("CardBus bridge");
    case PCIClassBridgeOther 	       : return QString("Bridge");

    case PCIClassCommunicationSerial   : return QString("Serial controller");
    case PCIClassCommunicationParallel : return QString("Parallel controller");
    case PCIClassCommunicationOther    : return QString("Communication controller");

    case PCIClassSystemPIC	       : return QString("PIC");
    case PCIClassSystemDMA	       : return QString("DMA controller");
    case PCIClassSystemTimer 	       : return QString("Timer");
    case PCIClassSystemRTC	       : return QString("Real time clock");
    case PCIClassSystemOther 	       : return QString("System peripheral");
 
    case PCIClassInputKeyboard	       : return QString("Keyboard controller");
    case PCIClassInputPen	       : return QString("Digitizer Pen");
    case PCIClassInputMouse  	       : return QString("Mouse controller");
    case PCIClassInputOther  	       : return QString("Input device controller");

    case PCIClassDockingGeneric	       : return QString("Generic Docking Station");
    case PCIClassDockingOther	       : return QString("Docking Station");

    case PCIClassProcessor386	       : return QString("386");
    case PCIClassProcessor486	       : return QString("486");
    case PCIClassProcessorPentium      : return QString("Pentium");
    case PCIClassProcessorAlpha	       : return QString("Alpha");
    case PCIClassProcessorPowerPC      : return QString("Power PC");
    case PCIClassProcessorCo 	       : return QString("Co-processor");

    case PCIClassSerialFirewire	       : return QString("FireWire (IEEE 1394)");
    case PCIClassSerialACCESS	       : return QString("ACCESS Bus");
    case PCIClassSerialSSA	       : return QString("SSA");
    case PCIClassSerialUSB	       : return QString("USB Controller");
    case PCIClassSerialFiber 	       : return QString("Fiber Channel");
    
    default                            : return QString("Unknown device");
  }
}


QString  PCIDevice::deviceName()
{
  return _devicename;
}


Device::devstate  PCIDevice::deviceState()
{
  Resource  *res;

  /*
   * If the device requires I/O addresses, I guess there must be
   * a range of 2 bytes at least; so if range > 1 that indicates
   * those I/O addresse were found in /proc/ioports and thus the
   * driver for this device must have registered them, that is,
   * the driver is present - otherwise we don't know.
   ***************************************************************/
  
  for (res = firstResource(); res; res = nextResource())
    if ((res->resourceType() == Resource::ResIOAddress) && (res->resourceRange() > 1)) 
      return DeviceStateReady;
    
  return DeviceStateUnknown;
}


QString  PCIDevice::vendorName()
{
  return _vendorname;
}

/*
 * Private methods
 ******************/
 
void  PCIDevice::lookupData()
{
  QString  idfile = kapp->kde_datadir()+"/ksysctrl/pci.ids";
  QString  entry;
  char     buffer[256];
  FILE     *fhd;
  uint     vid, did;
  
  if ((fhd = fopen(idfile.data(), "r")))
  {
    while (fgets(buffer, sizeof(buffer), fhd))
    {
      entry = QString(buffer);
      
      if (entry.length() > 4)					// Line contains data
      {
        entry = entry.simplifyWhiteSpace();
	
        if ((entry.at(0) != ' ') && (entry.at(0) != '#'))  	// Line contains vendor information
	{
	  sscanf(buffer, "%04x", &vid);
	  
	  if (vid == _vendorid)					// Vendor matches
	  {
            _vendorname = entry.right(entry.length()-4);
	    _vendorname = _vendorname.stripWhiteSpace();
	    
	    while (fgets(buffer, sizeof(buffer), fhd))
            {
	      entry = QString(buffer);
              
	      if (entry.length() > 4)			      	// Line contains data
              {
 
                if (entry.at(0) == '\t')			// Line contains device information
                {
                  entry = entry.simplifyWhiteSpace();
	          entry = entry.stripWhiteSpace();
		  
		  sscanf(entry.data(), "%04x", &did);
		  
		  if (did == _deviceid)				// Device matches
		  {
                    _devicename = entry.right(entry.length()-4);
	            _devicename = _devicename.stripWhiteSpace();
		  
		    break;
		  }
		} 
		else 
		{
		  if ((entry.at(0) != ' ') && (entry.at(0) != '#')) 
		  {
		    _devicename = i18n("Unknown device");
		    break;
		  }
		}		 
	      }	      
	    }
	    
	    break;
	  }
	}
      }
    }
  
    fclose(fhd);
  } else perror("fopen");
}


