# Copyright 2010-2014 Meik Michalke <meik.michalke@hhu.de>
#
# This file is part of the R package rkwarddev.
#
# rkwarddev is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# rkwarddev is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with rkwarddev.  If not, see <http://www.gnu.org/licenses/>.


#' Create RKWard help file skeleton
#'
#' @param summary An object of class \code{XiMpLe.node} to be pasted as the \code{<summary>} section. See
#'    \code{\link[rkwarddev:rk.rkh.summary]{rk.rkh.summary}} for details.
#' @param usage An object of class \code{XiMpLe.node} to be pasted as the \code{<usage>} section. See
#'    \code{\link[rkwarddev:rk.rkh.usage]{rk.rkh.usage}} for details.
#' @param sections A (list of) objects of class \code{XiMpLe.node} to be pasted as \code{<section>} sections. See
#'    \code{\link[rkwarddev:rk.rkh.section]{rk.rkh.section}} for details.
#' @param settings An object of class \code{XiMpLe.node} to be pasted as the \code{<settings>} section. See
#'    \code{\link[rkwarddev:rk.rkh.settings]{rk.rkh.settings}} for details.
#'    Refer to \code{\link{rk.rkh.scan}} for a function to create this from an existing plugin XML file.
#' @param related An object of class \code{XiMpLe.node} to be pasted as the \code{<related>} section. See
#'    \code{\link[rkwarddev:rk.rkh.related]{rk.rkh.related}} for details.
#' @param technical An object of class \code{XiMpLe.node} to be pasted as the \code{<technical>} section. See
#'    \code{\link[rkwarddev:rk.rkh.technical]{rk.rkh.technical}} for details.
#' @param title An object of class \code{XiMpLe.node} to be pasted as the \code{<title>} section. See
#'    \code{\link[rkwarddev:rk.rkh.title]{rk.rkh.title}} for details.
#' @param hints Logical, if \code{TRUE} and you leave out optional entries (like \code{technical=NULL}), empty dummy sections will be added.
#' @param gen.info Logical, if \code{TRUE} a comment note will be written into the document,
#'    that it was generated by \code{rkwarddev} and changes should be done to the script.
#'    You can also provide a character string naming the very rkwarddev script file that generates this help file,
#'    which will then also be added to the comment.
#' @return An object of class \code{XiMpLe.doc}.
#' @seealso
#'    \code{\link[rkwarddev:rk.rkh.summary]{rk.rkh.summary}},
#'    \code{\link[rkwarddev:rk.rkh.usage]{rk.rkh.usage}},
#'    \code{\link[rkwarddev:rk.rkh.settings]{rk.rkh.settings}},
#'    \code{\link{rk.rkh.scan}},
#'    \code{\link[rkwarddev:rk.rkh.related]{rk.rkh.related}},
#'    \code{\link[rkwarddev:rk.rkh.technical]{rk.rkh.technical}}
#'    and the \href{help:/rkwardplugins/index.html}{Introduction to Writing Plugins for RKWard}
#' @export

rk.rkh.doc <- function(summary=NULL, usage=NULL, sections=NULL, settings=NULL, related=NULL, technical=NULL, title=NULL, hints=TRUE, gen.info=TRUE){

  all.children <- list()

  if(isTRUE(gen.info)){
    all.children[[length(all.children)+1]] <- generator.info()
  } else if(is.character(gen.info)){
    all.children[[length(all.children)+1]] <- generator.info(script=gen.info)
  } else {}

  if(!is.null(title)){
    # check if this is *really* a title section
    valid.parent(parent="title", node=title, warn=FALSE, see="rk.rkh.title")
    all.children[[length(all.children)+1]] <- title
  } else {}

  if(!is.null(summary)){
    # check if this is *really* a summary section
    valid.parent(parent="summary", node=summary, warn=FALSE, see="rk.rkh.summary")
    all.children[[length(all.children)+1]] <- summary
  } else if(isTRUE(hints)){
    all.children[[length(all.children)+1]] <- rk.rkh.summary()
  } else {}

  if(!is.null(usage)){
    # check if this is *really* a usage section
    valid.parent(parent="usage", node=usage, warn=FALSE, see="rk.rkh.usage")
    all.children[[length(all.children)+1]] <- usage
  } else if(isTRUE(hints)){
    all.children[[length(all.children)+1]] <- rk.rkh.usage()
  } else {}

  if(!is.null(sections)){
    for(this.section in sections){
      # check if this is *really* a section
      valid.parent(parent="section", node=this.section, warn=FALSE, see="rk.rkh.section")
      all.children[[length(all.children)+1]] <- this.section
    }
  } else if(isTRUE(hints)){
    all.children[[length(all.children)+1]] <- XMLNode("!--",
        rk.rkh.section("EDIT OR DELETE ME", text="EDIT OR DELETE ME"))
  } else {}

  if(!is.null(settings)){
    # check if this is *really* a settings section
    valid.parent(parent="settings", node=settings, warn=FALSE, see="rk.rkh.settings")
    all.children[[length(all.children)+1]] <- settings
  } else if(isTRUE(hints)){
    all.children[[length(all.children)+1]] <- rk.rkh.settings()
  } else {}

  if(!is.null(related)){
    # check if this is *really* a related section
    valid.parent(parent="related", node=related, warn=FALSE, see="rk.rkh.related")
    all.children[[length(all.children)+1]] <- related
  } else if(isTRUE(hints)){
    all.children[[length(all.children)+1]] <- XMLNode("!--", rk.rkh.related(rk.rkh.link("...")))
  } else {}

  if(!is.null(technical)){
    # check if this is *really* a technical section
    valid.parent(parent="technical", node=technical, warn=FALSE, see="rk.rkh.technical")
    all.children[[length(all.children)+1]] <- technical
  } else if(isTRUE(hints)){
    all.children[[length(all.children)+1]] <- rk.rkh.technical()
  } else {}

  rkh.document <- XMLNode("document", .children=child.list(all.children, empty=FALSE))

  rkh.main <- XMLTree(rkh.document, dtd=list(doctype="rkhelp"))

  return(rkh.main)
}
