/*
 * @(#)xcubes.c
 *
 * Copyright 1993 - 2008  David A. Bagley, bagleyd@tux.org
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Driver file for Cubes */

#include "file.h"
#ifdef WINVER
#include "CubesP.h"
#define TITLE "wcubes"

static CubesRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->cubes.started = b
#else
#include "xwin.h"
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#ifdef MOUSEBITMAPS
#include "pixmaps/mouse-l.xbm"
#include "pixmaps/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) printState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Cubes.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "pixmaps/cubes.t.xpm"
#include "pixmaps/cubes.p.xpm"
#include "pixmaps/cubes.s.xpm"
#include "pixmaps/cubes.m.xpm"
#include "pixmaps/cubes.l.xpm"
#include "pixmaps/cubes.xpm"
#define RESIZE_XPM(s) ((char **) ((s)<=32)?\
(((s)<=22)?(((s)<=16)?cubes_t_xpm:cubes_p_xpm):\
(((s)<=24)?cubes_s_xpm:cubes_m_xpm)):\
(((s)<=48)?cubes_l_xpm:cubes_xpm))
#endif
#include "pixmaps/cubes.xbm"
#define DEFINE_XBM (char *) cubes_bits, cubes_width, cubes_height
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef WINVER
static const char aboutHelp[] = {
"Cubes Version 7.4\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd@tux.org>\n"
"The latest version is at: "
"http://www.tux.org/~bagleyd/puzzles.html\n"
};

static const char optionsHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-block {color}] [-text {color}] [-[no]install]\n"
"[-picture {filename}] [-delay msecs] [-[no]sound]\n"
"[-bumpSound {filename}] [-dripSound {filename}]\n"
"[-{font|fn} {fontname}] [-size{x|y|z} {int}] [-base {int}]\n"
"[-userName {string}] [-scoreFile {filename}] [-scores]\n"
"[-version]\n"
};
#endif

#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
"A 3D  sliding block puzzle.  The cubes represented are  "
"flattened out on the plane.  If the Blocks Z slider is\n"
"greater than 1, then each group of boxes is a layer of the "
"puzzle.  If the Blocks Z slider = 1, then this is a\n"
"simple 15 tile puzzle.  The 15 puzzle was originally "
"made around 1879 (the 14-15  puzzle of Sam Loyd  was made\n"
"soon after.)  Similar 3D puzzles include  BloxBox; "
"Qrazy Qube (2x2x2) by P. Hein of Denmark and Varikon Box 'L\n"
"Box 'L (3x3x3) (Hungary) and Mad Marbles (3x3x3) by "
"Toys & Games International USA.\n"
};

static const char featuresHelp[] = {
"Click \"mouse-left\" button to move a block.  Clicks on "
"a space, or clicks on blocks  that are not in line\n"
"with a space, will not move the blocks.  Notice, one can move "
"more than  one block at a time by clicking\n"
"on the block farthest from the space but still in line "
"with the space.  Since you only have a 2-D screen\n"
"the next level the down is  either the box to the right of "
"the current box  or the bottom of the current\n"
"box.\n"
"\n"
"Click \"mouse-right\" button, or press \"C\" or \"c\" "
"keys, to clear the puzzle.\n"
"\n"
"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"Z\" or \"z\" keys to randomize the puzzle.  New records "
"can only be set from a cleared state.\n"
"\n"
"Press \"S\" or \"s\" keys to start the auto-solver.  Only "
"works on lxmx1 where l > 3 and m > 3.\n"
"\n"
"Press \"I\" or \"i\" keys to move a block in.\n"
"\n"
"Press \"O\" or \"o\" keys to move a block out.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of blocks.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of blocks.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Cubes as:\n"
"  /    Out\n"
"  8    Up\n"
"  ^\n"
"4<5>6  Left, In, Right\n"
"  v\n"
"  2    Down\n"
"\n"
"The control key allows you to change the complexity of "
"xcubes.  Hold down control key and use the keypad\n"
"or arrow keys to reduce or enlarge puzzle complexity.\n"
};

static const char referencesHelp[] = {
"L. E. Horden, Sliding Piece Puzzles  (Recreations in "
"Mathematics Series), Oxford University Press  1986,\n"
"pp 1, 157-159.\n"
"Jerry Slocum & Jack Botermans, Puzzles Old & New (How to Make "
"and Solve Them),  University of Washington\n"
"Press, Seattle 1987, p 126, 127.\n"
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, only implemented for lxmx1 where "
"l > 3 and m > 3.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "cubes.scores"
#endif

#define MAX_CUBES 8
#define NEVER (-1)
#define FILE_NAME_LENGTH 1024
#define USER_NAME_LENGTH 120
#define MESSAGE_LENGTH (USER_NAME_LENGTH+64)
#define TITLE_LENGTH 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USER_NAME_LENGTH];
} PuzzleRecord;

static PuzzleRecord puzzleRecord[MAX_CUBES - MIN_CUBES + 1]
[MAX_CUBES - MIN_CUBES + 1][MAX_CUBES - MIN_CUBES + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGE_LENGTH] = "Welcome";
static char recordDsp[MESSAGE_LENGTH] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MIN_SPEED 1
#define MAX_SPEED 50
static Widget movesText, recordText, message;
static Widget blockXSlider, blockYSlider, blockZSlider, speedSlider;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, clearDialog;
static Arg arg[3];
#else
static char titleDsp[TITLE_LENGTH] = "";
#endif
static char scoreFileName[FILE_NAME_LENGTH] = SCOREFILE;
static char fileName[FILE_NAME_LENGTH];
static Boolean randomized = False;
#ifdef WINVER
#define PROGRAM_NAME_LENGTH 80
static char progDsp[PROGRAM_NAME_LENGTH] = TITLE;
static char userNameDsp[USER_NAME_LENGTH] = "Guest";
#else
static Pixmap pixmap = None;
static Widget topLevel, puzzle;
static char *progDsp;
static char userNameDsp[USER_NAME_LENGTH] = "";

#ifdef HAVE_MOTIF
static void
printState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("printState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w, XmNlabelString, xmstr, NULL);
}
#endif

static void
printRecords(void)
{
	int i, j, k;

	(void) printf("          CUBES  HALL OF FAME\n\n");
	(void) printf("   Z   Y   X USER             MOVES\n");
	for (i = 0; i < MAX_CUBES - MIN_CUBES + 1; i++)
		for (j = i; j < MAX_CUBES - MIN_CUBES + 1; j++)
			for (k = j; k < MAX_CUBES - MIN_CUBES + 1; k++) {
				if (puzzleRecord[k][j][i].score > 0)
					(void) printf("%4d%4d%4d %-12s%9d\n",
						i + 1, j + 1, k + 1,
						puzzleRecord[k][j][i].name,
						puzzleRecord[k][j][i].score);
			}
}
#endif

static void
initRecords(void)
{
	int i, j, k;

	for (i = 0; i < MAX_CUBES - MIN_CUBES + 1; i++)
		for (j = i; j < MAX_CUBES - MIN_CUBES + 1; j++)
			for (k = j; k < MAX_CUBES - MIN_CUBES + 1; k++) {
				puzzleRecord[k][j][i].score = puzzleRecord[k][i][j].score =
					puzzleRecord[j][k][i].score = puzzleRecord[j][i][k].score =
					puzzleRecord[i][k][j].score = puzzleRecord[i][j][k].score = NEVER;
				(void) strncpy(puzzleRecord[k][j][i].name,
					NOACCESS, USER_NAME_LENGTH);
				(void) strncpy(puzzleRecord[k][i][j].name,
					NOACCESS, USER_NAME_LENGTH);
				(void) strncpy(puzzleRecord[j][k][i].name,
					NOACCESS, USER_NAME_LENGTH);
				(void) strncpy(puzzleRecord[j][i][k].name,
					NOACCESS, USER_NAME_LENGTH);
				(void) strncpy(puzzleRecord[i][k][j].name,
					NOACCESS, USER_NAME_LENGTH);
				(void) strncpy(puzzleRecord[i][j][k].name,
					NOACCESS, USER_NAME_LENGTH);
			}
}

static void
readRecords(void)
{
	FILE *fp;
	int i, j, k, n;
	char userName[USER_NAME_LENGTH];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USER_NAME_LENGTH);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USER_NAME_LENGTH);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (i = 0; i < MAX_CUBES - MIN_CUBES + 1; i++)
		for (j = i; j < MAX_CUBES - MIN_CUBES + 1; j++)
			for (k = j; k < MAX_CUBES - MIN_CUBES + 1; k++) {
				(void) fscanf(fp, "%d %s", &n, userName);
				if (n <= puzzleRecord[i][j][k].score ||
						puzzleRecord[i][j][k].score <= NEVER) {
					puzzleRecord[k][j][i].score = puzzleRecord[k][i][j].score =
						puzzleRecord[j][k][i].score = puzzleRecord[j][i][k].score =
						puzzleRecord[i][k][j].score = puzzleRecord[i][j][k].score = n;
					(void) strncpy(puzzleRecord[k][j][i].name,
						userName, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[k][i][j].name,
						userName, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[j][k][i].name,
						userName, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[j][i][k].name,
						userName, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[i][k][j].name,
						userName, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[i][j][k].name,
						userName, USER_NAME_LENGTH);
				}
			}
	(void) fclose(fp);
}

static void
writeRecords(void)
{
	FILE *fp;
	int i, j, k;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Can not write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILE_NAME_LENGTH];

		(void) strncpy(lockFile, fileName, FILE_NAME_LENGTH - 6);
		(void) strcat(lockFile, ".lock");
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (i = 0; i < MAX_CUBES - MIN_CUBES + 1; i++) {
			for (j = i; j < MAX_CUBES - MIN_CUBES + 1; j++) {
				for (k = j; k < MAX_CUBES - MIN_CUBES + 1; k++)
					(void) fprintf(fp, "%d %s\n",
						puzzleRecord[i][j][k].score,
						puzzleRecord[i][j][k].name);
				(void) fprintf(fp, "\n");
			}
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
printRecord(int sizeX, int sizeY, int sizeZ)
{
	int i = sizeX - MIN_CUBES, j = sizeY - MIN_CUBES, k = sizeZ - MIN_CUBES;

	if (sizeX > MAX_CUBES || sizeY > MAX_CUBES || sizeZ > MAX_CUBES) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGE_LENGTH);
	} else if (puzzleRecord[i][j][k].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			puzzleRecord[i][j][k].score, puzzleRecord[i][j][k].name);
	}
#ifdef HAVE_MOTIF
	printState(recordText, recordDsp);
#endif
}

static void
printStatus(char *msg, int nMoves
#ifndef HAVE_MOTIF
		, int sizeX, int sizeY, int sizeZ
#endif
		)
{
#ifdef HAVE_MOTIF
	printState(message, msg);
	(void) sprintf(buff, "%d", nMoves);
	printState(movesText, buff);
#else
	(void) sprintf(titleDsp, "%s: %dx%dx%d@ (%d/%s) - %s",
		progDsp, sizeX, sizeY, sizeZ, nMoves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent(puzzle), XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
handleSolved(int counter, int sizeX, int sizeY, int sizeZ)
{
	int i = sizeX - MIN_CUBES, j = sizeY - MIN_CUBES, k = sizeZ - MIN_CUBES;

	if (sizeX <= MAX_CUBES && sizeY <= MAX_CUBES && sizeZ <= MAX_CUBES &&
			(counter < puzzleRecord[i][j][k].score ||
			puzzleRecord[i][j][k].score <= NEVER)) {
		readRecords();	/* Maybe its been updated by another */

		puzzleRecord[i][j][k].score = puzzleRecord[i][k][j].score =
			puzzleRecord[j][i][k].score =
			puzzleRecord[j][k][i].score =
			puzzleRecord[k][i][j].score =
			puzzleRecord[k][j][i].score =
			counter;
		(void) strncpy(puzzleRecord[i][j][k].name, userNameDsp,
			USER_NAME_LENGTH);
		(void) strncpy(puzzleRecord[i][k][j].name, userNameDsp,
			USER_NAME_LENGTH);
		(void) strncpy(puzzleRecord[j][i][k].name, userNameDsp,
			USER_NAME_LENGTH);
		(void) strncpy(puzzleRecord[j][k][i].name, userNameDsp,
			USER_NAME_LENGTH);
		(void) strncpy(puzzleRecord[k][i][j].name, userNameDsp,
			USER_NAME_LENGTH);
		(void) strncpy(puzzleRecord[k][j][i].name, userNameDsp,
			USER_NAME_LENGTH);
		writeRecords();
		printRecord(sizeX, sizeY, sizeZ);
		return True;
	}
	return False;
}

static void
initialize(
#ifdef WINVER
CubesWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int sizeX, sizeY, sizeZ;
	char *userName, *scoreFile;

#ifdef WINVER
	initializePuzzle(w, brush);

	sizeX = w->cubes.sizeX;
	sizeY = w->cubes.sizeY;
	sizeZ = w->cubes.sizeZ;
	userName = w->cubes.userName;
	scoreFile = w->cubes.scoreFile;
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILE_NAME_LENGTH);
#else
	Boolean scoreOnly, versionOnly;

	XtVaGetValues(w,
		XtNuserName, &userName,
		XtNscoreFile, &scoreFile,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ,
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf(aboutHelp);
		exit(0);
	}
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILE_NAME_LENGTH);
	if (scoreOnly) {
		initRecords();
		readRecords();
		printRecords();
		exit(0);
	}
#ifdef HAVE_MOTIF
	if (sizeX > MAX_CUBES)
		XtVaSetValues(blockXSlider, XmNmaximum, sizeX, NULL);
	XmScaleSetValue(blockXSlider, sizeX);
	if (sizeY > MAX_CUBES)
		XtVaSetValues(blockYSlider, XmNmaximum, sizeY, NULL);
	XmScaleSetValue(blockYSlider, sizeY);
	if (sizeZ > MAX_CUBES)
		XtVaSetValues(blockZSlider, XmNmaximum, sizeZ, NULL);
	XmScaleSetValue(blockZSlider, sizeZ);
	{
		int delay;

		XtVaGetValues(w, XtNdelay, &delay, NULL);
		XmScaleSetValue(speedSlider,
			MAX_SPEED + MIN_SPEED - delay - 1);
	}
#endif
#endif
	SET_STARTED(w, False);
	initRecords();
	readRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH);
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH);
#else
		char *login = getlogin();

		if (login == NULL) {
			(void) strcpy(userNameDsp, "");
		} else {
			(void) sprintf(userNameDsp, "%s", login);
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(userNameDsp, "%s", "guest");
#endif
	}
	printRecord(sizeX, sizeY, sizeZ);
	printStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, sizeX, sizeY, sizeZ
#endif
		);
}

#ifdef WINVER
void
setPuzzle(CubesWidget w, int reason)
#else
static void
puzzleListener(Widget w, caddr_t clientData, cubesCallbackStruct *callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int sizeX, sizeY, sizeZ;
	Boolean cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	sizeX = w->cubes.sizeX;
	sizeY = w->cubes.sizeY;
	sizeZ = w->cubes.sizeZ;
	cheat = w->cubes.cheat;
#else
	XtVaGetValues(w,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
	case ACTION_HIDE:
#ifdef WINVER
		ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
		break;
#ifndef WINVER
	case ACTION_CLEAR_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(clearDialog);
#else
		XtVaSetValues(puzzle, XtNmenu, ACTION_CLEAR, NULL);
#endif
		break;
#endif
	case ACTION_SOLVE_MESSAGE:
#ifdef WINVER
		(void) MessageBox(w->core.hWnd, solveHelp,
			"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
		XtManageChild(solveDialog);
#else
		(void) strncpy(messageDsp, solveHelp, MESSAGE_LENGTH);
#endif
#endif
			break;
	case ACTION_RESTORE:
	case ACTION_RESET:
		movesDsp = 0;
		randomized = False;
		break;
	case ACTION_BLOCKED:
		(void) strncpy(messageDsp, "Blocked", MESSAGE_LENGTH);
		break;
	case ACTION_SPACE:
#if 0
		/* Too annoying */
		(void) strncpy(messageDsp, "A space can not slide",
			MESSAGE_LENGTH);
#endif
		break;
	case ACTION_IGNORE:
		(void) strncpy(messageDsp, "Clear to start",
			MESSAGE_LENGTH);
		break;
	case ACTION_MOVED:
		movesDsp++;
		SET_STARTED(w, True);
		break;
	case ACTION_SOLVED:
		if (cheat)
			(void) sprintf(messageDsp,
				"No cheating %s!!", userNameDsp);
		else if (!randomized && handleSolved(movesDsp,
				sizeX, sizeY, sizeZ))
			(void) sprintf(messageDsp,
				"Congratulations %s!!", userNameDsp);
		else
			(void) strncpy(messageDsp, "Solved!",
				MESSAGE_LENGTH);
		SET_STARTED(w, False);
		break;
	case ACTION_COMPUTED:
		SET_STARTED(w, False);
		break;
	case ACTION_RANDOMIZE:
		movesDsp = 0;
		randomized = True;
		SET_STARTED(w, False);
		break;
	case ACTION_INCX:
		movesDsp = 0;
		sizeX++;
		printRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
		w->cubes.sizeX = sizeX;
#else
		XtVaSetValues(w, XtNsizeX, sizeX, NULL);
#ifdef HAVE_MOTIF
		if (sizeX > MAX_CUBES)
			XtVaSetValues(blockXSlider, XmNmaximum, sizeX, NULL);
		XmScaleSetValue(blockXSlider, sizeX);
#endif
#endif
		break;
	case ACTION_DECX:
		movesDsp = 0;
		sizeX--;
		printRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
		w->cubes.sizeX = sizeX;
#else
		XtVaSetValues(w, XtNsizeX, sizeX, NULL);
#ifdef HAVE_MOTIF
		XmScaleSetValue(blockXSlider, sizeX);
		if (sizeX >= MAX_CUBES)
			XtVaSetValues(blockXSlider, XmNmaximum, sizeX, NULL);
#endif
#endif
		break;
	case ACTION_INCY:
		movesDsp = 0;
		sizeY++;
		printRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
		w->cubes.sizeY = sizeY;
#else
		XtVaSetValues(w, XtNsizeY, sizeY, NULL);
#ifdef HAVE_MOTIF
		if (sizeY > MAX_CUBES)
			XtVaSetValues(blockYSlider, XmNmaximum, sizeY, NULL);
		XmScaleSetValue(blockYSlider, sizeY);
#endif
#endif
		break;
	case ACTION_DECY:
		movesDsp = 0;
		sizeY--;
		printRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
		w->cubes.sizeY = sizeY;
#else
		XtVaSetValues(w, XtNsizeY, sizeY, NULL);
#ifdef HAVE_MOTIF
		XmScaleSetValue(blockYSlider, sizeY);
		if (sizeY >= MAX_CUBES)
			XtVaSetValues(blockYSlider, XmNmaximum, sizeY, NULL);
#endif
#endif
		break;
	case ACTION_INCZ:
		movesDsp = 0;
		sizeZ++;
		printRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
		w->cubes.sizeZ = sizeZ;
#else
		XtVaSetValues(w, XtNsizeZ, sizeZ, NULL);
#ifdef HAVE_MOTIF
		if (sizeZ > MAX_CUBES)
			XtVaSetValues(blockZSlider, XmNmaximum, sizeZ, NULL);
		XmScaleSetValue(blockZSlider, sizeZ);
#endif
#endif
		break;
	case ACTION_DECZ:
		movesDsp = 0;
		sizeZ--;
		printRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
		w->cubes.sizeZ = sizeZ;
#else
		XtVaSetValues(w, XtNsizeZ, sizeZ, NULL);
#ifdef HAVE_MOTIF
		XmScaleSetValue(blockZSlider, sizeZ);
		if (sizeZ >= MAX_CUBES)
			XtVaSetValues(blockZSlider, XmNmaximum, sizeZ, NULL);
#endif
#endif
		break;
	case ACTION_UNDO:
		movesDsp--;
		SET_STARTED(w, True);
		break;
	case ACTION_REDO:
		movesDsp++;
		SET_STARTED(w, True);
		break;
#ifdef HAVE_MOTIF
	case ACTION_SPEED:
		{
			int oldDelay, delay;

			XtVaGetValues(puzzle,
				XtNdelay, &delay, NULL);
			oldDelay = delay;
			if (delay > MAX_SPEED - MIN_SPEED)
				delay = MAX_SPEED - MIN_SPEED;
			if (delay < 0)
				delay = 0;
			if (delay != oldDelay) {
				XtVaSetValues(w,
					XtNdelay, delay, NULL);
			}
			XmScaleSetValue(speedSlider,
				MAX_SPEED + MIN_SPEED - delay - 1);
		}
		return;
#endif
	}
	printStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, sizeX, sizeY, sizeZ
#endif
		);
}

#ifdef WINVER
static LRESULT CALLBACK
about(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND && LOWORD(wParam) == IDOK) {
		(void) EndDialog(hDlg, TRUE);
		return TRUE;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.cubes.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			enterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.cubes.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			leavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
	case WM_CREATE:
		initialize(&widget, brush);
		break;
	case WM_DESTROY:
		destroyPuzzle(&widget, brush);
		break;
	case WM_SIZE:
		resizePuzzle(&widget);
		(void) InvalidateRect(hWnd, NULL, TRUE);
		break;
	case WM_PAINT:
		widget.core.hDC = BeginPaint(hWnd, &paint);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		exposePuzzle(&widget);
		(void) EndPaint(hWnd, &paint);
		break;
	case WM_RBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		clearPuzzle(&widget);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		selectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam));
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONUP:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		releasePuzzle(&widget);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
	case WM_MOUSEWHEEL:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		{
			int zDelta = ((short) HIWORD(wParam));
			POINT cursor, origin;

			origin.x = 0, origin.y = 0;
			ClientToScreen(hWnd, &origin);
			(void) GetCursorPos(&cursor);
			if (zDelta > (WHEEL_DELTA >> 1)) {
				movePuzzle(&widget, TOP,
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
				if (GetKeyState(VK_CONTROL) >> 1) {
					sizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
				}
			} else if (zDelta < -(WHEEL_DELTA >> 1)) {
				movePuzzle(&widget, BOTTOM,
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
				if (GetKeyState(VK_CONTROL) >> 1) {
					sizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
				}
			}
		}
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#endif
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ACTION_GET:
			getPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_WRITE:
			writePuzzle(&widget);
			break;
		case ACTION_EXIT:
			destroyPuzzle(&widget, brush);
			break;
		case ACTION_HIDE:
			hidePuzzle(&widget);
			break;
		case ACTION_UNDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			undoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_REDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			redoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_CLEAR:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			clearPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_RANDOMIZE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			randomizePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_SOLVE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			solvePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_UP:
		case ACTION_RIGHT:
		case ACTION_DOWN:
		case ACTION_LEFT:
		case ACTION_IN:
		case ACTION_OUT:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			(void) movePuzzle(&widget,
				(int) LOWORD(wParam) - ACTION_UP, FALSE);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_INCX:
		case ACTION_DECX:
		case ACTION_INCY:
		case ACTION_DECY:
		case ACTION_INCZ:
		case ACTION_DECZ:
			if (movePuzzle(&widget,
					(int) LOWORD(wParam) - ACTION_DECY,
					TRUE)) {
				sizePuzzle(&widget);
				(void) InvalidateRect(hWnd, NULL, TRUE);
			}
			break;
		case ACTION_SPEED:
			speedUpPuzzle(&widget);
			break;
		case ACTION_SLOW:
			slowDownPuzzle(&widget);
			break;
		case ACTION_SOUND:
			toggleSoundPuzzle(&widget);
			break;
		case ACTION_DESCRIPTION:
			(void) MessageBox(hWnd, descriptionHelp,
				"Description", MB_OK);
			break;
		case ACTION_FEATURES:
			(void) MessageBox(hWnd, featuresHelp,
				"Features", MB_OK);
			break;
		case ACTION_REFERENCES:
			(void) MessageBox(hWnd, referencesHelp,
				"References", MB_OK);
			break;
		case ACTION_ABOUT:
			(void) DialogBox(widget.core.hInstance,
				"About", hWnd, (DLGPROC) about);
			break;
		}
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#if 0
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#else
		512 + 10 + 4, 512 + 10 + 4 + 34,
#endif
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
usage(char *programName)
{
	unsigned int i;

	(void) fprintf(stderr, "usage: %s\n", programName);
	for (i = 0; i < strlen(optionsHelp); i++) {
		if (i == 0 || optionsHelp[i - 1] == '\n')
			(void) fprintf(stderr, "\t");
		(void) fprintf(stderr, "%c", (optionsHelp[i]));
	}
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*cubes.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*cubes.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*cubes.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*cubes.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*cubes.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*cubes.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*cubes.foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*cubes.foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-block", (char *) "*cubes.blockColor", XrmoptionSepArg, NULL},
	{(char *) "-text", (char *) "*cubes.textColor", XrmoptionSepArg, NULL},
	{(char *) "-install", (char *) "*cubes.install", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noinstall", (char *) "*cubes.install", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-picture", (char *) "*cubes.picture", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*cubes.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*cubes.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*cubes.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bumpSound", (char *) "*cubes.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-dripSound", (char *) "*cubes.dripSound", XrmoptionSepArg, NULL},
	{(char *) "-base", (char *) "*cubes.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*cubes.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*cubes.font", XrmoptionSepArg, NULL},
	{(char *) "-sizex", (char *) "*cubes.sizeX", XrmoptionSepArg, NULL},
	{(char *) "-sizey", (char *) "*cubes.sizeY", XrmoptionSepArg, NULL},
	{(char *) "-sizez", (char *) "*cubes.sizeZ", XrmoptionSepArg, NULL},
	{(char *) "-userName", (char *) "*cubes.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*cubes.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*cubes.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*cubes.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
puzzleListenerClear(Widget w, XtPointer clientData, XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(puzzle, XtNmenu, ACTION_CLEAR, NULL);
	}
}

static void
blockXChangeListener(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int sizeX = cbs->value, sizeY, sizeZ, oldX;

	XtVaGetValues(puzzle,
		XtNsizeX, &oldX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ, NULL);
	if (oldX == sizeX)
		return;
	XtVaSetValues(puzzle,
		XtNsizeX, sizeX, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, sizeZ);
	(void) strcpy(messageDsp, "");
	printState(message, messageDsp);
}

static void
blockYChangeListener(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int sizeX, sizeY = cbs->value, sizeZ, oldY;

	XtVaGetValues(puzzle,
		XtNsizeX, &sizeX,
		XtNsizeY, &oldY,
		XtNsizeZ, &sizeZ, NULL);
	if (oldY == sizeY)
		return;
	XtVaSetValues(puzzle,
		XtNsizeY, sizeY, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, sizeZ);
	(void) strcpy(messageDsp, "");
	printState(message, messageDsp);
}

static void
blockZChangeListener(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int sizeX, sizeY, sizeZ = cbs->value, oldZ;

	XtVaGetValues(puzzle,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &oldZ, NULL);
	if (oldZ == sizeZ)
		return;
	XtVaSetValues(puzzle,
		XtNsizeZ, sizeZ, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	printState(movesText, buff);
	printRecord(sizeX, sizeY, sizeZ);
	(void) strcpy(messageDsp, "");
	printState(message, messageDsp);
}

static void
speedChangeListener(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int delay = MAX_SPEED + MIN_SPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(puzzle,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(puzzle, XtNdelay, delay, NULL);
	}
}

static void
fileMenuListener(Widget w, void *value, void *clientData)
{
	int val = (int) value + ACTION_GET;

	if (val == ACTION_EXIT)
		exit(0);
	XtVaSetValues(puzzle, XtNmenu, val, NULL);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value + ACTION_UNDO;

	XtVaSetValues(puzzle, XtNmenu, val, NULL);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILE_NAME_LENGTH + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILE_NAME_LENGTH + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}

static void
helpMenuListener(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpMenuListener: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
	int pixmapSize;
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, blocksRowCol, speedRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString;
	XmString clearString, randomizeString, solveString;
	XmString speedString, slowString, soundString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Cubes",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		usage(argv[0]);

#ifdef HAVE_MOTIF
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu",
		0, fileMenuListener,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
	randomizeString = XmStringCreateSimple((char *) "RandomiZe");
	solveString = XmStringCreateSimple((char *) "Auto-Solve");
	speedString = XmStringCreateSimple((char *) ">Speed");
	slowString = XmStringCreateSimple((char *) "<Slow");
	soundString = XmStringCreateSimple((char *) "@Sound");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu",
		1, playCB,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'Z', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	clearDialog = createQuery(topLevel,
		(char *) "Are you sure you want to start over?",
		(char *) "Clear Query",
		(XtCallbackProc) puzzleListenerClear);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move block", 11, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Clear", 6, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	movesText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	recordText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	blocksRowCol = XtVaCreateManagedWidget("blocksRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	blockXSlider = XtVaCreateManagedWidget("blockXSlider",
		xmScaleWidgetClass, blocksRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Blocks X", 9,
		XmNminimum, MIN_CUBES,
		XmNmaximum, MAX_CUBES,
		XmNvalue, DEFAULT_CUBESX,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(blockXSlider, XmNvalueChangedCallback,
		(XtCallbackProc) blockXChangeListener, (XtPointer) NULL);
	blockYSlider = XtVaCreateManagedWidget("blockYSlider",
		xmScaleWidgetClass, blocksRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Blocks Y", 9,
		XmNminimum, MIN_CUBES,
		XmNmaximum, MAX_CUBES,
		XmNvalue, DEFAULT_CUBESY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(blockYSlider, XmNvalueChangedCallback,
		(XtCallbackProc) blockYChangeListener, (XtPointer) NULL);
	blockZSlider = XtVaCreateManagedWidget("blockZSlider",
		xmScaleWidgetClass, blocksRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Blocks Z", 9,
		XmNminimum, MIN_CUBES,
		XmNmaximum, MAX_CUBES,
		XmNvalue, DEFAULT_CUBESZ,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(blockZSlider, XmNvalueChangedCallback,
		(XtCallbackProc) blockZChangeListener, (XtPointer) NULL);
	speedRowCol = XtVaCreateManagedWidget("speedRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	speedSlider = XtVaCreateManagedWidget("speedSlider",
		xmScaleWidgetClass, speedRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MIN_SPEED,
		XmNmaximum, MAX_SPEED,
		XmNvalue, MAX_SPEED - DEFAULT_DELAY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speedSlider, XmNvalueChangedCallback,
		(XtCallbackProc) speedChangeListener, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Cubes! (use mouse or keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	puzzle = XtCreateManagedWidget("cubes",
		cubesWidgetClass, mainPanel, NULL, 0);
#else
	puzzle = XtCreateManagedWidget("cubes",
		cubesWidgetClass, topLevel, NULL, 0);
#endif
	XtVaGetValues(puzzle,
		XtNpixmapSize, &pixmapSize, NULL);
#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			RESIZE_XPM(pixmapSize), &pixmap, NULL,
			&xpmAttributes);
	}
	if (pixmap == (Pixmap) NULL)
#endif
		pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			DEFINE_XBM);
	XtVaSetValues(topLevel,
#ifdef HAVE_MOTIF
		XmNkeyboardFocusPolicy, XmPOINTER, /* not XmEXPLICIT */
#else
		XtNinput, True,
#endif
		XtNiconPixmap, pixmap, NULL);
	XtAddCallback(puzzle, XtNselectCallback,
		(XtCallbackProc) puzzleListener, (XtPointer) NULL);
	initialize(puzzle);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(puzzle), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle),
		XCreateFontCursor(XtDisplay(puzzle), XC_hand2));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
