/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef ISSUECONDITIONSETTINGS_H
#define ISSUECONDITIONSETTINGS_H

#include <QObject>
#include <QList>
#include <QStringList>

class QSettings;

namespace WebIssues
{

class ColumnCondition;

/**
* Class for loading and modifying filter settings.
*/
class ColumnConditionSettings : public QObject
{
    Q_OBJECT
public:
    /**
    * Default constructor.
    */
    ColumnConditionSettings();

    /**
    * Destructor.
    */
    ~ColumnConditionSettings();

public:
    /**
    * Read filter settings for given type of issue.
    *
    * If no settings exist for the given type, some default filters are created.
    */
    void openIssueFilters( int typeId );

    /**
    * Return the names of existing filters.
    */
    const QStringList& filterNames() const { return m_names; }

    /**
    * Check if the given filter exists.
    */
    bool filterExists( const QString& name ) const;

    /**
    * Load the conditions of the given filter.
    */
    QList<ColumnCondition> loadFilter( const QString& name );

    /**
    * Save the filter conditions.
    */
    void saveFilter( const QString& name, const QList<ColumnCondition>& conditions );

    /**
    * Rename an existing filter.
    */
    void renameFilter( const QString& oldName, const QString& newName );

    /**
    * Delete the given filter.
    */
    void deleteFilter( const QString& name );

    /**
    * Return the list of watches for the given folder.
    *
    * Empty string represents the "All Issues" watch.
    */
    QStringList loadWatches( int folderId );

    /**
    * Save the watches for the given folder.
    */
    void saveWatches( int folderId, const QStringList& watches );

private:
    void createDefaultIssueFilters();

    void deleteAt( QSettings* settings, int index );
    void copyConditions( QSettings* settings, int oldIndex, int newIndex );

    QStringList serializeConditions( const QList<ColumnCondition>& conditions );
    const QList<ColumnCondition> deserializeConditions( const QStringList& list );

    void deleteWatches( QSettings* settings, int filterIndex );

private:
    QString m_group;
    QString m_watchesGroup;

    QStringList m_names;
};

}

#endif
