# Author:: Nathaniel Talbott.
# Copyright:: Copyright (c) 2000-2002 Nathaniel Talbott. All rights reserved.
# License:: Ruby license.

require 'test/unit/testcase'

module Test
  module Unit
    class TC_TestCase
      attr_reader(:result)
  
      def initialize(test)
        @test = test
      end
      
      def self.simple_suite
        [TC_TestCase.new("test_creation"),
          TC_TestCase.new("test_add_failed_assertion"),
          TC_TestCase.new("test_add_error"),
          TC_TestCase.new("test_no_tests"),
          TC_TestCase.new("test_suite"),
          TC_TestCase.new("test_setup_teardown"),
          TC_TestCase.new("test_assertion_failed_not_called")]
      end
      
      def size
        1
      end
      
      def run(result)
        @result = result
        catch(:failure) { send(@test) }
        result.add_run
      end
      
      def add_successful_assertion(message)
        @my_result.add_successful_assertion(message)
      end
      
      def add_failed_assertion(message)
        @my_result.add_failed_assertion(message)
      end
      
      class TC_Creation < TestCase
        def test_with_arguments(arg1, arg2)
        end
      end
      
      def test_creation
        caught = true
        catch(:invalid_test) do
          TC_Creation.new(:test_with_arguments)
          caught = false
        end
        check("Should have caught an invalid test when there are arguments", caught)
        
        caught = true
        catch(:invalid_test) do
          TC_Creation.new(:non_existent_test)
          caught = false
        end
        check("Should have caught an invalid test when the method does not exist", caught)
      end
      
      class TC_FailureError < TestCase
        def test_failure
          assert_block("failure") { false }
        end
        def test_error
          1 / 0
        end
        def return_passed?
          return passed?
        end
      end
      
      def test_add_failed_assertion
        test_case = TC_FailureError.new(:test_failure)
        check("passed? should start out true", test_case.return_passed?)
        result = TestResult.new
        called = false
        result.add_listener(TestResult::FAULT) {
          | fault |
          check("Should have a Failure", fault.instance_of?(Failure))
          check("The Failure should have the correct message", "failure" == fault.message)
          check("The Failure should have the correct location (was <#{fault.location}>)", fault.location =~ /test_failure\(Test::Unit::TC_TestCase::TC_FailureError\) \[.*tc_testcase\.rb:\d+\]/)
          called = true
        }
        progress = []
        test_case.run(result) { |arguments| progress << arguments }
        check("The failure should have triggered the listener", called)
        check("The failure should have set passed?", !test_case.return_passed?)
        check("The progress block should have been updated correctly", [[TestCase::STARTED, test_case.name], [TestCase::FINISHED, test_case.name]] == progress)
      end
      
      def test_add_error
        test_case = TC_FailureError.new(:test_error)
        check("passed? should start out true", test_case.return_passed?)
        result = TestResult.new
        called = false
        result.add_listener(TestResult::FAULT) {
          | fault |
          check("Should have a TestError", fault.instance_of?(Error))
          check("The Error should have the correct message", "ZeroDivisionError: divided by 0" == fault.message)
          check("The Error should have the correct location", "test_error(Test::Unit::TC_TestCase::TC_FailureError)" == fault.location)
          check("The Error should have the correct exception", fault.exception.instance_of?(ZeroDivisionError))
          called = true
        }
        test_case.run(result) {}
        check("The error should have triggered the listener", called)
        check("The error should have set passed?", !test_case.return_passed?)
      end

      def test_no_tests      
        suite = TestCase.suite
        check("Should have a test suite", suite.instance_of?(TestSuite))
        check("Should have one test", suite.size == 1)
        check("Should have the default test", suite.tests.first.name == "default_test(Test::Unit::TestCase)")
        
        result = TestResult.new
        suite.run(result) {}
        check("Should have had one test run", result.run_count == 1)
        check("Should have had one test failure", result.failure_count == 1)
        check("Should have had no errors", result.error_count == 0)
      end

      class TC_Run < TestCase
        def test_succeed
          assert_block {true}
        end
        def test_fail
          assert_block {false}
        end
        def test_error
          1/0
        end
        def dont_run
          assert_block {true}
        end
        def test_dont_run(argument)
          assert_block {true}
        end
        def test
          assert_block {true}
        end
      end
      
      def test_suite
        suite = TC_Run.suite
        check("Should have a test suite", suite.instance_of?(TestSuite))
        check("Should have three tests", suite.size == 3)
  
        result = TestResult.new
        suite.run(result) {}
        check("Should have had three test runs", result.run_count == 3)
        check("Should have had one test failure", result.failure_count == 1)
        check("Should have had one test error", result.error_count == 1)
        @result.add_run
      end
      
      class TC_SetupTearDown < TestCase
        attr_reader(:setup_called, :teardown_called)
        def initialize(test)
          super(test)
          @setup_called = false
          @teardown_called = false
        end
        def setup
          @setup_called = true
        end
        def teardown
          @teardown_called = true
        end
        def test_succeed
          assert_block {true}
        end
        def test_fail
          assert_block {false}
        end
        def test_error
          raise "Error!"
        end
      end
      
      def test_setup_teardown
        result = TestResult.new
  
        test = TC_SetupTearDown.new(:test_succeed)
        test.run(result) {}
        check("Should have called setup the correct number of times", test.setup_called)
        check("Should have called teardown the correct number of times", test.teardown_called)
  
        test = TC_SetupTearDown.new(:test_fail)
        test.run(result) {}
        check("Should have called setup the correct number of times", test.setup_called)
        check("Should have called teardown the correct number of times", test.teardown_called)
  
        test = TC_SetupTearDown.new(:test_error)
        test.run(result) {}
        check("Should have called setup the correct number of times", test.setup_called)
        check("Should have called teardown the correct number of times", test.teardown_called)
  
        check("Should have had two test runs", result.run_count == 3)
        check("Should have had a test failure", result.failure_count == 1)
        check("Should have had a test error", result.error_count == 1)
        @result.add_run
      end
      
      class TC_AssertionFailedNotCalled < TestCase
        def test_thing
          raise AssertionFailedError.new
        end
      end
      
      def test_assertion_failed_not_called
        suite = TC_AssertionFailedNotCalled.suite
        check("Should have one test", suite.size == 1)
        result = TestResult.new
        suite.run(result) {}
        check("Should have had one test run", result.run_count == 1)
        check("Should have had one assertion failure", result.failure_count == 1)
        check("Should not have any assertion errors but had #{result.error_count}", result.error_count == 0)
      end
      
      def check(message, passed)
        @result.add_assertion
        if ! passed
          @result.add_failure(Failure.new(name, message))
          throw :failure
        end
      end
      
      def name
        "#{@test}(#{self.class.name})"
      end
    end
  end
end
