/* rpackagelister.h - package cache and list manipulation
 * 
 * Copyright (c) 2000, 2001 Conectiva S/A 
 * 
 * Author: Alfredo K. Kojima <kojima@conectiva.com.br>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */



#ifndef _RPACKAGELISTER_H_
#define _RPACKAGELISTER_H_


#include <vector>
#include <map>

#include <regex.h>


#include "rpackage.h"

class OpProgress;

class RPackageCache;
class RPackageFilter;

class pkgRecords;
class pkgAcquireStatus;
class pkgPackageManager;


struct RFilter;


class RInstallProgress;


class RPackageObserver {
public:
   virtual void notifyChange(RPackage *pkg) = 0;
};


class RUserDialog {
public:
   virtual bool confirm(char *title, char *message) = 0;
};


class RPackageLister
{
private:
   RPackageCache *_cache;

   pkgRecords *_records;
   OpProgress *_progMeter;
   
   RPackage **_packages;
   unsigned _count;
   
   bool _updating; // can't access anything while this is true
   
   int _installedCount; // # of installed packages
   
   vector<RFilter*> _filterL;
   RFilter *_filter; // effective filter, NULL for none
   vector<RPackage*> _displayList; // list of packages after filter

   vector<string> _sectionList; // list of all available package sections

   pkgPackageManager *_packMan;

   void applyInitialSelection();
   
   void makePresetFilters();
   
   bool applyFilters(RPackage *package);

   bool lockPackageCache(FileFd &lock);

   void getFilteredPackages(vector<RPackage*> &packages);
   
   void sortPackagesByName(vector<RPackage*> &packages);
   
   struct {
       char *pattern;       
       regex_t regex;
       bool isRegex;
       int last;
   } _searchData;

   vector<RPackageObserver*> _observers;
   
   RUserDialog *_userDialog;
   
public:
   inline void getSections(vector<string> &sections) { sections=_sectionList; };
   
   void storeFilters();
   void restoreFilters();

   // filter management
   void setFilter(int index=-1);
   inline RFilter *getFilter() { return _filter; };
   bool registerFilter(RFilter *filter);
   void unregisterFilter(RFilter *filter);
   void getFilterNames(vector<string> &filters);
   inline RFilter *findFilter(int index) { return _filterL[index]; };

   void reapplyFilter();
   
   // find 
   int findPackage(char *pattern);
   int findNextPackage();

   inline unsigned count() { return _updating ? 0 : _displayList.size(); };
   inline RPackage *getElement(int row) { 
       if (!_updating && row < (int)_displayList.size()) 
	   return _displayList[row];
       else
	   return NULL;
   };
   int getElementIndex(RPackage *pkg);

   RPackage *getElement(pkgCache::PkgIterator &pkg);

   void getStats(int &installed, int &broken, int &toinstall, int &toremove,
		 double &sizeChange);
   
   void getSummary(int &held, int &kept, int &essential,
		   int &toInstall, int &toUpgrade, int &toRemove,
		   double &sizeChange);
   

   void getDetailedSummary(vector<RPackage*> &held, 
			   vector<RPackage*> &kept, 
			   vector<RPackage*> &essential,
			   vector<RPackage*> &toInstall, 
			   vector<RPackage*> &toUpgrade, 
			   vector<RPackage*> &toRemove,
			   double &sizeChange);   

   void getDownloadSummary(int &dlCount, double &dlSize);

   bool openCache(bool reset);

   bool fixBroken();
   
   bool check();
   
   bool upgrade();
   bool distUpgrade();
   
   bool cleanPackageCache();

   bool updateCache(pkgAcquireStatus *status);
   bool commitChanges(pkgAcquireStatus *status, RInstallProgress *iprog);
   
   inline void setProgressMeter(OpProgress *progMeter) { _progMeter = progMeter; };

   inline void setUserDialog(RUserDialog *dialog) { _userDialog = dialog; };

   // notification stuff about changes in packages
   void notifyChange(RPackage *pkg);
   void registerObserver(RPackageObserver *observer);
   
   RPackageLister();   
};


#endif
