
#include "config.h"

#include "i18n.h"

#include <pwd.h>
#include <sys/types.h>

#include <fstream.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <apt-pkg/init.h>
#include <apt-pkg/error.h>
#include <apt-pkg/configuration.h>

#include "rconfiguration.h"

static string ConfigFilePath;
static string ConfigFileDir;


#ifndef HAVE_RPM
bool _ReadConfigFile(Configuration &Conf,string FName,bool AsSectional = false,
		    unsigned Depth = 0);
#endif

static void dumpToFile(const Configuration::Item *Top, ostream &out, 
		       string pad)
{
    while (Top) {	    
	out << pad << Top->Tag << " \"" << Top->Value << "\"";

	if (Top->Child) {
	    out << " {" << endl;
	    dumpToFile(Top->Child, out, pad+"  ");
	    out << pad << "};" << endl;
	    
	    if (Top->Next)
		out << endl;
	} else {
	    out << ";" << endl;
	}

	if (pad.empty())
	    break; // dump only synaptic section
	    
	Top = Top->Next;
    }
}



bool RWriteConfigFile(Configuration &Conf)
{
    const Configuration::Item *Raptor;

    ofstream cfile(ConfigFilePath.c_str(), ios::out);
    if (!cfile != 0)
	return _error->Errno("ofstream", "couldn't open %s for writing",
			     ConfigFilePath.c_str());
    
    Raptor = Conf.Tree(0);
    while (Raptor) {
	if (Raptor->Tag == "Synaptic")
	    break;
	Raptor = Raptor->Next;
    }
    
    dumpToFile(Raptor, cfile, "");

    cfile.close();

    return true;
}


static bool checkConfigDir(string &path)
{
    struct stat stbuf;
    struct passwd *pwd;
    
    pwd = getpwuid(0);
    if (!pwd) {
	return _error->Errno("getpwuid", _("Could not get password entry for superuser"));
    }
    
    path = string(pwd->pw_dir)+"/.synaptic";
    
    if (stat(path.c_str(), &stbuf) < 0) {
	if (mkdir(path.c_str(), 0700) < 0) {
	    return _error->Errno("mkdir", _("could not create configuration directory %s"),
				 path.c_str());
	}
    }
    return true;
}




bool RInitConfiguration(string confFileName)
{
    string configDir;
    
#ifdef HAVE_RPM
    if (!pkgInitialize(*_config))
       return false;
#else
    if (!pkgInitConfig(*_config))
       return false;
    if (!pkgInitSystem(*_config, _system))
       return false;
#endif

    if (!checkConfigDir(configDir))
       return false;

    ConfigFilePath = configDir+"/"+confFileName;
    ConfigFileDir = configDir;

    if (!ReadConfigFile(*_config, ConfigFilePath)) {
	_error->Discard();
    }
    
    return true;
}


bool RReadFilterData(Configuration &config)
{
    string path = ConfigFileDir+"/filters";

#ifdef HAVE_RPM
    if (!ReadConfigFile(config, path, true)) {
//	_error->DumpErrors();
	_error->Discard();
	return false;
    }
#else
    if (!_ReadConfigFile(config, path), true) {
//      _error->DumpErrors();
        _error->Discard();
        return false;
    }
#endif

    return true;
}


bool RFilterDataOutFile(ofstream &out)
{
    string path = ConfigFileDir+"/filters";

    out.open(path.c_str(), ios::out);

    if (!out != 0)
	return _error->Errno("ofstream", _("couldn't open %s for writing"),
			     path.c_str());
    
    return true;
}




#ifndef HAVE_RPM
#include <apt-pkg/strutl.h>

// ReadConfigFile - Read a configuration file				/*{{{*/
// ---------------------------------------------------------------------
/* The configuration format is very much like the named.conf format
   used in bind8, in fact this routine can parse most named.conf files. 
   Sectional config files are like bind's named.conf where there are 
   sections like 'zone "foo.org" { .. };' This causes each section to be
   added in with a tag like "zone::foo.org" instead of being split 
   tag/value. */
bool _ReadConfigFile(Configuration &Conf,string FName,bool AsSectional,
		    unsigned Depth)
{   
   // Open the stream for reading
   ifstream F(FName.c_str(),ios::in | ios::nocreate);
   if (!F != 0)
      return _error->Errno("ifstream::ifstream","Opening configuration file %s",FName.c_str());
   
   char Buffer[300];
   string LineBuffer;
   string Stack[100];
   unsigned int StackPos = 0;
   
   // Parser state
   string ParentTag;
   
   int CurLine = 0;
   bool InComment = false;
   while (F.eof() == false)
   {
      F.getline(Buffer,sizeof(Buffer));
      CurLine++;
      _strtabexpand(Buffer,sizeof(Buffer));
      _strstrip(Buffer);

      // Multi line comment
      if (InComment == true)
      {
	 for (const char *I = Buffer; *I != 0; I++)
	 {
	    if (*I == '*' && I[1] == '/')
	    {
	       memmove(Buffer,I+2,strlen(I+2) + 1);
	       InComment = false;
	       break;
	    }	    
	 }
	 if (InComment == true)
	    continue;
      }
      
      // Discard single line comments
      bool InQuote = false;
      for (char *I = Buffer; *I != 0; I++)
      {
	 if (*I == '"')
	    InQuote = !InQuote;
	 if (InQuote == true)
	    continue;
	 
	 if (*I == '/' && I[1] == '/')
         {
	    *I = 0;
	    break;
	 }
      }

      // Look for multi line comments
      InQuote = false;
      for (char *I = Buffer; *I != 0; I++)
      {
	 if (*I == '"')
	    InQuote = !InQuote;
	 if (InQuote == true)
	    continue;
	 
	 if (*I == '/' && I[1] == '*')
         {
	    InComment = true;
	    for (char *J = Buffer; *J != 0; J++)
	    {
	       if (*J == '*' && J[1] == '/')
	       {
		  memmove(I,J+2,strlen(J+2) + 1);
		  InComment = false;
		  break;
	       }	       
	    }
	    
	    if (InComment == true)
	    {
	       *I = 0;
	       break;
	    }	    
	 }
      }
      
      // Blank
      if (Buffer[0] == 0)
	 continue;
      
      // We now have a valid line fragment
      InQuote = false;
      for (char *I = Buffer; *I != 0;)
      {
	 if (*I == '"')
	    InQuote = !InQuote;
	 
	 if (InQuote == false && (*I == '{' || *I == ';' || *I == '}'))
	 {
	    // Put the last fragment into the buffer
	    char *Start = Buffer;
	    char *Stop = I;
	    for (; Start != I && isspace(*Start) != 0; Start++);
	    for (; Stop != Start && isspace(Stop[-1]) != 0; Stop--);
	    if (LineBuffer.empty() == false && Stop - Start != 0)
	       LineBuffer += ' ';
	    LineBuffer += string(Start,Stop - Start);
	    
	    // Remove the fragment
	    char TermChar = *I;
	    memmove(Buffer,I + 1,strlen(I + 1) + 1);
	    I = Buffer;
	    
	    // Syntax Error
	    if (TermChar == '{' && LineBuffer.empty() == true)
	       return _error->Error("Syntax error %s:%u: Block starts with no name.",FName.c_str(),CurLine);
	    
	    // No string on this line
	    if (LineBuffer.empty() == true)
	    {
	       if (TermChar == '}')
	       {
		  if (StackPos == 0)
		     ParentTag = string();
		  else
		     ParentTag = Stack[--StackPos];
	       }
	       continue;
	    }
	    
	    // Parse off the tag
	    string Tag;
	    const char *Pos = LineBuffer.c_str();
	    if (ParseQuoteWord(Pos,Tag) == false)
	       return _error->Error("Syntax error %s:%u: Malformed Tag",FName.c_str(),CurLine);	    

	    // Parse off the word
	    string Word;
	    if (ParseCWord(Pos,Word) == false &&
		ParseQuoteWord(Pos,Word) == false)
	    {
	       if (TermChar != '{')
	       {
		  Word = Tag;
		  Tag = "";
	       }	       
	    }
	    if (strlen(Pos) != 0)
	       return _error->Error("Syntax error %s:%u: Extra junk after value",FName.c_str(),CurLine);

	    // Go down a level
	    if (TermChar == '{')
	    {
	       if (StackPos <= 100)
		  Stack[StackPos++] = ParentTag;
	       
	       /* Make sectional tags incorperate the section into the
	          tag string */
	       if (AsSectional == true && Word.empty() == false)
	       {
		  Tag += "::" ;
		  Tag += Word;
		  Word = "";
	       }
	       
	       if (ParentTag.empty() == true)
		  ParentTag = Tag;
	       else
		  ParentTag += string("::") + Tag;
	       Tag = string();
	    }
	    
	    // Generate the item name
	    string Item;
	    if (ParentTag.empty() == true)
	       Item = Tag;
	    else
	    {
	       if (TermChar != '{' || Tag.empty() == false)
		  Item = ParentTag + "::" + Tag;
	       else
		  Item = ParentTag;
	    }
	    
	    // Specials
	    if (Tag[0] == '#')
	    {
	       if (ParentTag.empty() == false)
		  return _error->Error("Syntax error %s:%u: Directives can only be done at the top level",FName.c_str(),CurLine);
	       Tag.erase(Tag.begin());
//akk	       if (Tag == "clear")
//akk		  Conf.Clear(Word);
//akk	       else 
		    if (Tag == "include")
	       {
		  if (Depth > 10)
		     return _error->Error("Syntax error %s:%u: Too many nested includes",FName.c_str(),CurLine);		     
		  if (_ReadConfigFile(Conf,Word,AsSectional,Depth+1) == false)
		     return _error->Error("Syntax error %s:%u: Included from here",FName.c_str(),CurLine);
	       }
	       else
		  return _error->Error("Syntax error %s:%u: Unsupported directive '%s'",FName.c_str(),CurLine,Tag.c_str());
	    }
	    else
	    {
	       // Set the item in the configuration class
	       Conf.Set(Item,Word);
	    }
	    
	    // Empty the buffer
	    LineBuffer = string();
	    
	    // Move up a tag, but only if there is no bit to parse
	    if (TermChar == '}')
	    {
	       if (StackPos == 0)
		  ParentTag = string();
	       else
		  ParentTag = Stack[--StackPos];
	    }
	    
	 }
	 else
	    I++;
      }

      // Store the fragment
      const char *Stripd = _strstrip(Buffer);
      if (*Stripd != 0 && LineBuffer.empty() == false)
	 LineBuffer += " ";
      LineBuffer += Stripd;
   }

   if (LineBuffer.empty() == false)
      return _error->Error("Syntax error %s:%u: Extra junk at end of file",FName.c_str(),CurLine);
   return true;
}
									/*}}}*/
#endif
