// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  parse.cc
  Contains parsing functions allowing the starconvert utility to
  read in a specification file.
*/

#include "parse.h"

void parse_config_line(char *, parsedata *);
void parse_config_none(StringList, comments *, unsigned int);
void parse_config_coords(StringList, coordinates *, unsigned int);
void parse_config_charact(StringList, characteristics *, unsigned int);
void parse_config_systems(StringList, systems *, unsigned int);
void parse_config_names(StringList, namedata *, unsigned int);
void parse_config_subst(StringList, namedata *, unsigned int);

enum errortype { TOOMANYARGS, BADKEYWORD, BADOPTION };

void printerror(errortype type, const char * keyword, 
		const char * option, unsigned int lineno)
{
  switch (type) {
    case TOOMANYARGS: cerr << "Too many arguments for keyword `" << keyword
			   << "' on line " << lineno << "; ignoring extras."
			   << endl;
                      break;
    case BADKEYWORD:  
      {
        cerr << "Unrecognized keyword `" << keyword
	     << "' on line " << lineno << "; ignoring." << endl;
	for (unsigned int i = 0; i < strlen(keyword); i++)
	  if (keyword[i] == ' ') {
	    cerr << "Did you forget a semicolon?" << endl;
	    break;
	  }
        break;
      }
    case BADOPTION:   
      {
        cerr << "Unrecognized option `" << option 
	     << "' to keyword `" << keyword << "' on line " 
	     << lineno << "; ignoring." << endl;
	for (unsigned int i = 0; i < strlen(keyword); i++)
	  if (keyword[i] == ' ') {
	    cerr << "Did you forget a semicolon?" << endl;
	    break;
	  }
	break;
      }
  }
  return;
}


// parse_config_file(): This is the only function called from main() in
//  the convert.cc file.  The rest in this file are only used internally.

void parse_config_file(istream &configfile, parsedata *data)
{
  // initialize everything with a default value.
  data->Comments.comments_start = -1; data->Comments.comments_len = 0;
  data->Coord.isCelestial = true;
  for (unsigned int i = 0; i < NUM_COORD_OPTIONS; i++) {
    data->Coord.coord_start[i] = -1; data->Coord.coord_len[i] = 0;
  }
  data->Charact.specclass_start = vector<int>();
  data->Charact.specclass_len = vector<int>();
  data->Charact.distarray = vector<stardistance>();
  data->Charact.magarray = vector<magnitude>();
  data->Systems.comp_start = data->Systems.sep_start = -1;
  data->Systems.comp_len = data->Systems.sep_len = 0;
  data->Systems.isSeparationCommented = false;
  data->Systems.sep_prefixes = StringList();
  data->Names.names = vector<name>();
  data->Names.isSubstCaseSensitive = false;
  data->Names.subst1 = data->Names.subst2 = StringList();

  // now, go through each line of the config file.
  char temp[256];
  while (configfile.getline(temp, 256, '\n')) {
    temp[255] = 0;
    parse_config_line(temp, data);
  }

  // finally, check that the most important fields have been filled in.
#define NO_INFO    "No information given in config file about how to obtain"
#define INSUF_INFO "Not enough information in config file about how to obtain"

  if (!(data->Coord.coord_len[0]           /* ra hrs */
	&& data->Coord.coord_len[3]        /* declination sign */
	&& data->Coord.coord_len[4]        /* declination deg */
	&& data->Charact.distarray.size()) /* # of distance options */) {
    cerr << "*** " << INSUF_INFO << endl << "*** star coordinates."
	 << endl << "*** Exiting." << endl;
    exit(EXIT_FAILURE);
  }
  if (! data->Names.names.size()) {
    cerr << "*** " << INSUF_INFO << endl << "*** star names."
	 << endl << "*** Exiting." << endl;
    exit(EXIT_FAILURE);
  }
  if (! data->Charact.distarray.size()) {
    cerr << "*** " << NO_INFO << endl << "*** star distances."
	 << endl << "*** Exiting." << endl;
    exit(EXIT_FAILURE);
  }
  if (data->Charact.magarray.size() != data->Charact.distarray.size()) {
    cerr << "*** Distance and magnitude measurement options do not occur"
	 << endl << "*** in pairs in config file." << endl 
	 << "*** Exiting." << endl;
    exit(EXIT_FAILURE);
  }
  for (unsigned int i = 0; i < data->Charact.distarray.size(); i++) {
    if (data->Charact.distarray[i].type == SPECCLASS
	&& data->Charact.magarray[i].type == ABSOLUTE) {
      cerr << "*** \"magnitude; absolute\" and \"distance; specclass\" is "
	   << "not a" << endl << "*** valid pair of measurements." << endl
	   << "*** Exiting." << endl;
      exit(EXIT_FAILURE);
    }
  }
  if (! data->Charact.specclass_len.size()) {
    for (unsigned int i = 0; i < data->Charact.distarray.size(); i++) {
      if (data->Charact.distarray[i].type == SPECCLASS) {
	cerr << "*** " << NO_INFO << endl << "*** spectral classes, "
	     << "but a rule for obtaining distances depends on it."
	     << endl << "*** Exiting." << endl;
	exit(EXIT_FAILURE);
      }
    }
    cerr << "Warning! " << NO_INFO << endl << "spectral classes." << endl;
  }
  return;
}


// parse_config_line(): Strips out comments and blank lines, then passes
//  whatever's left to helper functions.

void parse_config_line(char *line, parsedata *data)
{
  static parsetype currenttype = NONE;
  static unsigned int lineno = 0;

  // increment line number
  lineno++;

  // first, destroy #'ed-out comments in the config file line.
  unsigned int i = 0;
  while (line[i] != '#' && line[i] != 0) i++;
  if (line[i] == '#') line[i] = 0;
  stripspace(line);
  if (strlen(line) == 0) return;

  StringList strlist = StringList(line, ';');
  strlist.stripspace();

  // check for type separators
  if (strcasecmp(strlist[0], "[coordinates]") == 0)
    { currenttype = COORDS; return; }
  else if (strcasecmp(strlist[0], "[characteristics]") == 0)
    { currenttype = CHARACT; return; }
  else if (strcasecmp(strlist[0], "[systems]") == 0)
    { currenttype = SYSTEMS; return; }
  else if (strcasecmp(strlist[0], "[names]") == 0)
    { currenttype = NAMES; return; }
  else if (strcasecmp(strlist[0], "[substitutions]") == 0)
    { currenttype = SUBST; return; }

  // check for keywords and parse with helper functions.
  switch (currenttype) {
  case NONE:    parse_config_none(strlist, &(data->Comments), lineno); break;
  case COORDS:  parse_config_coords(strlist, &(data->Coord), lineno); break;
  case CHARACT: parse_config_charact(strlist, &(data->Charact), lineno); break;
  case SYSTEMS: parse_config_systems(strlist, &(data->Systems), lineno); break;
  case NAMES:   parse_config_names(strlist, &(data->Names), lineno); break;
  case SUBST:   parse_config_subst(strlist, &(data->Names), lineno); break;
  }
  return;
}


// Following are all the helper functions, each of which parses a set
//  of keywords.

void parse_config_none(StringList strlist, comments *com, unsigned int lineno)
{
  if (strcasecmp(strlist[0], "comments") == 0) {
    if (strlist.size() > 1) {
      com->comments_start = myatoi(strlist[1]) - 1;
      com->comments_len = (strlist.size() > 2)
	? myatoi(strlist[2]) - com->comments_start : 1;
    }
    if (strlist.size() > 3)
      printerror(TOOMANYARGS, "comments", 0, lineno);
  }
  else printerror(BADKEYWORD, strlist[0], 0, lineno);
  return;
}


void parse_config_coords(StringList strlist, coordinates *coor,
			 unsigned int lineno)
{
  if (strcasecmp(strlist[0], "type") == 0) {
    if (strlist.size() > 1) {
      if (strcasecmp(strlist[1], "galactic") == 0)
	coor->isCelestial = false;
      else if (strcasecmp(strlist[1], "celestial") != 0)
	printerror(BADOPTION, "type", strlist[1], lineno);
      if (strlist.size() > 2)
	printerror(TOOMANYARGS, "type", 0, lineno);
    }
  }

  else {
    if (strlist.size() > 1) {
      unsigned int i, j;
      if (strlist.size() > 3)
	printerror(TOOMANYARGS, strlist[0], 0, lineno);
      for (i = 0; i < NUM_COORD_OPTIONS; i++)
	for (j = 0; j < 2; j++)
	  if (strcasecmp(strlist[0], coordoptions[i][j]) == 0) {
	    coor->coord_start[i] = myatoi(strlist[1]) - 1;
	    coor->coord_len[i] = (strlist.size() > 2)
	      ? myatoi(strlist[2]) - coor->coord_start[i] : 1;
	    return;
	  }
      printerror(BADKEYWORD, strlist[0], 0, lineno);
    }
  }
  return;
}


void parse_config_charact(StringList strlist, characteristics * ch,
			  unsigned int lineno)
{
  if (strcasecmp(strlist[0], "specclass") == 0) {
    if (strlist.size() > 1) {
      ch->specclass_start.push_back(myatoi(strlist[1]) - 1);
      ch->specclass_len.
	push_back((strlist.size() > 2)
		  ? myatoi(strlist[2]) - myatoi(strlist[1]) + 1 : 1);
      if (strlist.size() > 3)
	printerror(TOOMANYARGS, "specclass", 0, lineno);
    }
  }
   
  else if (strcasecmp(strlist[0], "distance") == 0) {
    if (strlist.size() > 1) {
      unsigned int i;
      stardistance d;
      for (i = 0; i < NUM_DISTANCE_UNITS; i++)
	if (strcasecmp(strlist[1], distanceunits[i]) == 0) break;
      if (i == NUM_DISTANCE_UNITS) {
	printerror(BADOPTION, "distance", strlist[1], lineno);
	return;
      }
      else if (strcasecmp(strlist[1], "specclass") == 0) {
	d.type = SPECCLASS;
	d.start = -1;
	d.len = 0;
	ch->distarray.push_back(d);
	if (strlist.size() > 2)
	  printerror(TOOMANYARGS, "distance", 0, lineno);
      }
      else if (strlist.size() > 2) {
	d.type = eunittype(i);
	d.start = myatoi(strlist[2]) - 1;
	d.len = (strlist.size() > 3) ? myatoi(strlist[3]) - d.start : 1;
	if (d.type == ARCSEC || d.type == MILLIARCSEC) {
	  if (strlist.size() >= 7) { // we're using max_error specifier
	    d.errorstart = myatoi(strlist[4]) - 1;
	    d.errorlen = myatoi(strlist[5]) - d.errorstart;
	    d.maxerror = myatof(strlist[6]);
	    d.minparallax = 0.0;
	  }
	  else {
	    d.errorstart = d.errorlen = 0;

	    if (strlist.size() >= 5) { // using min_parallax specifier
	      d.minparallax = myatof(strlist[4]);
	    }
	    else { // plain vanilla parallax
	      d.minparallax = 0.0;
	    }
	  }
	  if (strlist.size() > 7 || strlist.size() == 6)
	    printerror(TOOMANYARGS, "distance", 0, lineno);
	}
	else {
	  d.minparallax = 0.0;
	  if (strlist.size() > 4)
	    printerror(TOOMANYARGS, "distance", 0, lineno);
	}
	ch->distarray.push_back(d);
      }
    }
  }

  else if (strcasecmp(strlist[0], "magnitude") == 0) {
    if (strlist.size() > 1) {
      magnitude m;
      if (strcasecmp(strlist[1], "absolute") == 0)
	m.type = ABSOLUTE;
      else if (strcasecmp(strlist[1], "visual") == 0)
	m.type = VISUAL;
      else {
	printerror(BADOPTION, "magnitude", strlist[1], lineno);
	return;
      }
      if (strlist.size() > 2) {
	m.start = myatoi(strlist[2]) - 1;
	m.len = (strlist.size() > 3) ? myatoi(strlist[3]) - m.start : 1;
	ch->magarray.push_back(m);
	if (strlist.size() > 4)
	  printerror(TOOMANYARGS, "magnitude", 0, lineno);
      }
    }
  }

  else printerror(BADKEYWORD, strlist[0], 0, lineno);
  return;
}


void parse_config_systems(StringList strlist, systems *sys, 
			  unsigned int lineno)
{
  if (strcasecmp(strlist[0], "component") == 0) {
    if (strlist.size() > 1) {
      sys->comp_start = myatoi(strlist[1]) - 1;
      sys->comp_len = (strlist.size() > 2)
	? myatoi(strlist[2]) - sys->comp_len : 1;
      if (strlist.size() > 3)
	printerror(TOOMANYARGS, "component", 0, lineno);
    }
  }

  else if (strcasecmp(strlist[0], "separation") == 0) {
    if (strlist.size() > 1) {
      if (strcasecmp(strlist[1], "comments") == 0) {
	sys->isSeparationCommented = true;
        sys->sep_start = -1; sys->sep_len = 0;
	for (unsigned int i = 2; i < strlist.size(); i++)
	  sys->sep_prefixes.append(strlist[i]);
      }
      else {
        sys->sep_start = myatoi(strlist[1]) - 1;
	sys->sep_len = (strlist.size() > 2)
	  ? myatoi(strlist[2]) - sys->sep_len : 1;
	if (strlist.size() > 3)
	  printerror(TOOMANYARGS, "separation", 0, lineno);
      }
    }
  }

  else printerror(BADKEYWORD, strlist[0], 0, lineno);
  return;
}


void parse_config_names(StringList strlist, namedata *nmd, unsigned int lineno)
{
  unsigned int i = 0;
  for (i = 0; i < NUM_NAME_TYPES; i++)
    if (strcasecmp(strlist[0], nametypes[i]) == 0)
      break;
  if (i == NUM_NAME_TYPES) {
    printerror(BADKEYWORD, strlist[0], 0, lineno);
    return;
  }
  
  if (strlist.size() > 1) {
    name n;
    n.name_prefixes = StringList();
    if (strcmp(strlist[0], "other") == 0) {
      n.type = OTHER;
      if (strcmp(strlist[1], "comments") == 0) {
	n.name_start = -1; n.name_len = 0;
	n.isNameCommented = true;
	for (unsigned int j = 2; j < strlist.size(); j++)
	  n.name_prefixes.append(strlist[j]);
      }
      else {
	n.name_start = myatoi(strlist[1]) - 1;
	n.name_len = (strlist.size() > 2) 
	  ? myatoi(strlist[2]) - n.name_start : 1;
	n.isNameCommented = false;
	for (unsigned int j = 3; j < strlist.size(); j++)
	  n.name_prefixes.append(strlist[j]);
      }
    }

    else {
      n.type = enametype(i);
      if (strcmp(strlist[1], "comments") == 0) {
	n.name_start = -1; n.name_len = 0;
	n.isNameCommented = true;
	if (strlist.size() > 2)
	  printerror(TOOMANYARGS, strlist[0], 0, lineno);
      }
      else {
	n.name_start = myatoi(strlist[1]) - 1;
	n.name_len = (strlist.size() > 2) 
	  ? myatoi(strlist[2]) - n.name_start : 1;
	n.isNameCommented = false;
	if (strlist.size() > 3)
	  printerror(TOOMANYARGS, strlist[0], 0, lineno);
      }
    }
    nmd->names.push_back(n);
  }
  return;
}


void parse_config_subst(StringList strlist, namedata *nmd, unsigned int lineno)
{
  static int count = 0;

  count++;
  if (count == 1 && strcasecmp(strlist[0], "case-sensitive") == 0) {
    nmd->isSubstCaseSensitive = true;
    if (strlist.size() > 1)
      printerror(TOOMANYARGS, "case-sensitive", 0, lineno);
  }
  else if (count == 1 && strcasecmp(strlist[0], "case-insensitive") == 0) {
    if (strlist.size() > 1)
      printerror(TOOMANYARGS, "case-insensitive", 0, lineno);
  }
  else if (count == 1 && strlist.size() == 1)
    printerror(BADKEYWORD, strlist[0], 0, lineno);
  else if (strlist.size() == 2) {
    strlist.stripspace();
    nmd->subst1.append(strlist[0]);
    nmd->subst2.append(strlist[1]);
  }
  else
    cerr << "Line " << lineno << " is not a valid substitution rule." << endl;

  return;
}


// Finally, more star-related functions.

// findabsmag(): Guesses the star's absolute magnitude from its spectral class.
//  This uses data from Carroll and Ostlie, _An Introduction to Modern
//  Astrophysics_, 1996 (Appendix E).

double _findabsmag(char, double, double);

double findabsmag(const char *Class)
{
  SpecClass s = SpecClass(Class);
  s.initialize();
  double mktype = s.getMKtype();
  char spectrum = s.getCoarseType();
  double subsp  = s.getSubType();

  // round to one decimal place
  return roundoff(_findabsmag(spectrum, subsp, mktype), 1);
}

double _findabsmag(char spectrum, double subsp, double mktype)
{
  if (mktype < 1.0) {
    // in this case, no MK type is available; 
    //  try to save a few cases from being thrown out
    if (spectrum == 'D') return 14;
    else if (spectrum == 'W') mktype = 3.0;
    else return -9999;
  }

  if (fabs(mktype - 5.0) < 0.2) {
    switch (spectrum) {
      case 'W': // fall through
      case 'O': return -7.4 + 0.34 * subsp;
      case 'B': return -4.0 + 0.34 * subsp;
      case 'A': return  0.6 + 0.21 * subsp;
      case 'F': return  2.7 + 0.17 * subsp;
      case 'G': return  4.4 + 0.15 * subsp;
      case 'K': return  5.9 + 0.29 * subsp;
      case 'M': return  8.8 + 0.9  * subsp;
      case 'D': return 14;
      default:  return -9999;
    }
  }

  else if (fabs(mktype - 3.0) < 0.2) {
    switch (spectrum) {
      case 'W': // fall through
      case 'O': return -7.5 + 0.24 * subsp;
      case 'B': return -5.1 + 0.51 * subsp;
      case 'A': return  0.0 + 0.15 * subsp;
      case 'F': return  1.6;
      case 'G': return  1.0 - 0.07 * subsp;
      case 'K': return  0.7 - 0.11 * subsp;
      case 'M': return -0.5;
      default:  return -9999;
    }
  }

  else if (fabs(mktype - 1.0) < 0.2) {
    switch (spectrum) {
      case 'W': // fall through
      case 'O': return -6.5;
      case 'B': return -6.4 + 0.02 * subsp;
      case 'A': return -6.3 - 0.03 * subsp;
      case 'F': return -6.6;
      case 'G': return -6.4 + 0.04 * subsp;
      case 'K': return -6.0 + 0.04 * subsp;
      case 'M': return -5.6;
      default:  return -9999;
    }
  }

  else { // linearly interpolate
    if (mktype < 3.0)
      return (mktype - 1.0) * _findabsmag(spectrum, subsp, 3.0) / 2.0
	+ (3.0 - mktype) * _findabsmag(spectrum, subsp, 1.0) / 2.0;
    else
      return (mktype - 3.0) * _findabsmag(spectrum, subsp, 5.0) / 2.0
	+ (5.0 - mktype) * _findabsmag(spectrum, subsp, 3.0) / 2.0;
  }
}
