/*
 * The Sleuth Kit
 *
 * $Date: 2005/01/17 22:40:17 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2005 Brian Carrier.  All rights reserved
 *
 * mmstat - Get stats on the volume system / media management
 *
 * This file is part of mmtools
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * mmtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "mm_tools.h"


extern char *progname;

FILE *logfp;

void
usage()
{
    fprintf(stderr,
	    "%s [-i imgtype] [-o imgoffset] [-vV] [-t mmtype] image [images]\n",
	    progname);
    fprintf(stderr, "\t-t mmtype: The type of partition system\n");
    fprintf(stderr, "\t-i imgtype: The format of the image file\n");
    fprintf(stderr,
	    "\t-o imgoffset: Offset to the start of the volume that contains the partition system (in sectors)\n");
    fprintf(stderr, "\t-v: verbose output\n");
    fprintf(stderr, "\t-V: print the version\n");
    fprintf(stderr, "Supported partition systems:\n");
    mm_print_types(stderr);
    fprintf(stderr, "Supported image formats:\n");
    img_print_types(stderr);
    exit(1);
}

static void
print_stats(MM_INFO * mm)
{
    printf("%s\n", mm_get_type(mm->mmtype));
    //printf("Type: %s\n", mm->str_type);
    return;
}

int
main(int argc, char **argv)
{
    MM_INFO *mm;
    char *mmtype = NULL;
    int ch;
    DADDR_T sect_offset = 0;
    char *imgtype = NULL;
    IMG_INFO *img;
    logfp = stderr;

    progname = argv[0];

    while ((ch = getopt(argc, argv, "i:o:t:vV")) > 0) {
	char *cp;
	switch (ch) {
	case 'i':
	    imgtype = optarg;
	    break;

	    // @@@ WE may want this to support the @ syntax...
	case 'o':
	    sect_offset = strtoul(optarg, &cp, 0);
	    if (*cp || cp == optarg) {
		fprintf(stderr, "Bad sector offset\n");
		usage();
	    }

	    break;
	case 't':
	    mmtype = optarg;
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    print_version(stdout);
	    exit(0);
	case '?':
	default:
	    fprintf(stderr, "Unknown argument\n");
	    usage();
	}
    }

    /* We need at least one more argument */
    if (optind >= argc) {
	fprintf(stderr, "Missing image name\n");
	usage();
    }

    /* open the image - do not use an offset */
    img =
	img_open(imgtype, "0", argc - optind,
		 (const char **) &argv[optind]);

    /* process the partition tables */
    mm = mm_open(img, mmtype, sect_offset);
    if (mm == NULL) {
	if (mmtype == NULL)
	    printf("Error determining partition system type\n");
	else
	    fprintf(stderr, "Error processing image\n");
	exit(1);
    }

    print_stats(mm);

    mm->close(mm);
    img->close(img);
    exit(0);
}
