############################################################################
##      Copyright (C) 2005 Subredu Manuel  <diablo@iasi.roedu.net>.        #
##                                                                         #
## This program is free software; you can redistribute it and/or modify    #
## it under the terms of the GNU General Public License v2 as published by #
## the Free Software Foundation.                                           #
##                                                                         #
## This program is distributed in the hope that it will be useful,         #
## but WITHOUT ANY WARRANTY; without even the implied warranty of          #
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           #
## GNU General Public License for more details.                            #
##                                                                         #
## You should have received a copy of the GNU General Public License       #
## along with this program; if not, write to the Free Software             #
## Foundation, Inc., 59 Temple Place - Suite 330, Boston,                  #
## MA 02111-1307,USA.                                                      #
############################################################################

package RoPkg::Simba::Plugin::GenSitemap;

use strict;
use warnings;

use RoPkg::Exceptions;
use DateTime;
use HTML::Template;
use Scalar::Util qw(blessed);

use vars qw($VERSION);

$VERSION='0.3.1';

my $defaults_values = {
  enabled      => 'yes',
  templatesDir => q{./},
  gmt_offset   => '+02:00',
};

sub new {
  my ($class, %opt) = @_;
  my $self;

  $self = bless { %opt }, $class;

  return $self;
}

#Returns 1 if plugin is enabled, 0 otherwise
sub _enabled {
  my ($self, $cfg) = @_;

  #if enabled found
  if ( $cfg->{enabled} ) {

    #return 1 if the value is yes (case insensitive)
    return 1 if ( $cfg->{enabled} =~ m{^yes$}xmi );

    #else, return 0
    return 0;
  }

  #if no enabled was found, assume yes
  return 1;
}

#load the configuration and initialize the parameters
sub _load_config {
  my ($self, $cfg) = @_;

  $self->{cfg} = $cfg;

  foreach(qw(enabled templatesDir gmt_offset)) {
    if ( $cfg->{$_} ) {
      $self->{$_} = $cfg->{$_};
    }
  }

  foreach(keys(%{ $defaults_values })) {
    if ( !$self->{$_} ) {
      $self->{$_} = $defaults_values->{$_};
    }
  }

  return $self;
}

#generates a sitemap
sub _genSitemap {
  my ($self, $hd, $mirrors) = @_;
  my @loop_data;
  my $tmpl;

  #create the template object.
  $tmpl = HTML::Template->new(
            filename => $self->{templatesDir} . $hd->{template},
            loop_context_vars => 1,
            die_on_bad_params => 0,
          );
  
  foreach(@{ $mirrors }) {
    my $mirror = $_;
    my %mdata;

    #init the template variables
    $mdata{UTC_DATE} = (DateTime->from_epoch( epoch => $mirror->LastUpdated) . $self->{gmt_offset});
    $mdata{Name}     = $mirror->Name;
    $mdata{LocalURL} = $mirror->LocalURL;
    push(@loop_data, \%mdata);
  }

  $tmpl->param(Mirrors => \@loop_data);

  #create the file and write the content
  RoPkg::Utils::CreateFile($hd->{outfile}, $tmpl->output);

  return 1;
}

#cycle trough the sitemaps list and generate them
sub _genSitemaps {
  my ($self, $mirrors) = @_;

  foreach(sort(keys(%{ $self->{cfg} }))) {
    next if ( $_ !~ m{^s_}xm );
    $self->_genSitemap($self->{cfg}->{$_}, $mirrors);
  }

  return 1;
}

sub genSitemap {
  my ($self, $cfg, $mirrors) = @_;
  my $scfg;
  
  if ( !blessed($self) ) {
    OutsideClass->throw('Called outside class instance');
  }

  #do nothing if no configuration items are found
  return 0 if ( !$cfg->{plugins}->{sitemap} );
  
  $scfg = $cfg->{plugins}->{sitemap};
  $self->_load_config($scfg);
  
  #do nothing if plugin is disabled
  return 0 if ( !$self->_enabled($cfg->{plugins}->{sitemap}) );

  #nothing to do if no mirrors are given
  return 0 if ((scalar @{ $mirrors }) <= 0 );

  return $self->_genSitemaps($mirrors);
}

1;


__END__

=head1 NAME

RoPkg::Simba::Plugin::GenSitemap

=head1 DESCRIPTION

GenSitemap is a google sitemap generator
plugin for Simba. GenSitemap allows you 
to create custom sitemaps from the mirror
updates information.

=head1 PREREQUISITES

GenSitemap needs the following perl modules:

=over 3

=item *) HTML::Template

=item *) Scalar::Util

=item *) DateTime

=back

=head1 METHODS

=head2 genSitemap($cfg, @mirrors)

Generate sitemaps. The $cfg is a hash reference to the
simba configuration file. This method is called by Simba.

=head1 CONFIGURATION

All configuration documentation is available on the website
(http://simba.packages.ro).

=head1 SEE ALSO

L<RoPkg::Simba> L<RoPkg::Exceptions>

=head1 AUTHOR

Subredu Manuel <diablo@iasi.roedu.net>

=head1 LICENSE

Copyright (C) 2005 Subredu Manuel.  All Rights Reserved.
This module is free software; you can redistribute it 
and/or modify it under the same terms as Perl itself.
The LICENSE file contains the full text of the license.

=cut
