#include "Python.h"
#include "domlette.h"


/*Internal Methods */

PyDocumentObject *Document_NEW(unsigned long *docIx, const char *refUri)
{
  StringPool *sp;
  register PyDocumentObject *doc;


  doc = (PyDocumentObject *) malloc(sizeof(PyDocumentObject));

  if (doc == NULL) {
    PyErr_NoMemory();
    return NULL;
  }

  Node_INIT((PyNodeObject*)doc, (PyDocumentObject*)Py_None);
  doc->ob_type = &PyDomletteDocument_Type;

  doc->documentData.documentElement = Py_None;
  doc->nodeData.docIndex = (*docIx)++;
  doc->documentData.refUri = PyString_FromString(refUri);

  Py_INCREF(Py_None);
  doc->documentData.childNodes = PyList_New(0);

  /* Init the string pool */
  sp = string_pool_create();

  if (sp == NULL) {
    PyErr_NoMemory();
    return NULL;
  }

  doc->documentData.stringPool = sp;

  _Py_NewReference(doc);
  return doc;
}


void Document_ReleaseNode(PyDocumentObject *doc)
{
  int ctr;

  Py_XDECREF(doc->documentData.documentElement);
  doc->documentData.documentElement = NULL;

  /*Recursively call on childNodes*/
  for(ctr = 0; ctr < PyList_GET_SIZE(doc->documentData.childNodes); ctr ++) {
    Node_ReleaseNode((PyNodeObject *)PyList_GET_ITEM(doc->documentData.childNodes,ctr));
  }
  Py_XDECREF(doc->documentData.childNodes);
  doc->documentData.childNodes = NULL;

  string_pool_destroy(doc->documentData.stringPool);
}

PyElementObject *Document_CreateElementNS(PyDocumentObject *doc,
					  const char *namespaceURI,
					  const char *prefix,
					  const char *localName,
					  unsigned long *docIx)
{
  char *nodeName;
  register PyElementObject *ele =
    (PyElementObject *) malloc(sizeof(PyElementObject));

  /*if (ele == NULL)
    return PyErr_NoMemory();
  */

  Node_INIT((PyNodeObject*)ele, doc);

  ele->ob_type = &PyDomletteElement_Type;
  ele->elementData.attributes = PyDict_New();
  ele->elementData.childNodes = PyList_New(0);
  ele->nodeData.docIndex = (*docIx)++;

  /*The from_pool does an incref*/
  ele->nodeData.namespaceURI = pystring_from_pool(doc->documentData.stringPool,
                                                  namespaceURI);
  ele->nodeData.prefix = pystring_from_pool(doc->documentData.stringPool,
                                            prefix);
  ele->nodeData.localName = pystring_from_pool(doc->documentData.stringPool,
                                               localName);
  nodeName = nodename_from_parts(prefix, localName);
  ele->nodeData.nodeName = pystring_from_pool(doc->documentData.stringPool,
                                              nodeName);
  free(nodeName);

  _Py_NewReference(ele);

  return ele;
}


PyTextObject *Document_CreateTextNode(PyDocumentObject *doc,
				      const char *data, unsigned long *docIx)
{
  register PyTextObject *text =
    (PyTextObject *) malloc(sizeof(PyTextObject));

  /*if (text == NULL)
    return PyErr_NoMemory();
  */

  Node_INIT((PyNodeObject*)text, doc);

  text->ob_type = &PyDomletteText_Type;
  text->textData.nodeValue = pystring_from_pool(doc->documentData.stringPool, data);
  text->nodeData.docIndex = (*docIx)++;

  _Py_NewReference(text);
  return text;
}


PyCommentObject *Document_CreateComment(PyDocumentObject *doc, const char *data, unsigned long *docIx)
{
  register PyCommentObject *comment =
    (PyCommentObject *) malloc(sizeof(PyCommentObject));

  /*if (text == NULL)
    return PyErr_NoMemory();
  */

  Node_INIT((PyNodeObject*)comment, doc);

  comment->ob_type = &PyDomletteComment_Type;
  comment->commentData.nodeValue = PyString_FromString(data);
  comment->nodeData.docIndex = (*docIx)++;

  _Py_NewReference(comment);
  return comment;
}


PyProcessingInstructionObject *Document_CreateProcessingInstruction(
                                    PyDocumentObject *doc,
                                    const char *target,
                                    const char *data, unsigned long *docIx)
{
  register PyProcessingInstructionObject *pi =
    (PyProcessingInstructionObject *) malloc(sizeof(PyProcessingInstructionObject));

  /*if (text == NULL)
    return PyErr_NoMemory();
  */

  Node_INIT((PyNodeObject*)pi, doc);

  pi->ob_type = &PyDomletteProcessingInstruction_Type;
  pi->piData.target = PyString_FromString(target);
  pi->piData.nodeValue = PyString_FromString(data);
  pi->nodeData.docIndex = (*docIx)++;

  _Py_NewReference(pi);
  return pi;
}


PyNodeObject *Document_AppendChild(PyDocumentObject * self,PyNodeObject *child) {

  if PyElement_Check(child)
    {
    Py_XDECREF(self->documentData.documentElement);
    self->documentData.documentElement = (PyObject *)child;
    Py_XINCREF(child);
  }

  /*Assumes ownership*/
  PyList_Append(self->documentData.childNodes,(PyObject *)child);
  Py_XDECREF(child);

  Py_XDECREF(child->nodeData.parentNode);
  child->nodeData.parentNode = (PyObject *)self;
  Py_XINCREF(self);

  return child;
}


/*External Methods*/
static struct PyMethodDef Document_methods[] = { NODE_METHODS,
  {NULL,     NULL}      /* sentinel */
};


/*Type methods*/

static PyObject *document_getattr(PyDocumentObject *self, char *name)
{
  PyObject *rt = NULL;
  int temp;

  if (!strcmp(name,"ownerDocument")) {
    rt = Py_None;
  }
  else if (!strcmp(name,"documentElement")) {
    rt = self->documentData.documentElement;
  }
  else if (!strcmp(name,"implementation")) {
    rt = g_implementation;
  }
  else if (!strcmp(name,"doctype")) {
    rt = Py_None;
  }
  else if (!strcmp(name,"refUri")) {
    rt = self->documentData.refUri;
  }
  else if (!strcmp(name,"nodeType")) {
    rt = g_documentNodeType;
  }
  else if (!strcmp(name,"nodeName")) {
    rt = g_documentNodeName;
  }
  else if (!strcmp(name,"childNodes")) {
    rt = self->documentData.childNodes;
  }
  else if (!strcmp(name,"lastChild")) {
    temp = PyList_GET_SIZE(self->documentData.childNodes);
    if (temp) {
      rt = PyList_GET_ITEM(self->documentData.childNodes,temp-1);
    } else {
      rt = Py_None;
    }
  }
  else if (!strcmp(name,"firstChild")) {
    if (PyList_GET_SIZE(self->documentData.childNodes)) {
      rt = PyList_GET_ITEM(self->documentData.childNodes,0);
    } else {
      rt = Py_None;
    }
  }
  if (rt) {
    Py_INCREF(rt);
    return rt;
  }
  return node_getattr((PyNodeObject*)self,name,Document_methods);
}


static void document_dealloc(PyDocumentObject *node)
{
  node_dealloc((PyNodeObject*)node);
}


static void document_build_display_string(char *buf, PyDocumentObject *v)
{
  sprintf(buf, "<cDomlette Document at %lx>", (long)v);
}


static int document_print(PyDocumentObject *v, FILE *fp, int flags)
{
  char buf[256];
  document_build_display_string(buf,v);
  fputs(buf, fp);
  return 0;
}


PyTypeObject PyDomletteDocument_Type = {
    PyObject_HEAD_INIT(0)
    0,
    "cDomlette.Document",
    sizeof(PyDocumentObject),
    0,
    (destructor)document_dealloc,   /*tp_dealloc*/
    (printfunc)document_print,  /*tp_print*/
    (getattrfunc)document_getattr,  /*tp_getattr*/
    0,              /*tp_setattr*/
    0,                          /*tp_compare*/
    0,                              /*tp_repr*/
    0,                          /*tp_as_number*/
    0,              /*tp_as_sequence*/
    0,              /*tp_as_mapping*/
    0,                             /*tp_hash*/
    0,          /*tp_call*/
    0,          /*tp_str*/
    0,                      /*tp_getattro*/
    0,          /*tp_setattro*/
};

