//
// C++ Implementation: %{MODULE}
//
// Description:
//
//
// Author: %{AUTHOR} <%{EMAIL}>, (C) %{YEAR}
//
// Copyright: See COPYING file that comes with this distribution
//
//

#include <algorithm>

#include <q3header.h>

#include "customizablelistview.h"
#include "columncontroldlg.h"


namespace NWidget {

CustomizableListView::CustomizableListView(Q3ListView* pLv, QObject *parent, const char *name)
 : QObject(parent, name), _sortOrder(Qt::Ascending)
{
	_pLv = pLv;
	
	// update the shown short information order whenever order changes
	connect(_pLv->header(), SIGNAL(indexChange(int, int, int)), this, SLOT(refreshShown()));
	connect(_pLv->header(), SIGNAL(sizeChange(int, int, int)), this, SLOT(refreshShown()));
	
	connect(_pLv->header(), SIGNAL(indexChange(int, int, int)), this, SLOT(refreshSorting()));
	connect(_pLv->header(), SIGNAL(clicked(int)), this, SLOT(refreshSorting()));
}


CustomizableListView::~CustomizableListView()
{
}

/////////////////////////////////////////////////////
// IXmlStorable Interface
/////////////////////////////////////////////////////

void CustomizableListView::saveSettings(NXml::XmlData& outData, QDomElement parent) const 
{
	QDomElement listView = outData.addElement(parent, "customListView");
	outData.addAttribute(listView, QString("1"), "settingsVersion");
	
	QDomElement shownColumns = outData.addElement(listView, "shownColumns");
	
	outData.addAttribute(shownColumns, sortColumn(), "shortSortColumn");
	outData.addAttribute(shownColumns, _pLv->sortOrder(), "shortSortOrder");
	
	int i = 0;
	for (ColumnList::const_iterator it = _shownColumns.begin(); it != _shownColumns.end(); ++it, ++i)
	{
		QDomElement column = outData.addElement(shownColumns, "column");
		outData.addAttribute(column, it->_caption, "name");
		outData.addAttribute(column, it->_width, "width");
	}
	
	QDomElement hiddenColumns = outData.addElement(listView, "hiddenColumns");
	for (ColumnList::const_iterator it = _hiddenColumns.begin(); it != _hiddenColumns.end(); ++it)
	{
		QDomElement column = outData.addElement(hiddenColumns, "column");
		outData.addAttribute(column, it->_caption, "name");
		outData.addAttribute(column, it->_width, "width");
	}
}

QDomElement CustomizableListView::loadSettings(const QDomElement source)
{
	if (source.tagName() != "customListView")
		return source;

	QString settingsVersion;
	NXml::getAttribute(source, settingsVersion, "settingsVersion", "0.1");
	QDomElement element = NXml::getFirstElement(source.firstChild());
	if (element.tagName() == "shownColumns")
	{

		if (settingsVersion >= QString("1"))
		{
			QDomElement e = NXml::getFirstElement(element.firstChild());
			while (!e.isNull())
			{
				QString name;
				NXml::getAttribute(e, name, "name", "");
				int width;
				NXml::getAttribute(e, width, "width", 50);
				addColumn(name, width);
				e = NXml::getNextElement(e);
			}
		}
		// note that we set the sort column, _after_ adding all the columns.
		QString sortColumn;
		int sortOrder;
		NXml::getAttribute(element, sortColumn, "shortSortColumn", _sortColumn);
//		cout<<"Sort column" <<sortColumn.toAscii()<<endl;
		qDebug(sortColumn);
		NXml::getAttribute(element, sortOrder, "shortSortOrder", _sortOrder);
		if (!sortColumn.isEmpty())
			setSorting(sortColumn, (Qt::SortOrder) sortOrder);
		
		element = NXml::getNextElement(element);
	}
	if (element.tagName() == "hiddenColumns")
	{
		if (settingsVersion >= QString("0.1"))
		{
			QDomElement e = NXml::getFirstElement(element.firstChild());
			while (!e.isNull())
			{
				QString name;
				NXml::getAttribute(e, name, "name", "");
				int width;
				NXml::getAttribute(e, width, "width", 50);
				_hiddenColumns.push_back(Column(name, width));
				e = NXml::getNextElement(e);
			}
		}
		element = NXml::getNextElement(element);	
	}
	return NXml::getNextElement(source);
}

/////////////////////////////////////////////////////
// Other Methods
/////////////////////////////////////////////////////


void CustomizableListView::setSorting(QString column, Qt::SortOrder sortOrder)
{
	_sortColumn = column;
	_sortOrder = sortOrder;
	for (int i=0; i<_pLv->columns(); ++i)
	{
		if (_pLv->columnText(i)==column)
		{
			_pLv->setSorting(i, ((Qt::SortOrder) sortOrder) == Qt::Ascending);
			return;
		}
	}
}

void CustomizableListView::refreshSorting()
{
	int column = _pLv->sortColumn();
	if (column == -1)
		_sortColumn = QString();
	else
		_sortColumn = _pLv->header()->label(column);
	_sortOrder = _pLv->sortOrder();
}


QString CustomizableListView::sortColumn() const
{
	return _sortColumn;
}

bool CustomizableListView::isExisting(QString caption) const
{
	return 
		_shownColumns.find(caption) != _shownColumns.end() || 
		_hiddenColumns.find(caption) != _hiddenColumns.end();
}


void CustomizableListView::addColumn(QString caption, int width)
{
	if (isExisting(caption))
		return;
	_shownColumns.push_back( Column(caption, width) );
	int column = _pLv->addColumn(caption, width);
	if (caption == _sortColumn)
		_pLv->setSorting(column, _sortOrder);
	emit shownColumnsChanged();
}

void CustomizableListView::addColumn(QString caption, int column, int width)
{
	if (isExisting(caption))
		return;
	_shownColumns.insert(column, Column(caption, width));
	updateListView();
	emit shownColumnsChanged();
}




void CustomizableListView::removeColumnsNotInList(const QStringList columns)
{
	// don't increment the iterator in the loop header as this is done inside
	for (ColumnList::iterator it = _shownColumns.begin(); it != _shownColumns.end(); )
	{
		if (columns.find(it->_caption) == columns.end())
		{
			it = _shownColumns.erase(it);
		}
		else
			++it;
	}
	// don't increment the iterator in the loop header as this is done inside
	for (ColumnList::iterator it = _hiddenColumns.begin(); it != _hiddenColumns.end(); )
	{
		if (columns.find(it->_caption) == columns.end())
		{
			it = _hiddenColumns.erase(it);
		}
		else
			++it;
	}
	updateListView();
}

void CustomizableListView::removeColumn(QString caption)
{
	ColumnList::iterator it  = _shownColumns.find(caption);
	if (it != _shownColumns.end())
	{
		_shownColumns.erase(it);
		updateListView();
		emit shownColumnsChanged();
	}
	it  = _hiddenColumns.find(caption);
	if (it != _hiddenColumns.end())
		_hiddenColumns.erase(it);
}



void CustomizableListView::showControlDialog()
{
	ColumnControlDlg uiDlg(0);
	uiDlg.setContent(_shownColumns.getCaptions(), _hiddenColumns.getCaptions());
	if (uiDlg.exec() == QDialog::Accepted)
	{
		QStringList shown = uiDlg.getShown();
		ColumnList shownColumns;
		for (QStringList::const_iterator it = shown.begin(); it != shown.end(); ++it)
		{
			shownColumns.push_back(Column(*it, getColumnWidth(*it)));
		}
		QStringList hidden = uiDlg.getHidden();
		ColumnList hiddenColumns;
		for (QStringList::const_iterator it = hidden.begin(); it != hidden.end(); ++it)
		{
			hiddenColumns.push_back(Column(*it, getColumnWidth(*it)));
		}
		_shownColumns.swap(shownColumns);
		_hiddenColumns.swap(hiddenColumns);
		updateListView();
		emit shownColumnsChanged();
	}
}

CustomizableListView::ColumnList::iterator CustomizableListView::getColumn(QString caption)
{
	ColumnList::iterator it  = _shownColumns.find(caption);
	if (it != _shownColumns.end())
		return it;
	it  = _hiddenColumns.find(caption);
	return it;
}

int CustomizableListView::getColumnWidth(QString caption) const 
{
	ColumnList::const_iterator it  = _shownColumns.find(caption);
	if (it != _shownColumns.end())
		return it->_width;
	it  = _hiddenColumns.find(caption);
	if (it != _hiddenColumns.end())
		return it->_width;
	return -1;
}
void CustomizableListView::updateListView()
{
	while ( _pLv->columns() > 0)
		_pLv->removeColumn(0);
	for (ColumnList::const_iterator it = _shownColumns.begin(); it != _shownColumns.end(); ++it)
	{
		int column = _pLv->addColumn(it->_caption, it->_width);
		if (it->_caption == _sortColumn)
		{
			_pLv->setSortColumn(column); 
			_pLv->setSortOrder(_sortOrder); 
		}
	}
}

void CustomizableListView::refreshShown()
{
	_shownColumns.clear();
	Q3Header* pHeader = _pLv->header();
	for (int index = 0; index < pHeader->count(); ++index)
	{
		int section = pHeader->mapToSection(index);
		_shownColumns.push_back( Column(pHeader->label(section), _pLv->columnWidth(section)) );
	}
}


int CustomizableListView::mapToSection(int index) const
{ 
	return _pLv->header()->mapToSection(index); 
}



};
