/* EnhancedTable.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.tables;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.JTable;
import javax.swing.event.TableModelEvent;
import javax.swing.table.JTableHeader;

/**
 * Table component which allows table column headers to be decorated with small
 * icons and buttons, i.e., objects of type {@link TableColumnDecoration}.
 * Typical usage is as follows:
 * <ul>
 * <li>Create a table object using the default constructor {@link EnhancedTable}.</li>
 * <li>Register decoration prototypes 
 * with {@link #addPrototypeDecoration(Class,TableColumnDecoration)} or 
 * {@link #addPrototypeDecoration(TableColumnDecoration)}.</li>
 * <li>Set the data model with {@link #setModel}.</li>
 * <li>Perform any further initialisations, as with {@link JTable}.
 * </ul>
 * The actual column headers are decorated with <i>clones</i> of the given prototypes
 * and each clone is initialized with the corresponding column index.
 * The choice of prototype is determined from the column class (as 
 * determined by the model), or if no prototype was registered for a particular
 * class, from a default prototype registered with the table.
 * 
 */
public class EnhancedTable extends JTable {
    
    /** Default constructor */
    public EnhancedTable () {
        super ();
        decorations = new ArrayList <List<TableColumnDecoration>> ();
        decorationsByClass = new HashMap<Class,List<TableColumnDecoration>> ();
        defaultDecorations = new ArrayList <TableColumnDecoration>();
    }
    
    // Actual decorations
    private final List<List<TableColumnDecoration>> decorations;
        
    /**
     * Return the list of decorations for the given column.
     * @param modelIndex model index of the column
     * @return the list of decorations or null when the list is empty.
     */
    public List<TableColumnDecoration> getDecorations (int modelIndex) {
        if (modelIndex < 0 || modelIndex > decorations.size ())
            return null;
        else
            return decorations.get (modelIndex);
    }
    
    //
    private final List<TableColumnDecoration> defaultDecorations;
    
    //
    private final Map<Class,List<TableColumnDecoration>> decorationsByClass;
    
    /**
     * Add a prototype decoration for a given column class. Every column of
     * this class will obtain a clone of this prototype.
     */
    public void addPrototypeDecoration (Class clazz, TableColumnDecoration decoration) {
        List<TableColumnDecoration> list = decorationsByClass.get (clazz);
        if (list == null) {
            list = new ArrayList <TableColumnDecoration>();
            decorationsByClass.put (clazz, list);
        }
        if (decoration != null)
            list.add (decoration);
    }
    
    /**
     * Add a prototype decoration for default columns, i.e., columns with
     * a column class for which no decorations were registered.
     */
    public void addPrototypeDecoration (TableColumnDecoration decoration) {
        defaultDecorations.add (decoration);
    }
    

    //
    protected JTableHeader createDefaultTableHeader () {
        return new EnhancedTableHeader (columnModel);
    }
    
    /**
     * Invoked when this table's <code>TableModel</code> generates
     * a <code>TableModelEvent</code>. In addition to the standard
     * behaviour, this implementation also (re)initializes
     * the column decorations.<p>
     * Should not be called by application code. For internal use only.
     */
    public void tableChanged (TableModelEvent e) {
        if (decorations != null && // null only during startup
            (e == null || e.getFirstRow () == TableModelEvent.HEADER_ROW) ) {
            decorations.clear ();
            int size = dataModel.getColumnCount ();
            for (int i=0; i < size; i++) {
                Class clazz = dataModel.getColumnClass (i);
                List<TableColumnDecoration> list = decorationsByClass.get (clazz);
                if (list == null)
                    list = defaultDecorations;
                if (list != null) {
                    List<TableColumnDecoration> decos = new ArrayList<TableColumnDecoration> ();
                    decorations.add (decos);
                    for (TableColumnDecoration deco : list) {
                        TableColumnDecoration decoClone = deco.copy();
                        decoClone.setColumnIndex (i);
                        decos.add (decoClone);
                    }
                }
            }
        }
        super.tableChanged (e);
    }
}
