/* DragHandler.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.dnd;

import javax.swing.JComponent;

/**
 * Handles the 'drag' part of a drag-and-drop operation of a local object of given type.
 * Used in conjunction with {@link LocalTransferHandler}. 
 */
public interface DragHandler {
    
    /**
     * Indicates whether this handler allows a transfer from the given source to the given target.
     * Typically this either returns {@code true} always, or returns
     * true if and only if the source and target are different. 
     * A more sophisticated implementation
     * would instead compare the underlying models of the components.
     * @see LocalTransferHandler#isTransferAllowed
     */
    public boolean isTransferAllowed (JComponent source, JComponent target);
    
    /**
     * Returns the type of transfer actions supported by the given component.
     * Should return one the constants {@code COPY}, {@code MOVE}, 
     * {@code COPY_OR_MOVE}, {@code LINK} or {@code NONE}, cf. {@link LocalTransferHandler}. 
     * @see LocalTransferHandler#getSourceActions
     */
    public int getSourceActions (JComponent source);
    
    /**
     * Invoked after data has been dragged from the given source 
     * component and dropped onto the given target component. Typically, when the action
     * is {@code MOVE} and source and target are not the same (or are not views
     * of the same model), 
     * the objects need to be removed from the source. When the
     * action is {@code COPY} or {@code LINK}, nothing needs to be done.<p>
     * @param objects Array of objects which have been exported
     * @param type Element type of this array
     * @param action the actual action that was performed 
     * @param source The component that was the source of the data. 
     * @param target The component that was the target of the data, or {@code null}
     * if the target is not a {@link LocalTransferHandler}.
     * @see LocalTransferHandler#exportDone
     */
    public void exportDone (JComponent source, JComponent target, 
            Object[] objects, Class<?> type, int action);
    
    /**
     * Return the object(s) to be exported by a drag
     * or cut-and-paste operation from the given source. If multiple
     * objects are exported at the same time, this should return an array. 
     * @see LocalTransferHandler#getExportedObjects
     */
    public Object getExportedObjects (JComponent source);
    
    /**
     * Return the class of objects being exported from the given source. If this
     * returns null, the class of the object
     * returned by {@link #getExportedObjects} is used, or the element type if an array is returned.
     * @see LocalTransferHandler#getExportedClass
     */
     public Class getExportedClass (JComponent source);
    
    
}
