/*
    LIBSNSP - A C Library for the Simple Network Scanning Protocol
    Copyright (C) 2001 Michael R. Kllejan 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: cmd.c,v 1.6 2001/09/29 15:13:09 michael Exp $ */

#include <stdio.h>

#include "cmd.h"
#include "reply.h"

#include "snsp_errno.h"
#include "macros.h"
#include "const.h"


snsp_cmd_t *
snsp_cmd_alloc ()
{
  snsp_cmd_t *cmd;

  if (!(cmd = (snsp_cmd_t *) malloc (sizeof (snsp_cmd_t))))
    ERROR(SNSPG_ENOMEM, return NULL;)

  cmd->cmd = NULL;

  if (!(cmd->parameters = LINKED_LIST_ALLOC))
    ERROR(SNSPG_ENOMEM, return NULL;)

  return cmd;
}

void
snsp_cmd_free (snsp_cmd_t ** cmd) 
{
  if (!(cmd)) ERROR ( SNSPG_EPARA, return; )
  if (!(*cmd)) ERROR ( SNSPG_EPARA, return; )

  if ((*cmd)->cmd)
    free ((*cmd)->cmd);

  if ((*cmd)->parameters)
    LINKED_LIST_FREE (&(*cmd)->parameters);

  free (*cmd);
  *cmd = NULL;
};

#define STR_ERROR ERROR (SNSPC_ELINETOOLONG, return NULL;)
char *
snsp_cmd_render (snsp_cmd_t *cmd)
{
  char *cmd_str = NULL; int cmd_str_cnt = SNSP_CMD_LEN_MAX+1;
  int para_cnt = 0, i = 0;
  int b_cont_dq = 0, b_cont_ws = 0;

  if (!((cmd == NULL) || (cmd->cmd == NULL))) ERROR (SNSPG_EPARA, return NULL;)
  if (!(cmd_str = (char*) malloc (SNSP_CMD_LEN_MAX+1)))
    ERROR (SNSPG_ENOMEM, return NULL;)

  if (strlen(cmd->cmd) != SNSP_CMD_CMD_LEN) 
    ERROR (SNSPC_ECMDFLSLEN, return NULL;)
  for (i = 0; i < SNSP_CMD_CMD_LEN; i++) 
    if (!(isalpha(cmd->cmd[i])))
      ERROR (SNSPC_ECMDILLGCHR, return NULL;)
  SSTRCPY (cmd_str, cmd->cmd, cmd_str_cnt, STR_ERROR)

  cmd->parameters = LINKED_LIST_FIRST (cmd->parameters);
  LINKED_LIST_WALK_BEGIN (cmd->parameters) {
    if (strlen(cmd->parameters->data) > SNSP_CMD_PARA_LEN_MAX)
      ERROR (SNSPC_EPARATOOLONG, return NULL;)

    b_cont_ws = b_cont_dq = 0;
              
    for (i = 0; i < strlen(cmd->parameters->data); i++) {
      if (!(snsp_isprint (cmd->parameters->data[i]) || 
            snsp_isspace (cmd->parameters->data[i])))
        ERROR (SNSPC_EPARAILLGCHR, return NULL;)
      if (snsp_isspace (cmd->parameters->data[i])) b_cont_ws = 1;
      if (cmd->parameters->data[i] == '\"') b_cont_dq = 1;
    };
              
    SSTRCAT (cmd_str, " ", cmd_str_cnt, STR_ERROR);

    if (b_cont_dq) ERROR (SNSPC_EPARACONTQUOTE, return NULL;)

    if (b_cont_ws) {
      SSTRCAT (cmd_str, "\"", cmd_str_cnt, STR_ERROR);
	    SSTRCAT (cmd_str, cmd->parameters->data, cmd_str_cnt, STR_ERROR);
	    SSTRCAT (cmd_str, "\"", cmd_str_cnt, STR_ERROR);
  	} else {
    	SSTRCAT (cmd_str, cmd->parameters->data, cmd_str_cnt, STR_ERROR);
    };

    if (++para_cnt == SNSP_CMD_PARA_MAX) 
      ERROR ( SNSPC_EPARATOOMUCH, return NULL;)
  } LINKED_LIST_WALK_END (cmd->parameters)
  
  SSTRCAT (cmd_str, "\n", cmd_str_cnt, STR_ERROR);
  return cmd_str;
};
#undef STR_ERROR

#define STR_ERROR goto str_error;
#define MEM_ERROR goto mem_error;
#define PARSE_ERROR goto parse_error;
snsp_cmd_t*
snsp_cmd_parse (char *cmd_str)
{
  int cmd_str_len;
  snsp_cmd_t *cmd;   
  char *str;  

  int i = 0, j = 0, k = 0;
  char qs;

  if (!cmd_str) PARSE_ERROR;
  if (!(cmd_str_len = SSTRLEN (cmd_str, SNSP_CMD_LEN_MAX))) PARSE_ERROR;

  if (!(cmd = snsp_cmd_alloc ())) MEM_ERROR;
  if (!(str = (char *) malloc (SNSP_CMD_LEN_MAX))) MEM_ERROR;
  
  /* Parse and verify command */
  if (cmd_str_len < 4) PARSE_ERROR
          
  for (i = 0; i < SNSP_CMD_CMD_LEN; i++) 
    if (!(isalpha(cmd_str[i])))
      PARSE_ERROR

  if (!(cmd->cmd = (char *) malloc (SNSP_CMD_CMD_LEN+1))) MEM_ERROR;
  strncpy (cmd->cmd, cmd_str, SNSP_CMD_CMD_LEN);
  cmd->cmd[SNSP_CMD_CMD_LEN] = '\0';

  /* Parse and verify parameters */
  while (1) {

    for (k=0; snsp_isspace (cmd_str[i]); i++) {
      IFMAX (i, cmd_str_len, STR_ERROR)
      //IFMAX (++k-1, SNSP_WHITESPACE_MAX, PARSE_ERROR)
    };
    
    qs = (cmd_str[i] == '\"') ? (i++, 1) : 0;
    IFMAX (i, cmd_str_len, PARSE_ERROR)

    j = 0;
            
    while ((qs) ? (cmd_str[i] != '\"') : (snsp_isspace (cmd_str[i]) == 0)) {

  	  if (snsp_isprint (cmd_str[i]) || snsp_isspace (cmd_str[i]))
	      str[j++] = cmd_str[i++];
  	  else {
#ifdef	DEBUG_NO_TELNET
	      if (cmd_str[i] == '\n')
#else
	      if (cmd_str[i] == '\r' )
#endif
		      break;
	      else
          PARSE_ERROR
	    };

	    if (j > SNSP_CMD_PARA_MAX)
	      goto error;
	  };

    str[j] = '\0';
    if (strlen(str) != 0)
      LINKED_LIST_PUSH (cmd->parameters, (void *) str, strlen (str) + 1);

#ifdef 	DEBUG_NO_TELNET
    if (cmd_str[i] == '\n')
#else
    if (cmd_str[i] == '\r')
#endif
	    break;

    i++;
  };

#ifndef	DEBUG_NO_TELNET
  i++;
#endif
  
  if (i >= cmd_str_len)
    goto error;
  if (cmd_str[i] == '\n')
    goto success;

  error:
  str_error:
  mem_error:
  parse_error:
    snsp_cmd_free (&cmd);
    cmd = NULL;
  success:
    free (str);
    return cmd;
};
#undef STR_ERROR
#undef MEM_ERROR
#undef PARSE_ERROR

snsp_cmd_t *
snsp_cmd_create (char *cmd_cmd, ...)
{
  snsp_cmd_t *cmd = snsp_cmd_alloc ();
  va_list ap;
  char *str;

  if (cmd_cmd == NULL)
    return NULL;

  va_start (ap, cmd_cmd);

  cmd->cmd = (char *) malloc (strlen (cmd_cmd) + 1);
  strcpy (cmd->cmd, cmd_cmd);

  while ((str = va_arg (ap, char *)) != NULL)
      LINKED_LIST_PUSH (cmd->parameters, str, strlen (str) + 1);

  va_end (ap);

  return cmd;
}
