/*
 * Licensed to the University Corporation for Advanced Internet Development, 
 * Inc. (UCAID) under one or more contributor license agreements.  See the 
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache 
 * License, Version 2.0 (the "License"); you may not use this file except in 
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package edu.internet2.middleware.shibboleth.common.attribute.resolver.provider.dataConnector;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Collection;
import java.util.Map;
import java.util.UUID;

import javax.sql.DataSource;

import org.opensaml.saml2.core.SubjectQuery;
import org.opensaml.saml2.core.AuthnRequest;
import org.opensaml.xml.XMLObject;
import org.opensaml.xml.util.Base64;
import org.opensaml.xml.util.DatatypeHelper;
import org.opensaml.xml.util.LazyMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import edu.internet2.middleware.shibboleth.common.attribute.BaseAttribute;
import edu.internet2.middleware.shibboleth.common.attribute.provider.BasicAttribute;
import edu.internet2.middleware.shibboleth.common.attribute.resolver.AttributeResolutionException;
import edu.internet2.middleware.shibboleth.common.attribute.resolver.provider.ShibbolethResolutionContext;
import edu.internet2.middleware.shibboleth.common.attribute.resolver.provider.dataConnector.StoredIDStore.PersistentIdEntry;
import edu.internet2.middleware.shibboleth.common.profile.provider.SAMLProfileRequestContext;

/**
 * A data connector that generates persistent identifiers in one of two ways. The generated attribute has an ID of
 * <tt>peristentId</tt> and contains a single {@link String} value.
 * 
 * If a salt is supplied at construction time the generated IDs will be the Base64-encoded SHA-1 hash of the user's
 * principal name, the peer entity ID, and the salt.
 * 
 * If a {@link DataSource} is supplied the IDs are created and managed as described by {@link StoredIDStore}.
 */
public class StoredIDDataConnector extends BaseDataConnector {

    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(StoredIDDataConnector.class);

    /** Persistent identifier data store. */
    private StoredIDStore pidStore;

    /** ID of the attribute generated by this data connector. */
    private String generatedAttribute;

    /** ID of the attribute whose first value is used when generating the computed ID. */
    private String sourceAttribute;

    /** Salt used when computing the ID. */
    private byte[] salt;

    /**
     * Constructor.
     * 
     * @param source datasource used to communicate with the database
     * @param queryTimeout SQL qeury timeout in seconds
     * @param generatedAttributeId ID of the attribute generated by this data connector
     * @param sourceAttributeId ID of the attribute whose first value is used when generating the computed ID
     * @param idSalt salt used when computing the ID
     */
    public StoredIDDataConnector(DataSource source, int queryTimeout, String generatedAttributeId,
            String sourceAttributeId, byte[] idSalt) {
        if (source == null) {
            throw new IllegalArgumentException("Data source may not be null");
        }
        pidStore = new StoredIDStore(source, queryTimeout);

        if (DatatypeHelper.isEmpty(generatedAttributeId)) {
            throw new IllegalArgumentException("Provided generated attribute ID must not be empty");
        }
        generatedAttribute = generatedAttributeId;

        if (DatatypeHelper.isEmpty(sourceAttributeId)) {
            throw new IllegalArgumentException("Provided source attribute ID must not be empty");
        }
        sourceAttribute = sourceAttributeId;

        if (idSalt.length < 16) {
            throw new IllegalArgumentException("Provided salt must be at least 16 bytes in size.");
        }
        salt = idSalt;
    }

    /**
     * Gets the data store used to manage stored IDs.
     * 
     * @return data store used to manage stored IDs
     */
    public StoredIDStore getStoredIDStore() {
        return pidStore;
    }

    /**
     * Gets the salt used when computing the ID.
     * 
     * @return salt used when computing the ID
     */
    public byte[] getSalt() {
        return salt;
    }

    /**
     * Gets the ID of the attribute whose first value is used when generating the computed ID.
     * 
     * @return ID of the attribute whose first value is used when generating the computed ID
     */
    public String getSourceAttributeId() {
        return sourceAttribute;
    }

    /**
     * Gets the ID of the attribute generated by this connector.
     * 
     * @return ID of the attribute generated by this connector
     */
    public String getGeneratedAttributeId() {
        return generatedAttribute;
    }

    /** {@inheritDoc} */
    public void validate() throws AttributeResolutionException {
        if (getDependencyIds() == null || getDependencyIds().size() != 1) {
            log.error("Stored ID " + getId() + " data connectore requires exactly one dependency");
            throw new AttributeResolutionException("Computed ID " + getId()
                    + " data connectore requires exactly one dependency");
        }

        try {
            pidStore.getActivePersistentIdEntry("1");
        } catch (SQLException e) {
            throw new AttributeResolutionException("Unable to connect to persistent ID store.");
        }
    }

    /** {@inheritDoc} */
    public Map<String, BaseAttribute> resolve(ShibbolethResolutionContext resolutionContext)
            throws AttributeResolutionException {
        Map<String, BaseAttribute> attributes = new LazyMap<String, BaseAttribute>();

        String principalName =
                DatatypeHelper.safeTrimOrNullString(resolutionContext.getAttributeRequestContext().getPrincipalName());

        String localId = getLocalId(resolutionContext);
        if (localId == null) {
            log.debug("No user local ID available, skipping ID creation.");
            return attributes;
        }

        String localEntityId =
                DatatypeHelper.safeTrimOrNullString(resolutionContext.getAttributeRequestContext().getLocalEntityId());
        if (localEntityId == null) {
            log.debug("No local entity ID available, skipping ID creation.");
            return attributes;
        }

        String peerEntityId = getPeerEntityId(resolutionContext);
        if (peerEntityId == null) {
            log.debug("No peer entity ID available, skipping ID creation.");
            return attributes;
        }

        String persistentId = getStoredId(principalName, localEntityId, peerEntityId, localId);
        if (persistentId != null) {
            BasicAttribute<String> attribute = new BasicAttribute<String>();
            attribute.setId(getGeneratedAttributeId());
            attribute.getValues().add(persistentId);
            attributes.put(attribute.getId(), attribute);
        }
        return attributes;
    }

    /**
     * Gets the persistent ID stored in the database. If one does not exist it is created.
     * 
     * @param principalName principal name of the user to whom the persistent ID belongs
     * @param localEntityId ID of the local entity associated with the persistent ID
     * @param peerEntityId ID of the peer entity associated with the persistent ID
     * @param localId principal the the persistent ID represents
     * 
     * @return persistent ID
     * 
     * @throws AttributeResolutionException thrown if there is a problem retrieving or storing the persistent ID
     */
    protected String getStoredId(String principalName, String localEntityId, String peerEntityId, String localId)
            throws AttributeResolutionException {
        PersistentIdEntry idEntry;
        try {
            log.debug("Checking for existing, active, stored ID for principal '{}'", principalName);
            idEntry = pidStore.getActivePersistentIdEntry(localEntityId, peerEntityId, localId);
            if (idEntry == null) {
                log.debug("No existing, active, stored ID, creating a new one for principal '{}'", principalName);
                idEntry = createPersistentId(principalName, localEntityId, peerEntityId, localId);
                pidStore.storePersistentIdEntry(idEntry);
                log.debug("Created stored ID '{}'", idEntry);
            } else {
                log.debug("Located existing stored ID {}", idEntry);
            }

            return idEntry.getPersistentId();
        } catch (SQLException e) {
            log.debug("Database error retrieving persistent identifier", e);
            throw new AttributeResolutionException("Database error retrieving persistent identifier", e);
        }
    }

    /**
     * Gets the local ID component of the persistent ID.
     * 
     * @param resolutionContext current resolution context
     * 
     * @return local ID component of the persistent ID
     * 
     * @throws AttributeResolutionException thrown if there is a problem resolving the local id
     */
    protected String getLocalId(ShibbolethResolutionContext resolutionContext) throws AttributeResolutionException {
        Collection<Object> sourceIdValues = getValuesFromAllDependencies(resolutionContext, getSourceAttributeId());
        if (sourceIdValues == null || sourceIdValues.isEmpty()) {
            log.debug("Source attribute {} for connector {} provide no values.  No identifier will be generated.",
                    getSourceAttributeId(), getId());
            return null;
        }

        if (sourceIdValues.size() > 1) {
            log.warn("Source attribute {} for connector {} has more than one value, only the first value is used",
                    getSourceAttributeId(), getId());
        }

        return sourceIdValues.iterator().next().toString();
    }

    /**
     * Gets the entity ID used for the peer. If the inbound request is a SAML 2 authentication context and contains a
     * NameIDPolicy than the SPNameQualifier is used if present, otherwise the inbound message issuer is used.
     * 
     * @param resolutionContext current attribute resolution context
     * 
     * @return the entity ID to use for the peer
     */
    protected String getPeerEntityId(ShibbolethResolutionContext resolutionContext) {
        SAMLProfileRequestContext requestContext = resolutionContext.getAttributeRequestContext();
        
        String peerEntityId = null;

        log.debug("Determining if peer entity ID will be the SPNameQualifier from a SAML 2 authentication statement");
        XMLObject inboundMessage = requestContext.getInboundSAMLMessage();
        if (inboundMessage instanceof AuthnRequest) {
            AuthnRequest authnRequest = (AuthnRequest) inboundMessage;
            if (authnRequest.getNameIDPolicy() != null) {
                peerEntityId = DatatypeHelper.safeTrimOrNullString(authnRequest.getNameIDPolicy().getSPNameQualifier());
                if (peerEntityId == null) {
                    log.debug("SAML 2 authentication request did not contain an SPNameQualifier within its NameIDPolicy");
                } else {
                    log.debug("SAML 2 authentication request contained an SPNameQualifier, within its NameIDPolicy.  Using that as peer entity ID");
                }
            } else {
                log.debug("SAML 2 authentication request did not contain a NameIDPolicy");
            }
        } else if (inboundMessage instanceof SubjectQuery) {
            SubjectQuery query = (SubjectQuery) inboundMessage;
            if (query.getSubject().getNameID().getSPNameQualifier() != null) {
                peerEntityId =
                        DatatypeHelper.safeTrimOrNullString(query.getSubject().getNameID().getSPNameQualifier());
                if (peerEntityId == null) {
                    log.debug("SAML 2 subject query did not contain an SPNameQualifier within its NameID");
                } else {
                    log.debug("SAML 2 subject query contained an SPNameQualifier, within its NameID.  Using that as peer entity ID");
                }
            } else {
                log.debug("SAML 2 attribute query did not contain a SPNameQualifier");
            }
        } else {
            peerEntityId = requestContext.getInboundMessageIssuer(); 
        }

        if (peerEntityId == null) {
            log.debug("Determining if inbound message issuer is available for use as peer entity ID");
            peerEntityId = resolutionContext.getAttributeRequestContext().getInboundMessageIssuer();
        }

        return peerEntityId;
    }

    /**
     * Creates a persistent ID that is unique for a given local/peer/localId tuple.
     * 
     * If an ID has never been issued for to the given tuple then an ID is created by taking a SHA-1 hash of the peer's
     * entity ID, the local ID, and a salt. This is to ensure compatability with IDs created by the now deprecated
     * {@link ComputedIDDataConnector}.
     * 
     * If an ID has been issued to the given tuple than a new, random type 4 UUID is generated as the persistent ID.
     * 
     * @param principalName principal name of the user to whom the persistent ID belongs
     * @param localEntityId ID of the local entity associated with the persistent ID
     * @param peerEntityId ID of the peer entity associated with the persistent ID
     * @param localId principal the the persistent ID represents
     * 
     * @return the created identifier
     * 
     * @throws SQLException thrown if there is a problem communication with the database
     */
    protected PersistentIdEntry createPersistentId(String principalName, String localEntityId, String peerEntityId,
            String localId) throws SQLException {
        PersistentIdEntry entry = pidStore.new PersistentIdEntry();
        entry.setLocalEntityId(localEntityId);
        entry.setPeerEntityId(peerEntityId);
        entry.setPrincipalName(principalName);
        entry.setLocalId(localId);

        String persistentId;
        int numberOfExistingEntries =
                pidStore.getNumberOfPersistentIdEntries(entry.getLocalEntityId(), entry.getPeerEntityId(),
                        entry.getLocalId());

        if (numberOfExistingEntries == 0) {
            try {
                MessageDigest md = MessageDigest.getInstance("SHA");
                md.update(entry.getPeerEntityId().getBytes());
                md.update((byte) '!');
                md.update(localId.getBytes());
                md.update((byte) '!');
                persistentId = Base64.encodeBytes(md.digest(salt));
            } catch (NoSuchAlgorithmException e) {
                log.error("JVM error, SHA-1 is not supported, unable to compute ID");
                throw new SQLException("SHA-1 is not supported, unable to compute ID");
            }
        } else {
            persistentId = UUID.randomUUID().toString();
        }

        while (pidStore.getPersistentIdEntry(persistentId, false) != null) {
            log.debug("Generated persistent ID was already assigned to another user, regenerating");
            persistentId = UUID.randomUUID().toString();
        }

        entry.setPersistentId(persistentId);

        entry.setCreationTime(new Timestamp(System.currentTimeMillis()));

        return entry;
    }
}