/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: BackgroundOp.hh,v 1.3 2001/12/19 12:45:58 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#ifndef __BACKGROUNDOP_HH_FLAG__
#define __BACKGROUNDOP_HH_FLAG__

#include <vector>
#include <iostream>
#include <pthread.h>

#include <fwbuilder/FWException.hh>
#include <fwbuilder/Tools.hh>
#include <fwbuilder/ThreadTools.hh>
#include <fwbuilder/Pool.hh>

#include <sigc++/signal_system.h>

namespace libfwbuilder
{

class Logger;

extern Logger &start (Logger &);
extern Logger &end   (Logger &);

class Logger
{
    protected:

    Mutex line_lock;
    friend Logger &start (Logger &);
    friend Logger &end   (Logger &);

    public:

    virtual ~Logger() {};

    virtual Logger& operator<< (char c)            = 0;
    virtual Logger& operator<< (char  *str)        = 0;
    virtual Logger& operator<< (const std::string &str) = 0;
    virtual Logger& operator<< (int    i  )        = 0;
    virtual Logger& operator<< (long   l  )        = 0;

    // Manipulator
    virtual Logger& operator<< (Logger&(*f)(Logger &l)) 
    {
        return f(*this);
    }

};

class NullLogger:public Logger
{
    public:

    virtual Logger& operator<< (char c)            ;
    virtual Logger& operator<< (char  *str)        ;
    virtual Logger& operator<< (const std::string &str) ;
    virtual Logger& operator<< (int    i  )        ;
    virtual Logger& operator<< (long   l  )        ;
};

class StreamLogger:public Logger
{
    private:

    std::ostream *stream;

    public:
    
    StreamLogger(std::ostream *s);
    
    virtual Logger& operator<< (char c)            ;
    virtual Logger& operator<< (char  *str)        ;
    virtual Logger& operator<< (const std::string &str) ;
    virtual Logger& operator<< (int    i  )        ;
    virtual Logger& operator<< (long   l  )        ;
};


/**
 *  Abstract class BackgroundOp represents operatioin executed in background
 */
class BackgroundOp : public SigC::Object
{
    friend void *background_thread(void *);

    private: 

    SyncFlag     stop_program ;
    SyncFlag     running      ;

    protected:

    FWException *error;
    Pool<BackgroundOp> *pool;
    Mutex pool_mutex;
        
    /**
     * Implementation of actual operation. 
     */
    virtual void  run_impl(Logger *) throw(FWException) = 0;

    /**
     * BackgroundOp calls this method from background_thread when
     * run_impl returns. 
     *
     * This method should call BackgroundOp::operation_complted and then
     * perform any specific actions which need to be taken after operation
     * finished. This may be destroying variables, sending report to the
     * Logger, etc.
     *
     * It should never be called directly.
     */
    virtual void operation_completed();

    void     set_running_flag();
    void     clear_running_flag();

    virtual void check_stop() throw(FWException)
    {
        if(get_stop_program_flag())
            throw FWException("Interrupted by user");
    }
    
    /**
     * User should never manipulate stop flag directly.
     * he should use 'stop_operation' method instead.
     */
    void     clear_stop_program_flag ();
    void     set_stop_program_flag   ();

    public:
    
    BackgroundOp();
    virtual ~BackgroundOp() {}

    void detach(Pool<BackgroundOp> *p);

    /**
     * These methods are called to initiate and
     * interrupt background operation, respectively.
     */
    virtual pthread_t start_operation(std::ostream *)  throw(FWException);
    virtual void stop_operation();

    bool get_running_flag();
    bool get_stop_program_flag();

    FWException *get_latest_error() { return error; }

    /**
     * This signal handler to be called externally from time to time
     * to ensure that completed signal is issued when task is done.
     * typical way would be:
     *
     * bio->start_operation(logger);
     * Gtk::Main::idle.connect(SigC::slot(bio, &BackgroundOp::monitor_operation));
     */
    gint monitor_operation();

    SigC::Signal1<int,int>completed;
};

}

#endif // __BACKGROUNDOP_HH_FLAG__
