/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2007, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * $Id$
 * ------------
 * (C) Copyright 2006-2007, by Pentaho Corporation.
 */
package org.jfree.formula.typing.sequence;

import org.jfree.formula.EvaluationException;
import org.jfree.formula.FormulaContext;
import org.jfree.formula.typing.ArrayCallback;
import org.jfree.formula.typing.Type;

/**
 * @author Cedric Pronzato
 */
public class NumberSequence
{
  private int rowCursor = 0;
  private int columnCursor = 0;
  private Number number;
  private ArrayCallback array;
  private FormulaContext context;

  /**
   * Empty number sequence.
   */
  public NumberSequence(final FormulaContext context)
  {
    this.context = context;
  }

  /**
   * Number sequence bounded to only one number item.
   *
   * @param n A number
   */
  public NumberSequence(final Number n, final FormulaContext context)
  {
    number = n;
    this.context = context;
  }

  /**
   * Number sequence bounded to an array.
   *
   * @param array
   */
  public NumberSequence(final ArrayCallback array, final FormulaContext context)
  {
    this.array = array;
    this.context = context;
  }

  public boolean hasNext() throws EvaluationException
  {
    // empty sequence
    if (number == null && array == null)
    {
      return false;
    }
    // sequence of one number
    if (number != null && rowCursor == 0)
    {
      return true;
    }

    // else array
    if (array != null)
    {
      final int rowCount = array.getRowCount();
      final int columnCount = array.getColumnCount();
      if (array != null && rowCursor < rowCount && columnCursor < columnCount)
      {
        for (; rowCursor < rowCount; rowCursor++)
        {
          for (; columnCursor < columnCount; columnCursor++)
          {
            final Type type = array.getType(rowCursor, columnCursor);
            final boolean b = type.isFlagSet(Type.NUMERIC_TYPE);

            if (b)
            {
              return true;
            }
          }
          columnCursor = 0; // go to start of the next row
        }
      }
    }

    return false;
  }

  public Number nextNumber() throws EvaluationException
  {
    if (number != null && rowCursor == 0)
    {
      rowCursor++;
      return number;
    }
    if (array != null)
    {
      final Type type = array.getType(rowCursor, columnCursor);
      final Object value = array.getValue(rowCursor, columnCursor);
      final Number number = context.getTypeRegistry().convertToNumber(type, value);
      // advance
      if (columnCursor == array.getColumnCount() - 1)
      {
        rowCursor++;
        columnCursor = 0;
      }
      else
      {
        columnCursor++;
      }
      return number;

    }
    return null;
  }

}
