/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "flux.h"
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include "test.h"

#define TESTS_NUM 20

struct test
{
  int (*func)();
  char *desc, *success, *failure;
  unsigned int timeout;
};


struct test tests[TESTS_NUM] =
{
  { 0,                    "Token trees",      0,           0,                0 },
  { test_tt_size,         "Node size",        "%d (good)", "%d (aberrant)", 10 },
  { test_tt_allocation,   "Allocation",       "passed",    "failed",        10 },
  { test_tt_navigation,   "Navigation",       "passed",    "failed",        10 },
  { test_tt_matching,     "Matching",         "passed",    "failed",        10 },
  { test_tt_dataops,      "Data ops",         "passed",    "failed",        10 },
  { test_tt_external,     "External data",    "passed",    "failed",        20 },
  { test_tt_hashing,      "Hashing",          "passed",    "failed",        10 },
  { test_tt_printability, "Print/scan",       "passed",    "failed",        20 },
  { test_tt_speed,        "TT ops/sec",       "%d",        "%d",            60 },
  { 0,                    "Markup trees",     0,           0,                0 },
  { test_mt_size,         "Node size",        "%d (good)", "%d (aberrant)", 10 },
  { test_mt_allocation,   "Allocation",       "passed",    "failed",        10 },
  { test_mt_navigation,   "Navigation",       "passed",    "failed",        10 },
  { test_mt_xml_simple,   "Print/scan XML (simple)","passed","failed",      20 },
  { 0,                    "FIFO buffers",     0,           0,                0 },
  { test_fifobuf_integrity_small,"Integrity (small)","passed","failed",     10 },
  { test_fifobuf_integrity_large,"Integrity (large)","passed","failed",     10 },
  { test_fifobuf_through_small,"Throughput (small)","%dK/s",  "%dK/s",      10 },
  { test_fifobuf_through_large,"Throughput (large)","%dK/s",  "%dK/s",      10 }
};



void crash(int sig)
{
  printf("crashed\n\n");
  abort();
}


void timeout(int sig)
{
  printf("timed out\n\n");
  abort();
}


int main(int argc, char *argv[])
{
  int i, r;
  char *result;

  signal(SIGSEGV, crash);
  signal(SIGALRM, timeout);

  printf("\nFlux robustness check and benchmarking commencing.\n\n"
         "Build environment\n"
         "  Host name... %s\n"
         "  Host OS... %s\n"
         "  Host CPU... %s\n"
         "  Flux version... %s\n",
         flux_get_build_host(), flux_get_build_os(), flux_get_build_cpu(),
         flux_get_version());
  
  for (i = 0; i < TESTS_NUM; i++)
  {
    if (!tests[i].func) printf("\n%s\n", tests[i].desc);
    else
    {
      printf("  %s... ", tests[i].desc);
      fflush(stdout);

      alarm(tests[i].timeout);
      r = tests[i].func();
      alarm(0);
      
      if (r >= 0) result = tests[i].success;
      else
      {
        result = tests[i].failure;
        r = -r;
      }
      
      if (strchr(result, '%'))
      {
        printf(result, r);
        fputc('\n', stdout);
        fflush(stdout);
      }
      else printf("%s\n", result);
    }
  }

  fputc('\n', stdout);
  return(0);
}
