/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#ifdef TIME_WITH_SYS_TIME
#  include <sys/time.h>
#  include <time.h>
#elif defined(HAVE_SYS_TIME_H)
#  include <sys/time.h>
#else
#  include <time.h>
#endif

#include <time.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/signal.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>

#ifdef HAVE_SYS_FILIO_H
#  include <sys/filio.h>
#endif

#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "log.h"
#include "sock.h"


int sock_init()
{
  signal(SIGPIPE, SIG_IGN);
  _sock_initialized = 1;
  return(1);
}


SOCK *sock_open(int port_local, unsigned long flags)
{
  SOCK *s;
  struct sockaddr_in sock_addr;
  int _keepalive = 1;


  /* TODO: Implement flags SOCK_RETRY and SOCK_SECURE */
  
  /* --- Check if first run-time socket operation --- */

  if (!_sock_initialized) sock_init();

  /* --- Allocate structure --- */

  s = malloc(sizeof(SOCK));
  memset(s, 0, sizeof(SOCK));

  /* --- Open socket --- */

  s->handle = socket(AF_INET, SOCK_STREAM, 0);
  if (s->handle < 0) { free(s); return(0); }

  /* --- Initialize and bind socket --- */

  memset((char *) &sock_addr, 0, sizeof(sock_addr));
  sock_addr.sin_family = AF_INET;
  sock_addr.sin_addr.s_addr = htonl(INADDR_ANY);
  sock_addr.sin_port = htons(port_local);

  if (bind(s->handle, (struct sockaddr *) &sock_addr, sizeof(sock_addr)) < 0)
  { close(s->handle); free(s); return(0); }

  s->port_bound = port_local;
  s->info_remote = 0;

  /* --- Prepare listening socket --- */
  
  if (flags & SOCK_ACCEPT)
  {
    if (listen(s->handle, 5) < 0) { close(s->handle); free(s); return(0); }
  }

  /* --- Prepare buffers --- */

  s->fib_in = fifobuf_new(2, 32, 512);
  s->fib_out = fifobuf_new(2, 64, 512);
  
  /* --- Set various defaults --- */

  s->parent = 0;
  s->timeout_sec = SOCK_TIMEOUT_DEFAULT;
  s->flags = flags;
  s->status = SOCK_STAT_OK;
  fcntl(s->handle, F_SETFL, 0);
  setsockopt(s->handle, SOL_SOCKET, SO_KEEPALIVE, (const void *) &_keepalive, sizeof(_keepalive));

  if (s->flags & SOCK_LOG) 
  {
    if (s->port_bound) log_put_opt(LOG_DEBUG, 0, "[Sock] Opened socket on port %d.", s->port_bound);
    else log_put_opt(LOG_DEBUG, 0, "[Sock] Opened socket, no port yet.");
  }
  
  return(s);
}


SOCK *sock_pipe_from_handles(int in, int out)
{
  SOCK *s;


  /* --- Allocate structure --- */

  s = malloc(sizeof(SOCK));
  memset(s, 0, sizeof(SOCK));

  /* --- Prepare buffers --- */

  s->fib_in = fifobuf_new(2, 32, 512);
  s->fib_out = fifobuf_new(2, 64, 512);

  /* --- Set pipes --- */
  
  s->pipe_fd0[0] = in;
  s->pipe_fd1[1] = out;

  /* --- Set various defaults --- */
  
  s->parent = 0;
  s->timeout_sec = SOCK_TIMEOUT_DEFAULT;
  s->status = SOCK_STAT_OK;
  
  fcntl(s->pipe_fd0[0], F_SETFL, 0);
  fcntl(s->pipe_fd1[1], F_SETFL, 0);

  s->flags = SOCK_PIPE_CHILD | SOCK_CONNECTED;
  
  return(s);
}


SOCK *sock_pipe(unsigned long flags)
{
  SOCK *s;
  
  /* --- Allocate structure --- */

  s = malloc(sizeof(SOCK));
  memset(s, 0, sizeof(SOCK));

  /* --- Prepare buffers --- */

  s->fib_in = fifobuf_new(2, 32, 512);
  s->fib_out = fifobuf_new(2, 64, 512);

  /* --- Open pipes --- */

  if (pipe(s->pipe_fd0) < 0)
  {
    fifobuf_free(s->fib_in); fifobuf_free(s->fib_out); free(s);
    if (flags & SOCK_DEBUG || flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Pipe open failed; out of file descriptors.");
    return(0); 
  }
  if (pipe(s->pipe_fd1) < 0)
  {
    close(s->pipe_fd0[0]);
    close(s->pipe_fd0[1]);
    fifobuf_free(s->fib_in); fifobuf_free(s->fib_out); free(s);
    if (flags & SOCK_DEBUG || flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Pipe open failed; out of file descriptors.");
    return(0);
  }

  /* --- Set various defaults --- */

  s->parent = 0;
  s->timeout_sec = SOCK_TIMEOUT_DEFAULT;
  s->status = SOCK_STAT_OK;
  
  fcntl(s->pipe_fd0[0], F_SETFL, 0);
  fcntl(s->pipe_fd0[1], F_SETFL, 0);
  fcntl(s->pipe_fd1[0], F_SETFL, 0);
  fcntl(s->pipe_fd1[1], F_SETFL, 0);
  
  s->flags = SOCK_PIPE_PARENT | SOCK_PIPE_CHILD | SOCK_CONNECTED | flags;
  
  if (s->flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Opened loopback pipe.");
  return(s);
}


int sock_pipe_parent(SOCK *s)
{
  if (!(s->flags & SOCK_PIPE))
  {
    if (s->flags & SOCK_DEBUG || s->flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Tried to make parent-pipe from non-pipe.");
    return(0);
  }
  
  close(s->pipe_fd0[0]);  /* Close reading end of first pipe. */
  close(s->pipe_fd1[1]);  /* Close writing end of second pipe. */

  /* Pipe setup:

     Write to  pipe_fd0[1].
     Read from pipe_fd1[0]. */

  s->flags |= SOCK_PIPE_PARENT;
  s->flags &= ~SOCK_PIPE_CHILD;
  if (s->flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Separated pipe parent.");
  return(1);
}

int sock_pipe_child(SOCK *s)
{
  if (!(s->flags & SOCK_PIPE))
  {
    if (s->flags & SOCK_DEBUG || s->flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Tried to make child-pipe from non-pipe.");
    return(0);
  }
  close(s->pipe_fd0[1]);  /* Close writing end of first pipe. */
  close(s->pipe_fd1[0]);  /* Close reading end of second pipe. */

  /* Pipe setup:

     Write to  pipe_fd1[1].
     Read from pipe_fd0[0]. */

  s->flags &= ~SOCK_PIPE_PARENT;
  s->flags |= SOCK_PIPE_CHILD;
  if (s->flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Separated pipe child.");
  return(1);
}


void sock_close(SOCK *s, unsigned long flags)
{
  if (s->flags & SOCK_CONNECTED)
    if (!(flags & SOCK_IMMEDIATE)) sock_flush(s);

  if (s->flags & SOCK_PIPE)
  {
    if (s->flags & SOCK_PIPE_PARENT)
    {
      close(s->pipe_fd0[1]);
      close(s->pipe_fd1[0]);
    }
    else
    {
      close(s->pipe_fd1[1]);
      close(s->pipe_fd0[0]);
    }
  }
  else close(s->handle);

  if (s->info_remote) free(s->info_remote);
  fifobuf_free(s->fib_in);
  fifobuf_free(s->fib_out);
  
  if (s->flags & SOCK_LOG)
    log_put_opt(LOG_DEBUG, 0, "[Sock] Closed socket.");
  free(s);
}


SOCK *sock_accept(SOCK *s)
{
  SOCK *s_child;
  struct sockaddr_in remote_addr;
  int remote_addr_len, child_handle, _keepalive = 1;


  /* --- Prepare socket for incoming call --- */

  remote_addr_len = sizeof(remote_addr);

  /* --- Wait for one --- */

  child_handle = accept(s->handle, (struct sockaddr *) &remote_addr, &remote_addr_len);
  _sock_interrupted = 0;

  if (child_handle < 0)
  {
    if (s->flags & (SOCK_DEBUG | SOCK_LOG)) log_put_opt(LOG_DEBUG, 0, "[Sock] Status change on port %d, but no peer connecting.", s->port_bound);
    return(0);
  }

  if (s->flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Got connection on port %d.", s->port_bound);

  /* --- Create wrapper for spawned socket --- */

  s_child = malloc(sizeof(SOCK));
  memset(s_child, 0, sizeof(SOCK));

  /* --- Prepare buffers --- */

  s_child->fib_in = fifobuf_new(2, 32, 512);
  s_child->fib_out = fifobuf_new(2, 64, 512);

  /* Prepare other child values */

  s_child->parent = s;
  s_child->handle = child_handle;

  if (remote_addr_len > sizeof(s_child->addr_remote)) remote_addr_len = sizeof(s_child->addr_remote);
  memset(&(s_child->addr_remote), 0, sizeof(s_child->addr_remote));
  memcpy(&(s_child->addr_remote), &remote_addr, remote_addr_len);

  s_child->port_bound = 0;
  s_child->timeout_sec = s->timeout_sec;
  s_child->flags = SOCK_CONNECTED | s->flags;
  s_child->status = SOCK_STAT_OK;
  fcntl(s_child->handle, F_SETFL, 0);

  setsockopt(s_child->handle, SOL_SOCKET, SO_KEEPALIVE, (const void *) &_keepalive, sizeof(_keepalive));

  return(s_child);
}


/* Finish socket list with NULL, or else. */

SOCK *sock_wait(unsigned long timeout_usec, unsigned int options, SOCK *s, ...)
{
  va_list args;
  SOCK *s_v;
  SOCK *s_a[256];
  int i;

  s_a[0] = s;

  va_start(args, s);
  for (i = 1, s_v = va_arg(args, SOCK *); i < 256 && s_v;
       i++, s_v = va_arg(args, SOCK *))
  {
    s_a[i] = s_v;
  }
  va_end(args);

  s_a[i] = 0;
  return(sock_wait_arr(timeout_usec, options, s_a));
}


/* Finish socket list with NULL, or else. */

SOCK *sock_wait_arr(unsigned long timeout_usec, unsigned int options, SOCK **s)
{
  SOCK *r;
  fd_set socks_read, socks_write, socks_exception;
  struct timeval timeout;
  int debug, log, hd, hd_max, i, t;
  
  /* --- Setup ---- */

  FD_ZERO(&socks_read); FD_ZERO(&socks_write); FD_ZERO(&socks_exception);
  debug = log = hd_max = 0;

  for (i = 0, t = 0; s[i]; i++)
  {
    if (!(s[i]->flags & (SOCK_CONNECTED | SOCK_ACCEPT)) && s[i]->flags & (SOCK_DEBUG | SOCK_LOG))
    {
      log_put_opt(LOG_DEBUG, 0, "[Sock] Wait found useless socket in list. Ignored.");
      t++; continue;
    }
    
    if (s[i]->fib_in->enqueued)
    {
      if (s[i]->flags & (SOCK_LOG)) log_put_opt(LOG_DEBUG, 0, "[Sock] Wait returned buffered data.");
      return(s[i]);
    }
    
    debug |= s[i]->flags & SOCK_DEBUG; log |= s[i]->flags & SOCK_LOG;
    hd = SOCK_FDREAD(s[i]);
    FD_SET(hd, &socks_read); FD_SET(hd, &socks_exception);
    if (hd > hd_max) hd_max = hd;
  }

  if (!i || i == t) return(0);  /* No useful sockets */
  
  /* --- Wait --- */
  
  if (timeout_usec)
  {
    timeout.tv_sec = timeout_usec / 1000000;
    timeout.tv_usec = timeout_usec % 1000000;

    if ((t = select(hd_max + 1, &socks_read, 0, &socks_exception, &timeout)) < 1)
    {
      if (log) log_put_opt(LOG_DEBUG, 0, "[Sock] Wait (timed) returned null or error condition.");
      return(0);
    }
  }
  else if ((t = select(hd_max + 1, &socks_read, 0, &socks_exception, 0)) < 1)
  {
    if (debug || log) log_put_opt(LOG_DEBUG, 0, "[Sock] Wait (indefinite) returned null or error condition.");
    return(0);
  }

  /* --- Choose one of the ready sockets --- */

  for (i = 0; s[i]; i++)
  {
    hd = SOCK_FDREAD(s[i]);
    if (FD_ISSET(hd, &socks_read) || FD_ISSET(hd, &socks_exception))
    {
      if (s[i]->flags & SOCK_ACCEPT && !(s[i]->flags & SOCK_CONNECTED))
      {
        /* --- New connection --- */

        r = sock_accept(s[i]);
        if (r) return(r);
      }
      else return(s[i]);
    }
  }

  return(0);
}


int sock_connect_addr(SOCK *s, struct sockaddr_in *addr)
{
  if (s->flags & (SOCK_CONNECTED | SOCK_PIPE))
  {
    s->status = SOCK_STAT_PARADOX;
    return(0);
  }

  memcpy(&s->addr_remote, addr, sizeof(struct sockaddr_in));

  /* --- Try to connect the socket --- */

  /* FIXME: Add timeout alarm. */

  if (connect(s->handle, (struct sockaddr *) &s->addr_remote, sizeof(struct sockaddr_in)) < 0)
  {
    s->status = SOCK_STAT_CONNREFUSED;
    return(0);
  }

  s->flags |= SOCK_CONNECTED | SOCK_CALLER;
  s->status = SOCK_STAT_OK;
  return(1);
}


char hostname_allow[] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789.-_";

int sock_connect(SOCK *s, char *desthost_str, int destport)
{
  struct hostent destent_temp;
  struct hostent *destent;
  struct in_addr destdef_addr;


  if (s->flags & (SOCK_CONNECTED | SOCK_PIPE))
  {
    if (s->flags & (SOCK_DEBUG | SOCK_LOG)) log_put_opt(LOG_DEBUG, 0, "[Sock] Tried to connect a pipe or already connected socket.");
    s->status = SOCK_STAT_PARADOX;
    return(-1);
  }

  /* --- Get data on remote host --- */

  /* FIXME: Avoid name lookups whenever possible. */

  if (!desthost_str) return(SOCK_STAT_NONAME);
  if (!strlen(desthost_str)) return(SOCK_STAT_NONAME);
  if (strspn(desthost_str, hostname_allow) < strlen(desthost_str))
    return(SOCK_STAT_NONAME);

  destent = gethostbyname(desthost_str);
  if (!destent)
  {
#ifdef HAVE_INET_ATON
    if (!inet_aton(desthost_str, &destdef_addr))
#else
    unsigned long adr = inet_addr(desthost_str);
    if (adr != -1) destdef_addr.s_addr = adr;
    else
#endif
    {
      if (s->flags & (SOCK_DEBUG | SOCK_LOG))
        log_put_opt(LOG_DEBUG, 0, "[Sock] Name lookup failed for <%s>.",
                    desthost_str);
      s->status = SOCK_STAT_NONAME;
      return(0);
    }
    destent = &destent_temp;
    destent->h_name = desthost_str;
/*
    destent->h_addr_list = addrlist;
*/
    destent->h_addr = (char *) &destdef_addr;
    destent->h_length = sizeof(struct in_addr);
    destent->h_addrtype = AF_INET;
    destent->h_aliases = 0;
  }

  /* --- Fill in address of remote host --- */

  memset(&(s->addr_remote), 0, sizeof(s->addr_remote));
  s->addr_remote.sin_family = AF_INET;
  s->addr_remote.sin_port = htons(destport);
  if (destent->h_length > sizeof(s->addr_remote.sin_addr))
    destent->h_length = sizeof(s->addr_remote.sin_addr);
  memcpy(&(s->addr_remote.sin_addr), destent->h_addr, destent->h_length);

  /* --- Try to connect the socket --- */

  /* FIXME: Add timeout alarm. */
  
  if (connect(s->handle, (struct sockaddr *) &(s->addr_remote), sizeof(s->addr_remote)) < 0)
  {
    if (s->flags & (SOCK_DEBUG | SOCK_LOG)) log_put_opt(LOG_DEBUG, 0, "[Sock] Connection refused by <%s>.", desthost_str);
    s->status = SOCK_STAT_CONNREFUSED;
    return(0);
  }

  if (s->flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Connected to <%s>.", desthost_str);

  s->flags |= SOCK_CONNECTED | SOCK_CALLER;
  s->status = SOCK_STAT_OK;
  return(1);
}


int sock_reconnect(SOCK *s)
{
  if (s->flags & SOCK_PIPE)
  {
    if (s->flags & SOCK_DEBUG) log_put_opt(LOG_DEBUG, 0, "[Sock] Tried to reconnect to pipe.");
    s->status = SOCK_STAT_PARADOX;
    return(0);
  }

  if (s->flags & SOCK_CONNECTED)
  {
    if (s->flags & SOCK_LOG)
      log_put_opt(LOG_DEBUG, 0, "[Sock] Reconnect - disconnecting from <%s>.",
                  sock_get_remote_name_or_ip(s));
    sock_disconnect(s, 0);  /* Flash the line - off and back on */
  }

  /* FIXME: Return failure and set s->status to SOCK_STAT_PARADOX if no host
     specified in structure. */

  if (connect(s->handle, (struct sockaddr *) &(s->addr_remote), sizeof(s->addr_remote)) < 0)
  {
    if (s->flags & (SOCK_LOG | SOCK_DEBUG))
      log_put_opt(LOG_DEBUG, 0, "[Sock] Reconnection to <%s> refused.",
                  sock_get_remote_name_or_ip(s));
    s->status = SOCK_STAT_CONNREFUSED;
    return(0);
  }

  if (s->flags & SOCK_LOG)
    log_put_opt(LOG_DEBUG, 0, "[Sock] Reconnected to <%s>.",
                sock_get_remote_name_or_ip(s));

  s->flags |= SOCK_CONNECTED | SOCK_CALLER;
  s->status = SOCK_STAT_OK;
  return(1);
}


/* If disconnection fails (returns 0), consider SOCK struct closed & freed instead. */

int sock_disconnect(SOCK *s, unsigned long flags)
{
  struct sockaddr_in sock_addr;
  int _keepalive = 1;


  if (!(s->flags & SOCK_CONNECTED))
  {
    if (s->flags & (SOCK_DEBUG | SOCK_LOG))
      log_put_opt(LOG_DEBUG, 0, "[Sock] Tried to disconnect from pipe or unconnected socket.");
    s->status = SOCK_STAT_PARADOX;
    return(0);
  }

  if (!(flags & SOCK_IMMEDIATE)) sock_flush(s);
  close(s->handle);

  /* --- Open socket --- */

  s->handle = socket(AF_INET, SOCK_STREAM, 0);
  if (s->handle < 0)
  {
    if (s->info_remote) free(s->info_remote);
    fifobuf_free(s->fib_in);
    fifobuf_free(s->fib_out);
    free(s);
    return(0);
  }

  /* --- Initialize and bind socket --- */

  memset((char *) &sock_addr, 0, sizeof(sock_addr));
  sock_addr.sin_family = AF_INET;
  sock_addr.sin_addr.s_addr = htonl(INADDR_ANY);
  sock_addr.sin_port = htons(s->port_bound);

  if (bind(s->handle, (struct sockaddr *) &sock_addr, sizeof(sock_addr)) < 0)
  { close(s->handle); free(s); return(0); }

  if (s->info_remote)
  {
    free(s->info_remote);
    s->info_remote = 0;
  }

  /* --- Reset buffers --- */

  fifobuf_drop(s->fib_in, s->fib_in->enqueued);
  fifobuf_drop(s->fib_out, s->fib_out->enqueued);

  /* --- Set options --- */

  s->parent = 0;
  s->flags &= (unsigned long) ~(SOCK_CONNECTED | SOCK_CALLER);
  if (flags & SOCK_INTERNAL) s->status = SOCK_STAT_DISCONNECTED;
  else s->status = SOCK_STAT_OK;

  fcntl(s->handle, F_SETFL, 0);
  setsockopt(s->handle, SOL_SOCKET, SO_KEEPALIVE, (const void *) &_keepalive, sizeof(_keepalive));

  if (s->flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Disconnected.");
  return(1);
}


/* RETURNS: s->buf_in_bytes for pending input, 0 for no input or -1 for error */

int sock_poll(SOCK *s)
{
  if (!(s->flags & SOCK_CONNECTED))
  {
    s->status = SOCK_STAT_PARADOX;
    return(-1);
  }

  _sock_feed(s);

  if (s->fib_in->enqueued)
  {
    if (s->flags & SOCK_LOG) log_put_opt(LOG_DEBUG, 0, "[Sock] Poll returned %d bytes.", s->fib_in->enqueued);
    return(s->fib_in->enqueued);
  }
  
  return(0);
}

void sock_set_timeout(SOCK *s, int timeout_sec)
{
  s->timeout_sec = timeout_sec;
}


void sock_set_options(SOCK *s, unsigned long flags)
{
  if (flags & SOCK_SECURE) s->flags |= SOCK_SECURE;
  else s->flags &= ~SOCK_SECURE;
}


/* Returns static string */
const char *sock_get_remote_name(SOCK *s)
{
  struct hostent *h;

  if (s->flags & SOCK_PIPE)
  {
    s->status = SOCK_STAT_PARADOX;
    return(0);
  }

  if (!s->info_remote)
  {
    h = gethostbyaddr((char *) &(s->addr_remote.sin_addr.s_addr),
                      sizeof(s->addr_remote.sin_addr.s_addr), AF_INET);
    if (h)
    {
      s->info_remote = malloc(sizeof(struct hostent));
      memcpy(s->info_remote, h, sizeof(struct hostent));
    }
  }

  if (!s->info_remote) return(0);
  return(s->info_remote->h_name);
}


/* Returns static string */
const char *sock_get_remote_ip(SOCK *s)
{
  if (s->flags & SOCK_PIPE)
  {
    s->status = SOCK_STAT_PARADOX;
    return(0);
  }

  sprintf(s->ip_remote, "%d.%d.%d.%d",
          (s->addr_remote.sin_addr.s_addr  & 255),
          (s->addr_remote.sin_addr.s_addr >> 8) & 255,
          (s->addr_remote.sin_addr.s_addr >> 16) & 255,
          s->addr_remote.sin_addr.s_addr >> 24);

  return(s->ip_remote);
}


const char *sock_get_remote_name_or_ip(SOCK *s)
{
  const char *remote_str;

  if (!(remote_str = sock_get_remote_name(s)))
    return(sock_get_remote_ip(s));

  return(remote_str);
}
