/**********************************************
*    eb.c
*    Ruby Extension Library for EB
*    $Id: eb.c,v 1.6 2001/06/22 17:43:54 nisikawa Exp nisikawa $
*    Author: Nishikawa,Yasuhiro <nyasu@osk.3web.ne.jp>
*    http://www.threeweb.ad.jp/~nyasu/software/ruby.html
*    Licence: GPL
*    Copyleft
************************************************/


#include "ruby.h"

#include <eb/eb.h>
#include <eb/text.h>
#include <eb/error.h>

#include <stdlib.h>

#if	EB_VERSION_MAJOR < 3

#define	EB_NUMBER_OF_HOOKS	EB_NUM_HOOKS
#define	EB_SUCCESS			0

#define	eb_initialize_library()
#define	eb_finalize_library()
#define	eb_initialize_book		eb_initialize
#define	eb_finalize_book		eb_clear
#define	eb_finalize_appendix	eb_clear_appendix
#define	eb_have_stop_code		eb_have_stopcode                     
#define	eb_error_message		eb_error_message2
#define	eb_seek_text			eb_seek


static inline
EB_Error_Code eb3_bind EB_P((
	EB_Book *book,
	const char *path))
{
	int err;
	
	err = eb_bind(book,path);
	return (err == -1) ? !EB_SUCCESS : EB_SUCCESS;
}

EB_Error_Code eb3_set_subbook EB_P((
	EB_Book *book,
	EB_Subbook_Code subbook_code))
{
	int	err;
	
	err = eb_set_subbook(book,subbook_code);
	return (err==-1) ? !EB_SUCCESS : EB_SUCCESS;
}

static inline
EB_Error_Code eb3_subbook_count EB_P((
	EB_Book *book,
	int *subbook_count))
{
	*subbook_count=eb_subbook_count(book);

	return eb_error;
}

static inline
EB_Error_Code eb3_read_text EB_P((
	EB_Book *book, 
	EB_Appendix *appendix, 
	EB_Hookset *hookset,
    void *container,
    size_t text_max_length, 
    char *text, 
    ssize_t *text_length))
{
	*text_length = eb_text(book,appendix,hookset,text,text_max_length);

	return eb_error;
}

static inline
EB_Error_Code eb3_read_heading EB_P((
	EB_Book *book,
	EB_Appendix *appendix,
	EB_Hookset *hookset,
    void *container,
    size_t text_max_length,
    char *text,
    ssize_t *text_length))
{
	*text_length = eb_heading(book,appendix,hookset,text,text_max_length);
	return eb_error;
}

static inline
EB_Error_Code eb3_hit_list EB_P((
	EB_Book *book,
	int max_hit_count,
	EB_Hit *hit_list,
	int *hit_count))
{
	*hit_count = eb_hit_list(book, hit_list, max_hit_count);
	return eb_error;
}

static inline
EB_Error_Code eb3_disc_type EB_P((
	EB_Book *book,
	EB_Disc_Code *disc_code))
{
	*disc_code =  eb_disc_type(book);
	return eb_error;
}

EB_Error_Code eb3_path EB_P((
    EB_Book *book,
    char *path))
{
	strcpy(path,eb_path(book));
	
	return eb_error;
}

EB_Error_Code eb3_character_code EB_P((
	EB_Book *book,
	EB_Character_Code *character_code))
{
	*character_code = eb_character_code(book);

	return eb_error;
}

EB_Error_Code eb3_subbook_list EB_P((
    EB_Book *book,
    EB_Subbook_Code *subbook_list,
    int *subbook_count))
{
	*subbook_count = eb_subbook_list(book, subbook_list);
	
	return eb_error;
}

EB_Error_Code eb3_subbook_title EB_P((
	EB_Book *book,
	char *title))
{
	strcpy(title,eb_subbook_title(book));

	return eb_error;
}

EB_Error_Code eb3_subbook_title2 EB_P((
	EB_Book *book,
	EB_Subbook_Code subbook_code,
	char *title))
{
	strcpy(title,eb_subbook_title2(book,subbook_code));

	return eb_error;
}

EB_Error_Code eb3_subbook_directory EB_P((
	EB_Book *book,
	char *directory))
{
	strcpy(directory,eb_subbook_directory(book));

	return eb_error;
}

EB_Error_Code eb3_subbook_directory2 EB_P((
	EB_Book *book,
	EB_Subbook_Code subbook_code,
    char *directory))
{
	strcpy(directory,eb_subbook_directory2(book,subbook_code));

	return eb_error;
}

EB_Error_Code eb3_subbook EB_P((
    EB_Book *book,
    EB_Subbook_Code *subbook_code))
{
	*subbook_code = eb_subbook(book);

	return eb_error;
}

#else

static	EB_Error_Code	eb_error;

#define	eb3_read_text			eb_read_text
#define	eb3_read_heading		eb_read_heading
#define	eb3_hit_list			eb_hit_list
#define	eb3_disc_type			eb_disc_type
#define	eb3_path				eb_path
#define	eb3_character_code		eb_character_code	
#define	eb3_subbook_list		eb_subbook_list
#define	eb3_subbook_title		eb_subbook_title
#define	eb3_subbook_title2		eb_subbook_title2
#define	eb3_subbook_directory	eb_subbook_directory
#define	eb3_subbook_directory2	eb_subbook_directory2
#define	eb3_subbook				eb_subbook
#define	eb3_bind				eb_bind

static inline
EB_Error_Code eb3_subbook_count EB_P((
	EB_Book *book,
	int *subbook_count))
{
	EB_Subbook_Code slist[EB_MAX_SUBBOOKS];
	eb_error=eb_subbook_list(book,slist,subbook_count);
	
	return eb_error;
}

EB_Error_Code eb3_set_subbook EB_P((
	EB_Book *book,
	EB_Subbook_Code subbook_code))
{
	eb_error = eb_set_subbook(book,subbook_code);
	return eb_error;
}

#endif

VALUE mEB;
VALUE cEBook;
VALUE cEBCancel;
VALUE cEBPosition;

VALUE reb_text_hookset;
VALUE reb_appendix;
VALUE hook_list;
ID id_call;

EB_Hook reb_text_hooks[EB_NUMBER_OF_HOOKS];

#define MAX_HITS 50
#define MAX_STRLEN 65530   /* (^^); */

int
text_hook (book, appendix, buff, code, argc, argv)
     EB_Book *book;
     EB_Appendix *appendix;
     char *buff;
     EB_Hook_Code code;
     int argc;
     const int *argv;
{
  VALUE func, rb_buff, ret_buff, rb_argv;
  int idx, appendix_bound;
    
  appendix_bound = eb_is_appendix_bound(appendix);

  switch (code) {
  case EB_HOOK_NARROW_FONT:
    if ((!appendix_bound)
	|| (eb_narrow_alt_character_text (appendix, argv[0], buff) >= 0)) {
      return 0;
    }
    break;
  case EB_HOOK_WIDE_FONT:
    if ((!appendix_bound)
	|| (eb_wide_alt_character_text (appendix, argv[0], buff) >= 0)) {
      return 0;
    }
    break;
  default:
    break;
  }

  func = rb_ary_entry(hook_list, code);

  rb_buff = rb_str_new2(buff);
  rb_argv = rb_ary_new2(argc);
  for (idx = 0; idx < argc; idx++)
    rb_ary_store(rb_argv, idx, INT2FIX(argv[idx]));

  ret_buff = rb_funcall(func, id_call, 2, rb_buff, rb_argv);

  if (ret_buff != Qnil)
    strcpy(buff, STR2CSTR(ret_buff));

  return 0;
}

/******************************
**  EB Book
*/

static VALUE
reb_initialize(VALUE klass){
	VALUE robj;
	EB_Book* eb;
	EB_Appendix *appendix;
	EB_Hookset *text_hookset;

	robj=Data_Make_Struct(klass,EB_Book,0,eb_finalize_book,eb);
	eb_initialize_book(eb);

	reb_appendix=Data_Make_Struct(cEBook,EB_Appendix,
				      0,free,appendix);
	reb_text_hookset=Data_Make_Struct(cEBook,EB_Hookset,
					  0,free,text_hookset);
	eb_initialize_hookset(text_hookset);

	eb_initialize_appendix(appendix);

	hook_list = rb_ary_new2(EB_NUMBER_OF_HOOKS);

	return robj;
}

static VALUE
rEB_error(VALUE module){
	return INT2NUM(eb_error);
}
static VALUE
rEB_errormsg(VALUE module){
	return rb_str_new2( eb_error_message(eb_error) );
}

static VALUE
reb_bind(VALUE obj,VALUE path){
	EB_Book* eb;
	int r;

	Data_Get_Struct(obj,EB_Book,eb);
	r=eb3_bind(eb,STR2CSTR(path));
	if(r!=EB_SUCCESS) {
		rb_raise(rb_eRuntimeError,"bind failed");
	}
	return obj;
}


static VALUE
reb_disktype(VALUE obj){
	EB_Book* eb;
	EB_Disc_Code r;

	Data_Get_Struct(obj,EB_Book,eb);

	eb_error =eb3_disc_type(eb,&r);
	switch(r){
		case EB_DISC_EB:
		  return rb_str_new2("EB/EBG/EBXA/EBXA-C/S-EBXA");
		  break;
		case EB_DISC_EPWING:
		  return rb_str_new2("EPWING");
		  break;
	}
	return rb_str_new2("Unknown");
}

static VALUE
reb_suspend(VALUE obj){
	EB_Book* eb;

	Data_Get_Struct(obj,EB_Book,eb);
    eb_suspend(eb);
    return Qnil;
}

static VALUE
reb_isbound(VALUE obj){
	EB_Book* eb;
    int r;
    
	Data_Get_Struct(obj,EB_Book,eb);
    r=eb_is_bound(eb);
    
    return (r)? Qtrue: Qfalse;
}

static VALUE
reb_path(VALUE obj){
	EB_Book* eb;
    char r[1024];				/*ͤϤޤȻפ*/

	Data_Get_Struct(obj,EB_Book,eb);
    eb_error=eb3_path(eb,r);

    return rb_str_new2(r);
}

static VALUE
reb_charcode(VALUE obj){
	EB_Book* eb;
    EB_Character_Code r;
	
	Data_Get_Struct(obj,EB_Book,eb);
    eb_error=eb3_character_code(eb,&r);

	switch(r){
		case EB_CHARCODE_ISO8859_1:
		  return rb_str_new2("ISO8859_1");
		  break;
		case EB_CHARCODE_JISX0208:
		  return rb_str_new2("JISX0208");
		  break;
	}
    return Qnil;
}

static VALUE
reb_subbookcount(VALUE obj){
	EB_Book* eb;
    int r;

	Data_Get_Struct(obj,EB_Book,eb);
	eb3_subbook_count(eb,&r);
	return INT2NUM(r);
}

static VALUE
reb_subbooklist(VALUE obj){
	EB_Book* eb;
	EB_Subbook_Code slist[EB_MAX_SUBBOOKS];
    int i,r;
	VALUE robj;

	Data_Get_Struct(obj,EB_Book,eb);
	eb_error=eb3_subbook_list(eb,slist,&r);
	robj=rb_ary_new2(r);
	for(i=0;i<r;i++) rb_ary_push(robj,INT2NUM(slist[i]));
	return robj;
}

static VALUE
reb_subbooktitle(int argc, VALUE* argv, VALUE obj){
	EB_Book* eb;
	EB_Subbook_Code sc;
	
    char r[1024];				/*ͤϤޤȻפ*/

	Data_Get_Struct(obj,EB_Book,eb);
	eb_error = (argc==0)? 
	      eb3_subbook_title(eb,r) : eb3_subbook_title2(eb,NUM2INT(argv[0]),r);
	return rb_str_new2(r);
}

static VALUE
reb_subbookdirectory(int argc, VALUE* argv, VALUE obj){
	EB_Book* eb;
	EB_Subbook_Code sc;
	
    char r[1024];				/*ͤϤޤȻפ*/

	Data_Get_Struct(obj,EB_Book,eb);
	eb_error = (argc==0)? 
	      eb3_subbook_directory(eb,r) : eb3_subbook_directory2(eb,NUM2INT(argv[0]),r);
	return rb_str_new2(r);
}

static VALUE
reb_setsubbook(VALUE obj,VALUE sbook){
	EB_Book* eb;
	int r;

	Data_Get_Struct(obj,EB_Book,eb);
	r=eb_set_subbook(eb,NUM2INT(sbook));

#if	EB_VERSION_MAJOR < 3
	return (r==-1)? Qfalse : obj;
#else
	return (r!=EB_SUCCESS)? Qfalse : obj;
#endif
}
static VALUE
reb_getsubbook(VALUE obj,VALUE sbook){
	EB_Book* eb;
	int r;

	Data_Get_Struct(obj,EB_Book,eb);
	eb_error=eb3_subbook(eb,&r);
	
	return INT2NUM(r);
}

static VALUE
reb_unsetsubbook(VALUE obj){
	EB_Book* eb;
	int r;

	Data_Get_Struct(obj,EB_Book,eb);
	eb_unset_subbook(eb);
	
	return obj;
}

static VALUE
reb_haveexactsearch(VALUE obj){
	EB_Book* eb;
    int r;
	Data_Get_Struct(obj,EB_Book,eb);
    r=eb_have_exactword_search(eb);
    if(!r && eb_error==EB_ERR_NO_CUR_SUB){
    	rb_raise(rb_eRuntimeError,eb_error_message(eb_error));
    }
    return (r)? Qtrue : Qfalse;
}
static VALUE
reb_havewordsearch(VALUE obj){
	EB_Book* eb;
    int r;
	Data_Get_Struct(obj,EB_Book,eb);
    r=eb_have_word_search(eb);
    if(!r && eb_error==EB_ERR_NO_CUR_SUB){
    	rb_raise(rb_eRuntimeError,eb_error_message(eb_error));
    }
    return (r)? Qtrue: Qfalse;
}
static VALUE
reb_haveendsearch(VALUE obj){
	EB_Book* eb;
    int r;
	Data_Get_Struct(obj,EB_Book,eb);
    r=eb_have_endword_search(eb);
    if(!r && eb_error==EB_ERR_NO_CUR_SUB){
    	rb_raise(rb_eRuntimeError,eb_error_message(eb_error));
    }
    return (r)? Qtrue: Qfalse;
}



/******************************
**  for result
*/

VALUE get_item(EB_Book* eb,EB_Hit* hit){
	VALUE item;
	char desc[MAX_STRLEN + 1];
	int r;
	int len;

	item=rb_ary_new2(2);

	if(eb_seek_text(eb,&(hit->heading))<0){
		rb_raise(rb_eRuntimeError,"fail seeking");
		return Qfalse;
	}
	eb_error = eb3_read_heading(eb, NULL, NULL, NULL, MAX_STRLEN, desc, &len);
	if(len<0){
		rb_raise(rb_eRuntimeError,"fail fetching heading");
		return Qfalse;
	}
	rb_ary_push( item,rb_str_new(desc,len) );

	if(eb_seek_text(eb,&(hit->text))<0){
		rb_raise(rb_eRuntimeError,"fail seeking(text)");
		return Qfalse;
	}
	eb_error = eb3_read_text(eb, NULL, NULL, NULL, MAX_STRLEN, desc, &len);
	if(len<0){
		rb_raise(rb_eRuntimeError,"fail fetching");
		return Qfalse;
	}
	rb_ary_push( item,rb_str_new(desc,len) );

	return item;
}


VALUE hitmaker(EB_Book* eb,unsigned int max,int flag){
	int hitpushed,hitcount;
	EB_Hit hits[MAX_HITS];
	VALUE robj,item,can;
	int i;

	hitpushed = 0;
	
	robj=rb_ary_new();

	while(1){
		eb_error=eb3_hit_list(eb, MAX_HITS, hits, &hitcount);
		if(hitcount==0) break;  /* return */
		if(hitcount<0){
			rb_raise(rb_eRuntimeError,"fail getting list");
			return Qfalse;
		}
		
		for(i=0;i<hitcount;i++){
			item=get_item(eb,&hits[i]);
			if(flag==0){
				rb_ary_push(robj,item);
			} else {
				can=rb_yield(item);
				if(rb_obj_id(can)==rb_obj_id(cEBCancel)){
					return robj;
				}
			}
			hitpushed++;
			if(hitpushed>=max) return robj;
		}
	}
	
	
	return robj;
}


#define PRESEARCH \
	EB_Book* eb;\
	char* word;\
	int max;\
	int r;\
\
	if(argc<1){\
		rb_raise(rb_eArgError,"missing searchstring");\
		return Qfalse;\
	}\
\
	Data_Get_Struct(obj,EB_Book,eb);\
\
	word=STR2CSTR(argv[0]);    max=(argc>1)? NUM2INT(argv[1]) : -1;

#define POSTSEARCH \
	if(r==-1){\
		rb_raise(rb_eRuntimeError,"fail searching");\
		return Qfalse;\
	}\
		 /* tricky max: signed<->unsigned -1 */ \
	return hitmaker( eb, (unsigned)max, (rb_iterator_p())? 1:0 );


static VALUE
reb_searchword(int argc, VALUE*argv, VALUE obj){
	PRESEARCH
	r=eb_search_word(eb,word);
	POSTSEARCH
}

static VALUE
reb_exactsearchword(int argc, VALUE*argv, VALUE obj){
	PRESEARCH
	r=eb_search_exactword(eb,word);
	POSTSEARCH
}
static VALUE
reb_endsearchword(int argc, VALUE*argv, VALUE obj){
	PRESEARCH
	r=eb_search_endword(eb,word);
	POSTSEARCH
}


/*   Thanks for Kuroda-san  */

#define PRESEARCH2 \
	int hitcount,i,len;\
	int hitpushed; \
	VALUE robj,item,can;\
	EB_Hit hits[MAX_HITS];\
	char *desc; \
	char descbuf1[MAX_STRLEN + 1]; \
/*** this 3 lines necessary? (1/4) eblook do like this ***/\
	char descbuf2[MAX_STRLEN + 1]; \
	char *prevdesc; \
	int prevpage,prevoffset; \
	EB_Position *ebpos;\
	PRESEARCH\


#define POSTSEARCH2 \
	if(r==-1){ \
		rb_raise(rb_eRuntimeError,"fail searching");\
		return Qfalse;\
	}\
	if(rb_iterator_p()){robj=(VALUE)NULL;} else {robj=rb_ary_new();} \
\
	desc = descbuf1; \
/*** this 2 lines necessary? (2/4) eblook do like this ***/\
	prevdesc = descbuf2; *prevdesc = '\0'; \
	prevpage = 0; prevoffset = 0; \
\
	hitpushed=0; \
	while (1) {\
	  eb_error = eb3_hit_list(eb, MAX_HITS, hits, &hitcount);\
	  if(hitcount==0) break;  /* return */\
	  if(hitcount<0){\
	    rb_raise(rb_eRuntimeError,"fail getting list");\
	    return Qfalse;\
	  }\
\
	  for(i=0;i<hitcount;i++){\
	    if(eb_seek_text(eb,&(hits[i].heading))<0){\
	      rb_raise(rb_eRuntimeError,"fail seeking");\
	      return Qfalse;\
	    }\
		eb_error = eb3_read_heading(eb, NULL, NULL, NULL, MAX_STRLEN, desc, &len);	\
	    if(len<0){\
	      rb_raise(rb_eRuntimeError,"fail fetching heading"); \
	      return Qfalse;\
	    } \
\
/*** this 4 lines necessary? (3/4) eblook do like this ***/\
	    if (prevpage == hits[i].text.page &&  \
		prevoffset == hits[i].text.offset &&  \
		strcmp(desc, prevdesc) == 0)          \
		continue;                             \
\
		item = rb_ary_new2(2); \
		rb_ary_push( item, Data_Make_Struct(cEBPosition,EB_Position,0,free,ebpos));  \
		rb_ary_push( item, rb_str_new(desc,len) ); \
	    ebpos->page = hits[i].text.page; \
	    ebpos->offset = hits[i].text.offset; \
\
		if(robj){   /* non-iterator */  \
			rb_ary_push(robj,item); \
		}else{      /* iterator */  \
			can=rb_yield(item); \
			if(rb_obj_id(can)==rb_obj_id(cEBCancel)){ \
				return robj; \
			}\
		}\
		hitpushed++; \
		if(hitpushed>=(unsigned)max) break;  /* tricky max: -1<->maxinteger*/\
\
/*** this 6 lines necessary? (4/4) eblook do like this ***/\
	    if (desc == descbuf1) {                  \
	      desc = descbuf2; prevdesc = descbuf1;  \
	    } else {                                 \
	      desc = descbuf1; prevdesc = descbuf2;  \
	    }                                        \
	    prevpage = hits[i].text.page; prevoffset = hits[i].text.offset; \
	  }\
	} \
	return (robj)? robj : Qnil; \


static VALUE
reb_exactsearchword2(int argc, VALUE*argv, VALUE obj){
	PRESEARCH2
	r=eb_search_exactword(eb,word);
	POSTSEARCH2
}

static VALUE
reb_searchword2(int argc, VALUE*argv, VALUE obj){
	PRESEARCH2
	r=eb_search_word(eb,word);
	POSTSEARCH2
}
static VALUE
reb_endsearchword2(int argc, VALUE*argv, VALUE obj){
	PRESEARCH2
	r=eb_search_endword(eb,word);
	POSTSEARCH2
}


static VALUE
reb_content(VALUE obj, VALUE position){
  char desc[MAX_STRLEN + 1];
  EB_Position pos, *ppos;
  VALUE robj;
  EB_Book* eb;
  EB_Hookset *text_hookset;
  EB_Appendix *appendix;
  int len;
  
  Data_Get_Struct(position,EB_Position,ppos);
  Data_Get_Struct(obj,EB_Book,eb);

  Data_Get_Struct(reb_text_hookset,EB_Hookset,text_hookset);
  Data_Get_Struct(reb_appendix,EB_Appendix,appendix);

  if (eb_seek_text(eb, ppos)<0) {
    rb_raise(rb_eRuntimeError,"fail seeking(text)");
    return Qfalse;
  }

  eb_error = eb3_read_text(eb, appendix, text_hookset, NULL, MAX_STRLEN, desc, &len);
  if (len < 0) {
    rb_raise(rb_eRuntimeError,"fail fetching");
    return Qfalse;
  }

  return rb_str_new2(desc);
}

static VALUE
reb_register_hook(int argc, VALUE *argv, VALUE self)
{	
  VALUE proc = Qnil;
  int hook_type;
  void *proc_tmp = NULL;
  EB_Hookset *text_hookset;

  switch (argc) {
  case 1:
    proc = rb_f_lambda();
    break;
  case 2:
    proc = argv[1];
    break;
  default:
    rb_raise(rb_eArgError, "wrong # of arguments");
    break;
  }
	
  hook_type = FIX2INT(argv[0]);
  rb_ary_store(hook_list, hook_type, proc);

  reb_text_hooks[hook_type].code = hook_type;
  reb_text_hooks[hook_type].function = text_hook;

  Data_Get_Struct(reb_text_hookset,EB_Hookset,text_hookset);

  eb_set_hooks (text_hookset, reb_text_hooks);
  
  return Qnil;
}

static VALUE
reb_pos_initialize(argc, argv, klass)
    int argc;
  VALUE *argv;
  VALUE klass;
{
  VALUE robj,page,offset;
  EB_Position* ebpos;

  robj=Data_Make_Struct(klass,EB_Position,0,free,ebpos);
  if (rb_scan_args(argc, argv, "02", &page, &offset) > 0) {
    Check_Type(page, T_FIXNUM);
    Check_Type(offset, T_FIXNUM);
    ebpos->page = FIX2INT(page);
    ebpos->offset = FIX2INT(offset);
  }
  return robj;
}

static VALUE
reb_pos_get_page(VALUE obj){
  EB_Position *pos;

  Data_Get_Struct(obj,EB_Position,pos);

  return INT2FIX(pos->page);
}

static VALUE
reb_pos_get_offset(VALUE obj){
  EB_Position *pos;

  Data_Get_Struct(obj,EB_Position,pos);

  return INT2FIX(pos->offset);
}

static VALUE
reb_pos_set_page(VALUE obj, VALUE spage){
  EB_Position *pos;

  Data_Get_Struct(obj,EB_Position,pos);
  pos->page = FIX2INT(spage);

  return obj;
}

static VALUE
reb_pos_set_offset(VALUE obj, VALUE soffset){
  EB_Position *pos;

  Data_Get_Struct(obj,EB_Position,pos);
  pos->offset = FIX2INT(soffset);

  return obj;
}

static VALUE
reb_mod_initialize(VALUE obj){
    eb_initialize_library();
    return obj;
}

static VALUE
reb_mod_finalize(VALUE obj){
/*	printf("mod_finalize\n"); */
	eb_finalize_library();
}

/******************************
**   Init
*/

void
Init_eb(){
	id_call = rb_intern("call");

	mEB = rb_define_module("EB");
	cEBook = rb_define_class_under(mEB,"Book",rb_cObject);
	cEBCancel = rb_define_class_under(mEB,"Cancel",rb_cObject);
	cEBPosition = rb_define_class_under(mEB,"Position",rb_cObject);

	rb_define_singleton_method(mEB,"errorcode",rEB_error,0);
	rb_define_singleton_method(mEB,"error_message",rEB_errormsg,0);

	rb_define_singleton_method(cEBook,"new",reb_initialize,0);
	rb_define_method(cEBook,"bind",reb_bind,1);
	rb_define_method(cEBook,"disctype",reb_disktype,0);
	rb_define_method(cEBook,"disktype",reb_disktype,0);
	rb_define_method(cEBook,"suspend",reb_suspend,0);
	rb_define_method(cEBook,"bound?",reb_isbound,0);
	rb_define_method(cEBook,"path",reb_path,0);
	rb_define_method(cEBook,"charcode",reb_charcode,0);

	rb_define_method(cEBook,"subbook_count",reb_subbookcount,0);
	rb_define_method(cEBook,"subbook_list",reb_subbooklist,0);
	rb_define_method(cEBook,"title",reb_subbooktitle,-1);
	rb_define_method(cEBook,"directory",reb_subbookdirectory,-1);

	rb_define_method(cEBook,"set",reb_setsubbook,1);
	rb_define_method(cEBook,"subbook=",reb_setsubbook,1);
	rb_define_method(cEBook,"subbook",reb_getsubbook,0);
	rb_define_method(cEBook,"unset",reb_unsetsubbook,0);

	rb_define_method(cEBook,"search",reb_searchword,-1);
	rb_define_method(cEBook,"exactsearch",reb_exactsearchword,-1);
	rb_define_method(cEBook,"endsearch",reb_endsearchword,-1);

	rb_define_method(cEBook,"search2",reb_searchword2,-1);
	rb_define_method(cEBook,"exactsearch2",reb_exactsearchword2,-1);
	rb_define_method(cEBook,"endsearch2",reb_endsearchword2,-1);

	rb_define_method(cEBook,"content",reb_content,1);

	rb_define_method(cEBook,"search_available?",reb_havewordsearch,0);
	rb_define_method(cEBook,"exactsearch_available?",reb_haveexactsearch,0);
	rb_define_method(cEBook,"endsearch_available?",reb_haveendsearch,0);

	rb_define_method(cEBook,"register_hook", reb_register_hook, -1);

#if	EB_VERSION_MAJOR < 3
	rb_define_const(mEB, "HOOK_BEGIN_NARROW", INT2FIX(EB_HOOK_BEGIN_NARROW));
	rb_define_const(mEB, "HOOK_NARROW", INT2FIX(EB_HOOK_NARROW));
	rb_define_const(mEB, "HOOK_END_NARROW", INT2FIX(EB_HOOK_END_NARROW));
	rb_define_const(mEB, "HOOK_BEGIN_SUBSCRIPT", INT2FIX(EB_HOOK_BEGIN_SUBSCRIPT));
	rb_define_const(mEB, "HOOK_SUBSCRIPT", INT2FIX(EB_HOOK_SUBSCRIPT));
	rb_define_const(mEB, "HOOK_END_SUBSCRIPT", INT2FIX(EB_HOOK_END_SUBSCRIPT));
	rb_define_const(mEB, "HOOK_SET_INDENT", INT2FIX(EB_HOOK_SET_INDENT));
	rb_define_const(mEB, "HOOK_NEWLINE", INT2FIX(EB_HOOK_NEWLINE));
	rb_define_const(mEB, "HOOK_BEGIN_SUPERSCRIPT", INT2FIX(EB_HOOK_BEGIN_SUPERSCRIPT));
	rb_define_const(mEB, "HOOK_SUPERSCRIPT", INT2FIX(EB_HOOK_SUPERSCRIPT));
	rb_define_const(mEB, "HOOK_END_SUPERSCRIPT", INT2FIX(EB_HOOK_END_SUPERSCRIPT));
	rb_define_const(mEB, "HOOK_BEGIN_TABLE", INT2FIX(EB_HOOK_BEGIN_TABLE));
	rb_define_const(mEB, "HOOK_END_TABLE", INT2FIX(EB_HOOK_END_TABLE));
	rb_define_const(mEB, "HOOK_BEGIN_NO_NEWLINE", INT2FIX(EB_HOOK_BEGIN_NO_NEWLINE));
	rb_define_const(mEB, "HOOK_NO_NEWLINE", INT2FIX(EB_HOOK_NO_NEWLINE));
	rb_define_const(mEB, "HOOK_END_NO_NEWLINE", INT2FIX(EB_HOOK_END_NO_NEWLINE));
	rb_define_const(mEB, "HOOK_BEGIN_EMPHASIS", INT2FIX(EB_HOOK_BEGIN_EMPHASIS));
	rb_define_const(mEB, "HOOK_EMPHASIS", INT2FIX(EB_HOOK_EMPHASIS));
	rb_define_const(mEB, "HOOK_END_EMPHASIS", INT2FIX(EB_HOOK_END_EMPHASIS));
	rb_define_const(mEB, "HOOK_BEGIN_PICTURE", INT2FIX(EB_HOOK_BEGIN_PICTURE));
	rb_define_const(mEB, "HOOK_PICTURE", INT2FIX(EB_HOOK_PICTURE));
	rb_define_const(mEB, "HOOK_END_PICTURE", INT2FIX(EB_HOOK_END_PICTURE));
	rb_define_const(mEB, "HOOK_BEGIN_MENU", INT2FIX(EB_HOOK_BEGIN_MENU));
	rb_define_const(mEB, "HOOK_MENU", INT2FIX(EB_HOOK_MENU));
	rb_define_const(mEB, "HOOK_END_MENU", INT2FIX(EB_HOOK_END_MENU));
	rb_define_const(mEB, "HOOK_BEGIN_SOUND", INT2FIX(EB_HOOK_BEGIN_SOUND));
	rb_define_const(mEB, "HOOK_SOUND", INT2FIX(EB_HOOK_SOUND));
	rb_define_const(mEB, "HOOK_END_SOUND", INT2FIX(EB_HOOK_END_SOUND));
	rb_define_const(mEB, "HOOK_BEGIN_REFERENCE", INT2FIX(EB_HOOK_BEGIN_REFERENCE));
	rb_define_const(mEB, "HOOK_REFERENCE", INT2FIX(EB_HOOK_REFERENCE));
	rb_define_const(mEB, "HOOK_END_REFERENCE", INT2FIX(EB_HOOK_END_REFERENCE));
	rb_define_const(mEB, "HOOK_BEGIN_KEYWORD", INT2FIX(EB_HOOK_BEGIN_KEYWORD));
	rb_define_const(mEB, "HOOK_KEYWORD", INT2FIX(EB_HOOK_KEYWORD));
	rb_define_const(mEB, "HOOK_END_KEYWORD", INT2FIX(EB_HOOK_END_KEYWORD));
	rb_define_const(mEB, "HOOK_ISO8859_1", INT2FIX(EB_HOOK_ISO8859_1));
	rb_define_const(mEB, "HOOK_NARROW_JISX0208", INT2FIX(EB_HOOK_NARROW_JISX0208));
	rb_define_const(mEB, "HOOK_WIDE_JISX0208", INT2FIX(EB_HOOK_WIDE_JISX0208));
	rb_define_const(mEB, "HOOK_NARROW_FONT", INT2FIX(EB_HOOK_NARROW_FONT));
	rb_define_const(mEB, "HOOK_WIDE_FONT", INT2FIX(EB_HOOK_WIDE_FONT));
	rb_define_const(mEB, "HOOK_INITIALIZE", INT2FIX(EB_HOOK_INITIALIZE));
	rb_define_const(mEB, "HOOK_STOPCODE", INT2FIX(EB_HOOK_STOPCODE));
	rb_define_const(mEB, "HOOK_GB2312", INT2FIX(EB_HOOK_GB2312));
#else
	rb_define_const(mEB, "HOOK_INITIALIZE", INT2FIX(EB_HOOK_INITIALIZE));
	rb_define_const(mEB, "HOOK_BEGIN_NARROW", INT2FIX(EB_HOOK_BEGIN_NARROW));
	rb_define_const(mEB, "HOOK_END_NARROW", INT2FIX(EB_HOOK_END_NARROW));
	rb_define_const(mEB, "HOOK_BEGIN_SUBSCRIPT", INT2FIX(EB_HOOK_BEGIN_SUBSCRIPT));
	rb_define_const(mEB, "HOOK_END_SUBSCRIPT", INT2FIX(EB_HOOK_END_SUBSCRIPT));
	rb_define_const(mEB, "HOOK_SET_INDENT", INT2FIX(EB_HOOK_SET_INDENT));
	rb_define_const(mEB, "HOOK_NEWLINE", INT2FIX(EB_HOOK_NEWLINE));
	rb_define_const(mEB, "HOOK_BEGIN_SUPERSCRIPT", INT2FIX(EB_HOOK_BEGIN_SUPERSCRIPT));
	rb_define_const(mEB, "HOOK_END_SUPERSCRIPT", INT2FIX(EB_HOOK_END_SUPERSCRIPT));
	rb_define_const(mEB, "HOOK_BEGIN_NO_NEWLINE", INT2FIX(EB_HOOK_BEGIN_NO_NEWLINE));
	rb_define_const(mEB, "HOOK_END_NO_NEWLINE", INT2FIX(EB_HOOK_END_NO_NEWLINE));
	rb_define_const(mEB, "HOOK_BEGIN_EMPHASIS", INT2FIX(EB_HOOK_BEGIN_EMPHASIS));
	rb_define_const(mEB, "HOOK_END_EMPHASIS", INT2FIX(EB_HOOK_END_EMPHASIS));
	rb_define_const(mEB, "HOOK_BEGIN_CANDIDATE", INT2FIX(EB_HOOK_BEGIN_CANDIDATE));
	rb_define_const(mEB, "HOOK_END_CANDIDATE_GROUP", INT2FIX(EB_HOOK_END_CANDIDATE_GROUP));
	rb_define_const(mEB, "HOOK_END_CANDIDATE_LEAF", INT2FIX(EB_HOOK_END_CANDIDATE_LEAF));
	rb_define_const(mEB, "HOOK_BEGIN_REFERENCE", INT2FIX(EB_HOOK_BEGIN_REFERENCE));
	rb_define_const(mEB, "HOOK_END_REFERENCE", INT2FIX(EB_HOOK_END_REFERENCE));
	rb_define_const(mEB, "HOOK_BEGIN_KEYWORD", INT2FIX(EB_HOOK_BEGIN_KEYWORD));
	rb_define_const(mEB, "HOOK_END_KEYWORD", INT2FIX(EB_HOOK_END_KEYWORD));
	rb_define_const(mEB, "HOOK_NARROW_FONT", INT2FIX(EB_HOOK_NARROW_FONT));
	rb_define_const(mEB, "HOOK_WIDE_FONT", INT2FIX(EB_HOOK_WIDE_FONT));
	rb_define_const(mEB, "HOOK_ISO8859_1", INT2FIX(EB_HOOK_ISO8859_1));
	rb_define_const(mEB, "HOOK_NARROW_JISX0208", INT2FIX(EB_HOOK_NARROW_JISX0208));
	rb_define_const(mEB, "HOOK_WIDE_JISX0208", INT2FIX(EB_HOOK_WIDE_JISX0208));
	rb_define_const(mEB, "HOOK_GB2312", INT2FIX(EB_HOOK_GB2312));

	rb_define_const(mEB, "HOOK_BEGIN_MONO_GRAPHIC", INT2FIX(EB_HOOK_BEGIN_MONO_GRAPHIC));
	rb_define_const(mEB, "HOOK_END_MONO_GRAPHIC", INT2FIX(EB_HOOK_END_MONO_GRAPHIC));

	rb_define_const(mEB, "HOOK_BEGIN_GRAY_GRAPHIC", INT2FIX(EB_HOOK_BEGIN_GRAY_GRAPHIC));
	rb_define_const(mEB, "HOOK_END_GRAY_GRAPHIC", INT2FIX(EB_HOOK_END_GRAY_GRAPHIC));

	rb_define_const(mEB, "HOOK_BEGIN_COLOR_BMP", INT2FIX(EB_HOOK_BEGIN_COLOR_BMP));
	rb_define_const(mEB, "HOOK_BEGIN_COLOR_JPEG", INT2FIX(EB_HOOK_BEGIN_COLOR_JPEG));
 	rb_define_const(mEB, "HOOK_END_COLOR_GRAPHIC", INT2FIX(EB_HOOK_END_COLOR_GRAPHIC));

	rb_define_const(mEB, "HOOK_STOP_CODE", INT2FIX(EB_HOOK_STOP_CODE));

	rb_define_const(mEB, "HOOK_BEGIN_IN_COLOR_BMP", INT2FIX(EB_HOOK_BEGIN_IN_COLOR_BMP));
	rb_define_const(mEB, "HOOK_BEGIN_IN_COLOR_JPEG", INT2FIX(EB_HOOK_BEGIN_IN_COLOR_JPEG));
	rb_define_const(mEB, "HOOK_END_IN_COLOR_GRAPHIC", INT2FIX(EB_HOOK_END_IN_COLOR_GRAPHIC));

	rb_define_const(mEB, "HOOK_BEGIN_WAVE", INT2FIX(EB_HOOK_BEGIN_WAVE));
	rb_define_const(mEB, "HOOK_END_WAVE", INT2FIX(EB_HOOK_END_WAVE));
	rb_define_const(mEB, "HOOK_BEGIN_MPEG", INT2FIX(EB_HOOK_BEGIN_MPEG));
	rb_define_const(mEB, "HOOK_END_MPEG", INT2FIX(EB_HOOK_END_MPEG));
#endif

	rb_define_singleton_method(cEBPosition,"new",reb_pos_initialize,-1);
	rb_define_method(cEBPosition,"page",reb_pos_get_page,0);
	rb_define_method(cEBPosition,"offset",reb_pos_get_offset,0);
	rb_define_method(cEBPosition,"page=",reb_pos_set_page,1);
	rb_define_method(cEBPosition,"offset=",reb_pos_set_offset,1);

	eb_initialize_library();
	/* Don't call the following methods manually */
	rb_define_module_function(mEB,"Initialize",reb_mod_initialize,0);
	rb_define_module_function(mEB,"Finalize",reb_mod_finalize,0);
	rb_eval_string("at_exit do EB::Finalize(); end\n");
}
