/***************************************************************************
                          cconnectionmanager.cpp  -  description
                             -------------------
    begin                : Don Mai 16 2002
    copyright            : (C) 2002-2005 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <dclib/cclient.h>
#include <dclib/cconfig.h>
#include <dclib/core/cmanager.h>
#include <dclib/cdownloadmanager.h>
#include <dclib/dcobject.h>
#include <dclib/core/cbytearray.h>
#include <dclib/chttp.h>
#include <dclib/core/cbz.h>
#include <dclib/cfilemanager.h>

#ifndef WIN32
#include <unistd.h>
#include <stdlib.h>
#endif

#include "cconnectionmanager.h"

/** */
CConnectionManager::CConnectionManager()
{
	m_pClientList = new CThreadList<CClient>();

	m_pCallback = new CCallback<CConnectionManager>( this, &CConnectionManager::Callback );
	CManager::Instance()->Add( m_pCallback );

	m_bUpdateMyinfo   = FALSE;

	SetInstance(this);
}

/** */
CConnectionManager::~CConnectionManager()
{
	SetInstance(0);

	CManager::Instance()->Remove( m_pCallback );
	if (m_pCallback)
	{
		delete m_pCallback;
		m_pCallback = 0;
	}
	
	m_Thread.Lock();
	m_pClientList->Lock();

	delete m_pClientList;
	m_pClientList = 0;

	m_Thread.UnLock();
}

/** thread callbackfunction */
int CConnectionManager::Callback( CObject *, CObject * )
{
	CClient * client;

	m_Thread.Lock();

	// update all clients
	if ( m_pClientList )
	{
		client = 0;
		
		while( (client=m_pClientList->Next(client)) != 0 )
		{
			client->Thread(0);
		}
		
		// update myinfo from all clients
		if ( m_bUpdateMyinfo )
		{
			client = 0;
		
			while( (client=m_pClientList->Next(client)) != 0 )
			{
				if ( client->IsHandshake() == TRUE )
					continue;
					
				UpdateMyInfo(client);
			}
		
			m_bUpdateMyinfo = FALSE;
		}
	}

	m_Thread.UnLock();

	return 0;
}

/** */
void CConnectionManager::ConnectClient( CString hubname, CString server )
{
	DCMessageConnectClient *msg = new DCMessageConnectClient();

	msg->m_sHubName = hubname;
	msg->m_sHubHost = server;

	if ( DC_CallBack(msg) == -1 )
	{
		delete msg;

		Connect( hubname, server, NULL );
	}
}

/** */
void CConnectionManager::Connect( CString hubname, CString server, CClient * client, bool sslconnect )
{
	if ( server == "" )
	{
		return;
	}

	if ( hubname == "" )
	{
		hubname = server;
	}

	if ( !client )
	{
		m_Thread.Lock();

		client = (CClient*)GetHub(hubname,server);

		m_Thread.UnLock();

		if ( client )
		{
			return;
		}

		client = new CClient();
	}

	client->SetHubName(hubname);

	AddHub(client);

	client->SetNick(CConfig::Instance()->GetNick( hubname, server ));
	client->SetConnectionType(CConfig::Instance()->GetSpeed());
	client->SetVersion(CString("DCGUI-") + VERSION);
	client->SetShareSize(CString().setNum(CFileManager::Instance()->GetShareSize()));
	client->SetMode(CConfig::Instance()->GetMode());
	client->SetAwayMode(CConfig::Instance()->GetAwayMode());
						
	client->SetComment(CConfig::Instance()->GetDescription(FALSE, hubname, server));
	
	// set ssl mode, if profile exist use profile settings
	DCConfigHubProfile pConfigHubProfile;

	if ( CConfig::Instance()->GetBookmarkHubProfile( hubname, server, &pConfigHubProfile ) == TRUE )
	{
		if ( pConfigHubProfile.m_bEMail )
			client->SetEMail(pConfigHubProfile.m_sEMail);
		else
			client->SetEMail(CConfig::Instance()->GetEMail());
		client->SetSSLMode(pConfigHubProfile.m_bSSL);
	}
	else
	{
		client->SetEMail(CConfig::Instance()->GetEMail());
		client->SetSSLMode(sslconnect);
	}
	
	client->Connect(server);
}

/** */
void CConnectionManager::AddHub( CClient * client )
{
	m_Thread.Lock();

	if ( m_pClientList )
	{
		m_pClientList->Lock();
		m_pClientList->Add(client);
		m_pClientList->UnLock();
	}

	m_Thread.UnLock();
}

/** */
void CConnectionManager::RemoveHub( CClient * client )
{
	m_Thread.Lock();

	if ( m_pClientList )
	{
		m_pClientList->Lock();
		client->SetCallBackFunction(0);
		m_pClientList->Remove(client);
		m_pClientList->UnLock();
	}

	m_Thread.UnLock();
}

/** */
int CConnectionManager::SM_ClientCallBack( CObject * /*Sender*/, CObject * /*Object*/ )
{
//	Transfer->SetCallBackFunction( new CCallback<CDownloadManager>( pDownloadManager, &CDownloadManager::DM_TransferCallBack ) );
	return -1;
}

/** */
long CConnectionManager::GetConnectedHubCount( bool admin )
{
	long i;
	CClient * client;

	if ( !m_pClientList )
	{
		return 0;
	}

	m_pClientList->Lock();

	i = 0;
	client = 0;

	while( (client=m_pClientList->Next(client)) != 0 )
	{
		if ( (client->IsHandshake() == TRUE) )
			continue;

		if ( (admin == TRUE) && (client->UserList()->IsAdmin( client->GetNick() ) == TRUE) )
			continue;

		i++;
	}

	m_pClientList->UnLock();

	return i;
}

/** */
long CConnectionManager::GetConnectedHubPasswordCount( )
{
	long i;
	CClient * client;

	if ( !m_pClientList )
	{
		return 0;
	}

	m_pClientList->Lock();

	i = 0;
	client = 0;

	while( (client=m_pClientList->Next(client)) != 0 )
	{
		if ( (client->IsHandshake() == TRUE) )
			continue;	
	
		if ( (client->GetUsedPassword() == FALSE) )
			continue;

		i++;
	}

	m_pClientList->UnLock();

	return i;
}

/** */
CClient * CConnectionManager::GetHub( CString hubname, CString hubhost )
{
	CClient* client = 0;

	if ( !m_pClientList )
	{
		return client;
	}

	m_pClientList->Lock();

	client = GetHubObject(hubname,hubhost);

	m_pClientList->UnLock();

	return client;
}

/** */
CClient * CConnectionManager::GetHubObject( CString hubname, CString hubhost )
{
	CString o_host,s;
	unsigned int i;
	unsigned int o_port;
	CClient* client = 0, * client_candidate = 0;

	if ( !m_pClientList )
	{
		return client;
	}

	CSocket::ParseHost( hubhost, o_host, o_port );

	// set default port
	if ( o_port == 0 )
		o_port = 411;

	o_host = o_host.ToUpper();

	while( (client=m_pClientList->Next(client)) != 0 )
	{
		// compare hubname
		if ( client->GetHubName() == hubname )
			client_candidate = client;

		// compare hubhost
		CSocket::ParseHost( client->GetHost().ToUpper(), s, i );

		if ( i == 0 )
			i = 411;

		if ( s == o_host )
		{
			// host and port match
			if ( i == o_port )
				break;
			// host and hubname match, we can connect to the same hub on different ports
			if ( client->GetHubName() == hubname )
				break;
		}

		// compare peer name
		CSocket::ParseHost( client->GetHost(TRUE).ToUpper(), s, i );

		if ( i == 0 )
			i = 411;

		if ( s == o_host )
		{
			// host and port match
			if ( i == o_port )
				break;
			// host and hubname match, we can connect to the same hub on different ports
			if ( client->GetHubName() == hubname )
				break;
		}
	}

	if ( client == 0 )
	{
		// no client found
		client = client_candidate; 
	}
	
	return client;
}

/** */
CStringList * CConnectionManager::GetConnectedHubServerList()
{
	CClient* client;
	CStringList * StringList;

	if ( !m_pClientList )
	{
		return 0;
	}

	m_pClientList->Lock();

 	StringList = new CStringList();

	if ( m_pClientList->Count() > 0 )
	{
		client = 0;
		while( (client=m_pClientList->Next(client)) != 0 )
		{
			if ( client->IsHandshake() == TRUE )
			{
				continue;
			}

			StringList->Add(client->GetHubName(),new CString(client->GetHost()));
		}
	}

	m_pClientList->UnLock();

	return StringList;
}

/** send message to all connected servers */
int CConnectionManager::SendStringToConnectedServers( CString s, CString hubname, bool encode )
{
	int i;
	CClient* client;

	if ( !m_pClientList )
	{
		return 0;
	}

	m_pClientList->Lock();

	i = 0;

	if ( s != "" )
	{
		client = 0;

		while( (client=m_pClientList->Next(client)) != 0 )
		{
			if ( client->IsHandshake() == TRUE )
			{
				continue;
			}

			if ( hubname != "" )
			{
				if ( hubname == client->GetHubName() )
				{
					client->SendString(s, encode);
					i++;
					break;
				}
				else
				{
					continue;
				}
			}
			else
			{
				client->SendString(s, encode);
			}

			i++;
		}
	}

	m_pClientList->UnLock();

	return i;
}

/** send a search string (extracted from message) to connected servers */
int CConnectionManager::SendSearchToConnectedServers( CMessageSearchFile *sf, CString hubhost )
{
	int i;
	CClient* client;

	if ( !m_pClientList )
	{
		return 0;
	}

	m_pClientList->Lock();

	i = 0;

	if ( hubhost != "" )
	{
		client = GetHubObject( "", hubhost );
		
		if ( client )
		{
			if ( !(client->IsHandshake() == TRUE) )
			{
				// fix the nick for passive searches
				if ( sf->m_bLocal == TRUE )
				{
					sf->m_sSource = client->GetNick();
				}

				client->SendSearch(sf);
				i++;
			}
		}
		else
		{
			printf("CConnectionManager::SendSearchToConnectedServers hub not found\n");
		}
	}
	else
	{
		client = 0;

		while( (client=m_pClientList->Next(client)) != 0 )
		{
			// fix the nick for passive searches
			if ( sf->m_bLocal == TRUE )
			{
				sf->m_sSource = client->GetNick();
			}

			if ( client->IsHandshake() == TRUE )
			{
				continue;
			}

			client->SendSearch(sf);

			i++;
		}
	}

	m_pClientList->UnLock();

	return i;
}

/** send myinfo to all connected servers */
int CConnectionManager::SendMyInfoToConnectedServers()
{
	m_bUpdateMyinfo = TRUE;
	
	return 0;
}

/** */
void CConnectionManager::UpdateMyInfo( CClient* client )
{
	CString description,speed,email,share;
	eUserAwayMode awaymode;
	DCConfigHubProfile pConfigHubProfile;

	if ( !m_pClientList ||
	     !CFileManager::Instance() ||
	     !CConfig::Instance() )
	{
		return;
	}

	speed       = CConfig::Instance()->GetSpeed();
	share       = CString().setNum(CFileManager::Instance()->GetShareSize());
	awaymode    = CConfig::Instance()->GetAwayMode();

	// get description
	description = CConfig::Instance()->GetDescription(FALSE,client->GetHubName(),client->GetHost());
		
	// get hub profile
	if ( CConfig::Instance()->GetBookmarkHubProfile( client->GetHubName(), client->GetHost(), &pConfigHubProfile ) == TRUE )
	{
		if ( pConfigHubProfile.m_bEMail )
			email = pConfigHubProfile.m_sEMail;
		else
			email = CConfig::Instance()->GetEMail();
	}
	else
	{
		email = CConfig::Instance()->GetEMail();
	}
		
	// update internal values
	client->SetComment(description);
	client->SetConnectionType(speed);
	client->SetEMail(email);
	client->SetShareSize(share);
	client->SetAwayMode(awaymode);
	
	return;
}

/** */
int CConnectionManager::SendConnectionRequest( CString nick, CString hubname, CString hubhost )
{
	int res;
	CClient* client;

	if ( !m_pClientList )
	{
		return 0;
	}

	m_pClientList->Lock();

	res = -3;

	client = GetHubObject(hubname,hubhost);

	if ( client )
	{
		if ( client->IsHandshake() == TRUE )
		{
			res = -2;
		}
		else if ( client->UserList()->IsUserOnline( nick ) == FALSE )
		{
			res = -1;
		}
		else if ( client->GetMode() == ecmPASSIVE )
		{
			// send connection request
			if ( client->SendRevConnectToMe( client->GetNick(), nick ) == 0 )
			{
				res = 0;
			}
			else
			{
				res = -4;
			}
		}
		else if ( client->GetMode() == ecmACTIVE )
		{
			CString s = CConfig::Instance()->GetTCPHostString();

			if ( s != "" )
			{
				// send connection request
				CDownloadManager::Instance()->DLM_AddTransferRequest( nick, "", hubname, client->GetHost() );

				if ( client->SendConnectToMe( nick, s ) == 0 )
				{
					res = 0;
				}
				else
				{
					res = -4;
				}
			}
			else
			{
				res = -4;
			}
		}
	}
	m_pClientList->UnLock();

	return res;
}

/** */
bool CConnectionManager::IsUserOnline( CString nick, CString hubname, CString hubhost, CList<DCHubObject> * list )
{
	bool res;
	CClient* client;
	DCHubObject * HubObject;

	if ( !m_pClientList )
	{
		return FALSE;
	}

	m_pClientList->Lock();

	res = FALSE;
	client = 0;

	hubhost = hubhost.ToUpper();

	while( (client=m_pClientList->Next(client)) != 0 )
	{
		if ( client->IsHandshake() == FALSE )
		{
			if ( hubname == "" )
			{
				if ( client->UserList()->IsUserOnline( nick ) == TRUE )
				{
					if ( list )
					{
						HubObject = new DCHubObject();
						HubObject->m_sHubName = client->GetHubName();
						HubObject->m_sHubHost = client->GetHost();
						list->Add(HubObject);
					}
					res = TRUE;
				}
			}
			if ( (client->GetHubName() == hubname) ||
			     (client->GetHost().ToUpper() == hubhost) ||
			     (client->GetHost(TRUE) == hubhost) )
			{
				if ( client->UserList()->IsUserOnline( nick ) == TRUE )
					res = TRUE;
				break;
			}
		}
	}
	m_pClientList->UnLock();

	return res;
}

/** */
bool CConnectionManager::SetUserTransferInfo( CString hubname, CString hubhost, CString nick, CDCMessage * msg )
{
	bool res;
	CClient* client;

	if ( !m_pClientList )
	{
		return FALSE;
	}

	m_pClientList->Lock();

	if ( (client = GetHubObject( hubname, hubhost )) != 0 )
	{
		if ( client->IsHandshake() == FALSE )
		{
			res = client->SetUserTransferInfo(nick,msg);
		}
		else
		{
			printf("CConnectionManager::SetUserTransferInfo: hub offline\n");
		}
	}
	else
	{
		printf("CConnectionManager::SetUserTransferInfo: can't find hub\n");
	}
	
	m_pClientList->UnLock();

	return res;
}

/** */
eHubState CConnectionManager::IsHubOnline( CString hubname, CString hubhost )
{
	eHubState res;
	CClient* client;

	if ( !m_pClientList )
	{
		return ehsNONE;
	}

	m_pClientList->Lock();

	res = ehsNONE;

	if ( (client = GetHubObject( hubname, hubhost )) != 0 )
	{
		if ( client->IsHandshake() == FALSE )
		{
			res = ehsONLINE;
		}
		else
		{
			res = ehsOFFLINE;
		}
	}
	m_pClientList->UnLock();

	return res;
}

/** */
CString CConnectionManager::GetNick( CString hubname, CString hubhost )
{
	CString res;
	CClient* client;

	if ( !m_pClientList )
	{
		return res;
	}

	m_pClientList->Lock();

	if ( (client = GetHubObject( hubname, hubhost )) != 0 )
	{
		res = client->GetNick();
	}

	m_pClientList->UnLock();

	return res;
}


/** */
bool CConnectionManager::IsAdmin( CString hubname, CString hubhost, CString nick )
{
	bool res;
	CClient* client;

	if ( !m_pClientList )
	{
		return FALSE;
	}

	m_pClientList->Lock();

	res = FALSE;

	if ( (client = GetHubObject(hubname,hubhost)) != 0 )
	{
		if ( client->IsHandshake() == FALSE )
		{
			if ( nick == "" )
				res = client->UserList()->IsAdmin( client->GetNick() );
			else
				res = client->UserList()->IsAdmin(nick);
		}
	}

	m_pClientList->UnLock();

	return res;
}

/** */
bool CConnectionManager::GetUserMyInfo( CString hubname, CString hubhost, CString nick, CMessageMyInfo * p )
{
	bool res;
	CClient* client;

	if ( !m_pClientList )
	{
		return FALSE;
	}

	m_pClientList->Lock();

	res = FALSE;

	if ( (client = GetHubObject(hubname,hubhost)) != 0 )
	{
		if ( client->IsHandshake() == FALSE )
		{
			res = client->UserList()->GetUserMyInfo(nick,p);
		}
	}

	m_pClientList->UnLock();

	return res;
}

/** TODO: remove function ! (used in ctransferview) */
CString CConnectionManager::GetHubHost( CString hubname )
{
	CString res;
	CClient* client;

	if ( !m_pClientList )
	{
		return "";
	}

	m_pClientList->Lock();

 	res = "";
	client = 0;

	while( (client=m_pClientList->Next(client)) != 0 )
	{
		if ( client->GetHubName() == hubname )
		{
			res = client->GetIP() + ":" + CString().setNum(client->GetPort());
			break;
		}
	}
	m_pClientList->UnLock();

	return res;
}
