// camellia.cpp - by Kevin Springle, 2003
// This code is hereby placed in the public domain.

#include "pch.h"

#ifdef WORD64_AVAILABLE

#include "camellia.h"
#include "misc.h"

NAMESPACE_BEGIN(CryptoPP)

// Define internal Camellia function macros

inline word64 Camellia::Base::F(word64 X)
{
	word32 t1 = (word32)(X >> 32);
	word32 t2 = (word32)(X & 0xFFFFFFFFL);
	t2=	(s2[GETBYTE(t2, 3)] << 24) |
		(s3[GETBYTE(t2, 2)] << 16) |
		(s4[GETBYTE(t2, 1)] <<	8) |
		(s1[GETBYTE(t2, 0)]);
	t1=	(s1[GETBYTE(t1, 3)] << 24) |
		(s2[GETBYTE(t1, 2)] << 16) |
		(s3[GETBYTE(t1, 1)] <<	8) |
		(s4[GETBYTE(t1, 0)]);
	t1 ^= rotlFixed(t2, 8);
	t2 ^= rotlFixed(t1, 16);
	t1 ^= rotlFixed(t2, 24);
	t2 ^= rotlFixed(t1, 24);
	return ((word64)t2 << 32) | (word64)t1;
}

#define ROUND2(Xp, K1, K2) \
	{ (Xp)[1] ^= F((Xp)[0] ^ K1); (Xp)[0] ^= F((Xp)[1] ^ K2); }

inline void Camellia::Base::FLlayer(word64 *x, word64 K1, word64 K2)
{
	word32 Xl = (word32)(x[0] >> 32);
	word32 Xr = (word32)(x[0] & 0xFFFFFFFFL);
	Xr ^= rotlFixed(Xl & (word32)(K1 >> 32), 1);
	Xl ^= (Xr | (word32)(K1 & 0xFFFFFFFFL));
    x[0] = ((word64)Xl << 32) | (word64)Xr;

	Xl = (word32)(x[1] >> 32);
	Xr = (word32)(x[1] & 0xFFFFFFFFL);
	Xl ^= (Xr | (word32)(K2 & 0xFFFFFFFFL));
	Xr ^= rotlFixed(Xl & (word32)(K2 >> 32), 1);
    x[1] = ((word64)Xl << 32) | (word64)Xr;
}

inline void rotl128(word64 *x, unsigned int bits)
{
	word64 temp = x[0] >> (64 - bits);
	x[0] = (x[0] << bits) | (x[1] >> (64 - bits));
	x[1] = (x[1] << bits) | temp;
}

void Camellia::Base::UncheckedSetKey(CipherDir dir, const byte *key, unsigned int keylen)
{
	AssertValidKeyLength(keylen);

	m_rounds = (keylen >= 24) ? 4 : 3;
	unsigned int kslen = (8 * m_rounds + 2);
	m_key.New(8 * kslen);
	word64 *ks = m_key;

	FixedSizeSecBlock<word64, 32> keyword;
	word64 *kw = keyword;

#define KL	(kw+0)
#define KR	(kw+2)
#define KA	(kw+4)
#define KB	(kw+6)

	if (keylen == 16)
	{
		GetUserKey(BIG_ENDIAN_ORDER, kw, 2, key, keylen);
		KA[0] = KL[0];
		KA[1] = KL[1];
	}
	else
	{
		if (keylen == 24)
		{
			GetUserKey(BIG_ENDIAN_ORDER, kw, 3, key, keylen);
			KR[1] = ~KR[0];
		}
		else
		{
			GetUserKey(BIG_ENDIAN_ORDER, kw, 4, key, keylen);
		}
		KA[0] = KL[0] ^ KR[0];
		KA[1] = KL[1] ^ KR[1];
	}
	ROUND2(KA, W64LIT(0xA09E667F3BCC908B), W64LIT(0xB67AE8584CAA73B2));
	KA[0] ^= KL[0];
	KA[1] ^= KL[1];
	ROUND2(KA, W64LIT(0xC6EF372FE94F82BE), W64LIT(0x54FF53A5F1D36F1C));

	if (keylen == 16)
	{
		ks[0] = KL[0]; ks[1] = KL[1];
		rotl128(KL, 15);
		ks[4] = KL[0]; ks[5] = KL[1];
		rotl128(KL, 30);
		ks[10] = KL[0]; ks[11] = KL[1];
		rotl128(KL, 15);
		ks[13] = KL[1];
		rotl128(KL, 17);
		ks[16] = KL[0]; ks[17] = KL[1];
		rotl128(KL, 17);
		ks[18] = KL[0]; ks[19] = KL[1];
		rotl128(KL, 17);
		ks[22] = KL[0]; ks[23] = KL[1];

		ks[2] = KA[0]; ks[3] = KA[1];
		rotl128(KA, 15);
		ks[6] = KA[0]; ks[7] = KA[1];
		rotl128(KA, 15);
		ks[8] = KA[0]; ks[9] = KA[1];
		rotl128(KA, 15);
		ks[12] = KA[0];
		rotl128(KA, 15);
		ks[14] = KA[0]; ks[15] = KA[1];
		rotl128(KA, 34);
		ks[20] = KA[0]; ks[21] = KA[1];
		rotl128(KA, 17);
		ks[24] = KA[0]; ks[25] = KA[1];
	}
	else
	{
		KB[0] = KA[0] ^ KR[0];
		KB[1] = KA[1] ^ KR[1];
		ROUND2(KB, W64LIT(0x10E527FADE682D1D), W64LIT(0xB05688C2B3E6C1FD));

		ks[0] = KL[0]; ks[1] = KL[1];
		rotl128(KL, 45);
		ks[12] = KL[0]; ks[13] = KL[1];
		rotl128(KL, 15);
		ks[16] = KL[0]; ks[17] = KL[1];
		rotl128(KL, 17);
		ks[22] = KL[0]; ks[23] = KL[1];
		rotl128(KL, 34);
		ks[30] = KL[0]; ks[31] = KL[1];

		rotl128(KR, 15);
		ks[4] = KR[0]; ks[5] = KR[1];
		rotl128(KR, 15);
		ks[8] = KR[0]; ks[9] = KR[1];
		rotl128(KR, 30);
		ks[18] = KR[0]; ks[19] = KR[1];
		rotl128(KR, 34);
		ks[26] = KR[0]; ks[27] = KR[1];

		rotl128(KA, 15);
		ks[6] = KA[0]; ks[7] = KA[1];
		rotl128(KA, 30);
		ks[14] = KA[0]; ks[15] = KA[1];
		rotl128(KA, 32);
		ks[24] = KA[0]; ks[25] = KA[1];
		rotl128(KA, 17);
		ks[28] = KA[0]; ks[29] = KA[1];

		ks[2] = KB[0]; ks[3] = KB[1];
		rotl128(KB, 30);
		ks[10] = KB[0]; ks[11] = KB[1];
		rotl128(KB, 30);
		ks[20] = KB[0]; ks[21] = KB[1];
		rotl128(KB, 51);
		ks[32] = KB[0]; ks[33] = KB[1];
	}

	if (dir == DECRYPTION)	// reverse key schedule order
	{
		std::swap(ks[0], ks[kslen-2]);
		std::swap(ks[1], ks[kslen-1]);
		for (unsigned int i=2; i<kslen/2; i++)
		{
			std::swap(ks[i], ks[kslen-1-i]);
		}
	}
}

typedef BlockGetAndPut<word64, BigEndian> Block;

void Camellia::Base::ProcessAndXorBlock(const byte *inBlock, const byte *xorBlock, byte *outBlock) const
{
	FixedSizeSecBlock<word64, 16> mb;
	word64 *m = mb;
	const word64 *ks = m_key;

	Block::Get(inBlock)(m[0])(m[1]);

	m[0] ^= ks[0];
	m[1] ^= ks[1];
	ks += 2;
	for (unsigned int i = m_rounds; i > 0; --i)
	{
		ROUND2(m, ks[0], ks[1]);
		ROUND2(m, ks[2], ks[3]);
		ROUND2(m, ks[4], ks[5]);
		if (i != 1)
		{
			FLlayer(m, ks[6], ks[7]);
			ks += 8;
		}
		else
		{
			m[0] ^= ks[7];
			m[1] ^= ks[6];
		}
	}

	Block::Put(xorBlock, outBlock)(m[1])(m[0]);
}

// The Camellia s-boxes

const byte Camellia::Base::s1[256] =
{
	112,130,44,236,179,39,192,229,228,133,87,53,234,12,174,65,
	35,239,107,147,69,25,165,33,237,14,79,78,29,101,146,189,
	134,184,175,143,124,235,31,206,62,48,220,95,94,197,11,26,
	166,225,57,202,213,71,93,61,217,1,90,214,81,86,108,77,
	139,13,154,102,251,204,176,45,116,18,43,32,240,177,132,153,
	223,76,203,194,52,126,118,5,109,183,169,49,209,23,4,215,
	20,88,58,97,222,27,17,28,50,15,156,22,83,24,242,34,
	254,68,207,178,195,181,122,145,36,8,232,168,96,252,105,80,
	170,208,160,125,161,137,98,151,84,91,30,149,224,255,100,210,
	16,196,0,72,163,247,117,219,138,3,230,218,9,63,221,148,
	135,92,131,2,205,74,144,51,115,103,246,243,157,127,191,226,
	82,155,216,38,200,55,198,59,129,150,111,75,19,190,99,46,
	233,121,167,140,159,110,188,142,41,245,249,182,47,253,180,89,
	120,152,6,106,231,70,113,186,212,37,171,66,136,162,141,250,
	114,7,185,85,248,238,172,10,54,73,42,104,60,56,241,164,
	64,40,211,123,187,201,67,193,21,227,173,244,119,199,128,158
};

const byte Camellia::Base::s2[256] =
{
	224,5,88,217,103,78,129,203,201,11,174,106,213,24,93,130,
	70,223,214,39,138,50,75,66,219,28,158,156,58,202,37,123,
	13,113,95,31,248,215,62,157,124,96,185,190,188,139,22,52,
	77,195,114,149,171,142,186,122,179,2,180,173,162,172,216,154,
	23,26,53,204,247,153,97,90,232,36,86,64,225,99,9,51,
	191,152,151,133,104,252,236,10,218,111,83,98,163,46,8,175,
	40,176,116,194,189,54,34,56,100,30,57,44,166,48,229,68,
	253,136,159,101,135,107,244,35,72,16,209,81,192,249,210,160,
	85,161,65,250,67,19,196,47,168,182,60,43,193,255,200,165,
	32,137,0,144,71,239,234,183,21,6,205,181,18,126,187,41,
	15,184,7,4,155,148,33,102,230,206,237,231,59,254,127,197,
	164,55,177,76,145,110,141,118,3,45,222,150,38,125,198,92,
	211,242,79,25,63,220,121,29,82,235,243,109,94,251,105,178,
	240,49,12,212,207,140,226,117,169,74,87,132,17,69,27,245,
	228,14,115,170,241,221,89,20,108,146,84,208,120,112,227,73,
	128,80,167,246,119,147,134,131,42,199,91,233,238,143,1,61
};

const byte Camellia::Base::s3[256] =
{
	56,65,22,118,217,147,96,242,114,194,171,154,117,6,87,160,
	145,247,181,201,162,140,210,144,246,7,167,39,142,178,73,222,
	67,92,215,199,62,245,143,103,31,24,110,175,47,226,133,13,
	83,240,156,101,234,163,174,158,236,128,45,107,168,43,54,166,
	197,134,77,51,253,102,88,150,58,9,149,16,120,216,66,204,
	239,38,229,97,26,63,59,130,182,219,212,152,232,139,2,235,
	10,44,29,176,111,141,136,14,25,135,78,11,169,12,121,17,
	127,34,231,89,225,218,61,200,18,4,116,84,48,126,180,40,
	85,104,80,190,208,196,49,203,42,173,15,202,112,255,50,105,
	8,98,0,36,209,251,186,237,69,129,115,109,132,159,238,74,
	195,46,193,1,230,37,72,153,185,179,123,249,206,191,223,113,
	41,205,108,19,100,155,99,157,192,75,183,165,137,95,177,23,
	244,188,211,70,207,55,94,71,148,250,252,91,151,254,90,172,
	60,76,3,53,243,35,184,93,106,146,213,33,68,81,198,125,
	57,131,220,170,124,119,86,5,27,164,21,52,30,28,248,82,
	32,20,233,189,221,228,161,224,138,241,214,122,187,227,64,79
};

const byte Camellia::Base::s4[256] =
{
	112,44,179,192,228,87,234,174,35,107,69,165,237,79,29,146,
	134,175,124,31,62,220,94,11,166,57,213,93,217,90,81,108,
	139,154,251,176,116,43,240,132,223,203,52,118,109,169,209,4,
	20,58,222,17,50,156,83,242,254,207,195,122,36,232,96,105,
	170,160,161,98,84,30,224,100,16,0,163,117,138,230,9,221,
	135,131,205,144,115,246,157,191,82,216,200,198,129,111,19,99,
	233,167,159,188,41,249,47,180,120,6,231,113,212,171,136,141,
	114,185,248,172,54,42,60,241,64,211,187,67,21,173,119,128,
	130,236,39,229,133,53,12,65,239,147,25,33,14,78,101,189,
	184,143,235,206,48,95,197,26,225,202,71,61,1,214,86,77,
	13,102,204,45,18,32,177,153,76,194,126,5,183,49,23,215,
	88,97,27,28,15,22,24,34,68,178,181,145,8,168,252,80,
	208,125,137,151,91,149,255,210,196,72,247,219,3,218,63,148,
	92,2,74,51,103,243,127,226,155,38,55,59,150,75,190,46,
	121,140,110,142,245,182,253,89,152,106,70,186,37,66,162,250,
	7,85,238,10,73,104,56,164,40,123,201,193,227,244,199,158
};

NAMESPACE_END

#endif // WORD64_AVAILABLE
