/* -*- indent-tabs-mode: t; tab-width: 8; c-basic-offset: 8; -*- */

#include "tree.h"
#include "util.h"
#include "volume.h"
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>

static void iso_tree_free_r(struct iso_tree_dir *dir)
{
	struct iso_tree_file *f;
	struct iso_tree_dir *d;
	int i;

	for (i = 0; i < dir->nfiles; ++i) {
		f = &dir->files[i];
		free(f->path);
		free(f->fullname);
		free(f->name);
		free(f->extension);
		free(f->isoname1);
		free(f->isoname2);
		free(f->jolietname);
		free(f->rrname);
		free(f->me);
	}
	free(dir->fullname);
	free(dir->isoname1);
	free(dir->isoname2);
	free(dir->jolietname);
	free(dir->rrname);
	free(dir->me);

	for (i = 0; i < dir->nchildren; ++i) {
		d = &dir->children[i];
		iso_tree_free_r(d);
	}

	free(dir->children);
	free(dir->files);
}

void iso_tree_free(struct iso_tree_dir **root)
{
	if (root && *root) {
		struct iso_tree_dir *d = *root;

		iso_tree_free_r(d);
		free(d);
	}
}

struct iso_tree_dir **iso_tree_new_root(struct iso_volumeset *volset)
{
	struct iso_tree_dir *d;

	d = calloc(1, sizeof(struct iso_tree_dir));
	/* initialize stuff? all zero's for now. */
	d->me = malloc(sizeof(struct iso_tree_dir *));
	*d->me = d;
	d->volset = volset;
	return d->me;
}

struct iso_tree_dir **iso_tree_add_dir(struct iso_tree_dir **dparent,
				       const char *name, int volume)
{
	struct iso_tree_dir *parent;
	struct iso_tree_dir *d;
	int i, j;

	if (!(dparent && *dparent && name))
		return NULL;

	parent = *dparent;
	parent->nchildren++;
	parent->children = realloc(parent->children,
				   sizeof(struct iso_tree_dir) *
				   parent->nchildren);
	d = &parent->children[parent->nchildren - 1];
	memset(d, 0, sizeof(struct iso_tree_dir));

	if (!iso_tree_dir_set_name(&d, name)) {
		/* we don't unalloc the end of the array but that's ok :> */
		parent->nchildren--;
		d = NULL;
	} else {
		d->volume = volume;
		d->parent = parent;
		d->me = malloc(sizeof(struct iso_tree_dir *));
		d->volset = parent->volset;
	}

	/* since 'parent->children' was set to a new memory location, fix the
	   'parent' references */
	for (i = 0; i < parent->nchildren; ++i) {
		*parent->children[i].me = &parent->children[i];
		for (j = 0; j < parent->children[i].nchildren; ++j) {
			parent->children[i].children[j].parent =
				&parent->children[i];
		}
	}

	return d->me;
}

struct iso_tree_file **iso_tree_add_file(struct iso_tree_dir **dparent,
					 const char *path, int volume)
{
	struct iso_tree_file *f;
	struct stat st;
	char *r;
	struct iso_tree_dir *parent;
	int i;

	if (!(dparent && *dparent && path))
		return NULL;
	if (path[0] != '/')
		return NULL;

	if (stat(path, &st) < 0)
		return NULL;
	if (!S_ISREG(st.st_mode))
		return NULL;

	parent = *dparent;
	parent->nfiles++;
	parent->files = realloc(parent->files,
				sizeof(struct iso_tree_file) * parent->nfiles);
	f = &parent->files[parent->nfiles - 1];
	memset(f, 0, sizeof(struct iso_tree_file));

	r = strrchr(path, '/');
	assert(r);

	if (!iso_tree_file_set_name(&f, r + 1)) {
		/* we don't unalloc the end of the array but that's ok :> */
		parent->nfiles--;
		f = NULL;
	} else {
		f->volume = volume;
		f->path = strdup(path);
		f->version = 1;
		f->size = st.st_size;
		f->me = malloc(sizeof(struct iso_tree_file *));
		*f->me = f;
		f->volset = parent->volset;
	}

	for (i = 0; i < parent->nfiles; ++i)
		*parent->files[i].me = &parent->files[i];

	return f->me;
}

const char *iso_tree_file_get_name(struct iso_tree_file **dfile,
				   enum iso_file_name_version ver)
{
	struct iso_tree_file *file;

	if (!(dfile && *dfile))
		return NULL;

	file = *dfile;

	switch (ver) {
	case ISO_FILE_NAME_FULL:
		return file->fullname;
	case ISO_FILE_NAME_ISO:
		switch (file->volset->options.iso_level) {
		case 1:
			return file->isoname1;
		case 2:
			return file->isoname2;
		default:
			assert(0);
		}
		break;
	case ISO_FILE_NAME_JOLIET:
		return file->jolietname;
	case ISO_FILE_NAME_ROCKRIDGE:
		return file->rrname;
	}
	return NULL;
}

char *get_iso_name_1(const char *name, const char *ext)
{
	char *out = NULL;
	char *n, *e;

	/* XXX avoid duplicates */
	n = iso_d_strndup(name, 8);
	e = iso_d_strndup(ext, 3);
	if (strlen(n) || strlen(e))
		out = iso_strconcat('.', n, e);

	free(n);
	free(e);
	return out;
}

char *get_iso_name_2(const char *name, const char *ext)
{
	char *out = NULL;
	char *n, *e;
	int nl;

	/* XXX avoid duplicates */
	n = iso_d_strndup(name, 30);
	nl = strlen(n);
	e = iso_d_strndup(ext, 30 - nl);
	if (nl || strlen(e))
		out = iso_strconcat('.', n, e);

	free(n);
	free(e);
	return out;
}

int iso_tree_file_set_name(struct iso_tree_file **dfile, const char *fullname)
{
	int ret = 0;
	char *name = NULL, *ext;
	char *isoname1 = NULL, *isoname2 = NULL;
	char *jolietname = NULL, *rrname = NULL;
	struct iso_tree_file *file;

	if (!(dfile && *dfile))
		return ret;
	if (!fullname)
		return ret;

	file = *dfile;

	name = strdup(fullname);
	iso_split_filename(name, &ext);

	if (!(strlen(name) || strlen(ext)))
		goto set_name_done;

	isoname1 = get_iso_name_1(name, ext);
	if (!isoname1)
		goto set_name_fail;
	isoname2 = get_iso_name_2(name, ext);
	if (!isoname2)
		goto set_name_fail;
	ret = 1;

	free(file->fullname);
	free(file->name);
	free(file->extension);
	free(file->isoname1);
	free(file->isoname2);
	free(file->jolietname);
	free(file->rrname);

	file->fullname = strdup(fullname);
	file->name = strdup(name);
	file->extension = strdup(ext);
	file->isoname1 = isoname1;
	file->isoname2 = isoname2;
	file->jolietname = jolietname;
	file->rrname = rrname;

	goto set_name_done;

      set_name_fail:
	free(isoname1);
	free(isoname2);
	free(jolietname);
	free(rrname);
      set_name_done:
	free(name);

	return ret;
}

const char *iso_tree_dir_get_name(struct iso_tree_dir **ddir,
				  enum iso_file_name_version ver)
{
	struct iso_tree_dir *dir;

	if (!(ddir && *ddir))
		return NULL;

	dir = *ddir;

	switch (ver) {
	case ISO_FILE_NAME_FULL:
		return dir->fullname;
	case ISO_FILE_NAME_ISO:
		switch (dir->volset->options.iso_level) {
		case 1:
			return dir->isoname1;
		case 2:
			return dir->isoname2;
		default:
			assert(0);
		}
		break;
	case ISO_FILE_NAME_JOLIET:
		return dir->jolietname;
	case ISO_FILE_NAME_ROCKRIDGE:
		return dir->rrname;
	}
	return NULL;
}

int iso_tree_dir_set_name(struct iso_tree_dir **ddir, const char *name)
{
	int ret = 0;
	char *isoname1 = NULL, *isoname2 = NULL;
	char *jolietname = NULL, *rrname = NULL;
	struct iso_tree_dir *dir;

	if (!(ddir && *ddir))
		return ret;
	if (!name)
		return ret;

	dir = *ddir;

	isoname1 = iso_d_strndup(name, 8);
	isoname2 = iso_d_strndup(name, 31);
	jolietname = NULL;
	rrname = NULL;

	if (strlen(isoname1) && strlen(isoname2)) {
		free(dir->fullname);
		free(dir->isoname1);
		free(dir->isoname2);
		free(dir->jolietname);
		free(dir->rrname);

		dir->fullname = strdup(name);
		dir->isoname1 = isoname1;
		dir->isoname2 = isoname2;
		dir->jolietname = jolietname;
		dir->rrname = rrname;
		ret = 1;
	} else {
		free(isoname1);
		free(isoname2);
		free(jolietname);
		free(rrname);
	}
	return ret;
}

static int fnamecmp(const void *v1, const void *v2)
{
	struct iso_tree_file *f1 = (struct iso_tree_file *)v1;
	struct iso_tree_file *f2 = (struct iso_tree_file *)v2;

	return strcmp(iso_tree_file_get_name(&f1, ISO_FILE_NAME_ISO),
		      iso_tree_file_get_name(&f2, ISO_FILE_NAME_ISO));
}

static int dnamecmp(const void *v1, const void *v2)
{
	struct iso_tree_dir *d1 = (struct iso_tree_dir *)v1;
	struct iso_tree_dir *d2 = (struct iso_tree_dir *)v2;

	return strcmp(iso_tree_dir_get_name(&d1, ISO_FILE_NAME_ISO),
		      iso_tree_dir_get_name(&d2, ISO_FILE_NAME_ISO));
}

void iso_tree_sort(struct iso_tree_dir **root)
{
	int i;

	qsort((*root)->files, (*root)->nfiles,
	      sizeof(struct iso_tree_file), fnamecmp);
	qsort((*root)->children, (*root)->nchildren,
	      sizeof(struct iso_tree_dir), dnamecmp);

	/* update the me pointers */
	for (i = 0; i < (*root)->nchildren; ++i)
		*(*root)->children[i].me = &(*root)->children[i];
	for (i = 0; i < (*root)->nfiles; ++i)
		*(*root)->files[i].me = &(*root)->files[i];

	for (i = 0; i < (*root)->nchildren; ++i)
		iso_tree_sort((*root)->children[i].me);
}

#include <stdio.h>
static void iso_tree_print_r(struct iso_tree_dir *dir, int spaces)
{
	int i, j;

	for (i = 0; i < spaces; ++i)
		printf(" ");
	if (dir->fullname)
		printf("%s", iso_tree_dir_get_name(&dir, ISO_FILE_NAME_ISO));
	printf("/\n");

	spaces += 2;

	for (j = 0; j < dir->nchildren; ++j)
		iso_tree_print_r(&dir->children[j], spaces);

	for (j = 0; j < dir->nfiles; ++j) {
		for (i = 0; i < spaces; ++i)
			printf(" ");
		printf("%s\n", iso_tree_file_get_name(dir->files[j].me,
						      ISO_FILE_NAME_ISO));
	}
}

void iso_tree_print(struct iso_tree_dir **root)
{
	iso_tree_print_r(*root, 0);
}
