/* 
 * $Id: ktree.c,v 1.15 2001/07/16 22:59:25 doviende Exp $
 *
 * libarr - a screen management toolkit
 *
 * Copyright (C) 2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Author: Chris Bond
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>

#include "ktree.h"
#include "tinfo.h"
#include "arr.h"
#include "types.h"
#include "cap_offset.h"

/* k_list is a list of keys that will be put in our FSA, which will will later
 * search each time a new key (or set of keys) is read.
 */
static kqueue_entry k_list[] = {
	KQUEUE_ENTRY(OFFSET_AR, AK_ARROW_RIGHT),	/* right arrow key */
	KQUEUE_ENTRY(OFFSET_AL, AK_ARROW_LEFT),		/* left arrow key */
	KQUEUE_ENTRY(OFFSET_AU, AK_ARROW_UP),		/* up arrow key */
	KQUEUE_ENTRY(OFFSET_AD, AK_ARROW_DOWN),		/* down arrow key */
	KQUEUE_ENTRY(OFFSET_KH, AK_HOME),		/* `home' key */
	KQUEUE_ENTRY(OFFSET_KE, AK_END),		/* `end' key */
	KQUEUE_ENTRY(OFFSET_KD, AK_DELETE),		/* `delete' key */
	KQUEUE_ENTRY(OFFSET_KP, AK_PAGEUP),		/* page up */
	KQUEUE_ENTRY(OFFSET_KN, AK_PAGEDOWN),		/* page down */
	KQUEUE_ENTRY(NULL, 0)
};

/* The following is a nasty, dirty hack that needs to go away.  It seems like
 * terminfo files for xterm and alike are broken.  Where my terminfo entry for
 * `xterm' says the up arrow sends '^[OA', it actually sends `^[[A'.  BORK.
 */
static kqueue_entry_c k_bork[] = {
	KQUEUE_ENTRY_C(AK_ARROW_RIGHT, "[C"),
	KQUEUE_ENTRY_C(AK_ARROW_LEFT, "[D"),
	KQUEUE_ENTRY_C(AK_ARROW_UP, "[A"),
	KQUEUE_ENTRY_C(AK_ARROW_DOWN, "[B"),
	KQUEUE_ENTRY_C((MODIFIER_SHIFT | '\t'), "\033$"),	/* input.c@88 */
	KQUEUE_ENTRY_C(NULL, NULL)
};

/* tree_head is the head node of our FSA. */
static ktree_t *tree_head;

static ktree_t *
find_node(parent, ch)
	ktree_t *parent;
	const char ch;
{
	ktree_t *start;

	for (start = parent->child; start; start = start->next)
		if ((char) start->key == ch)
			return start;

	return NULL;
}

static int
mod_key(ch)
	int ch;
{
	switch (ch) {
		case '\n':
		case '\r':
			ch = AK_ENTER;
			break;
		case 127:
			ch = AK_BACKSPACE;
			break;
		case '\t':
			break;
		default:
			if (iscntrl(ch))
				ch = (MODIFIER_CTRL | (ch + 'A' - 1));

			break;
	}

	return ch;
}

static arr_kqueue *
queue_flush(node, queue)
	ktree_t *node;
	arr_kqueue *queue;
{
	arr_kqueue *qp;

	while (node->parent) {
		if ((qp = (arr_kqueue *) malloc(sizeof(arr_kqueue))) == NULL) {
			perror("Unable to allocate memory chunk");
			return NULL;
		}

		qp->key = mod_key(node->key);
		qp->next = queue;
		queue = qp;

		node = node->parent;
	}

	return queue;
}

static void
read_modifiers(qp)
	arr_kqueue *qp;
{
#if defined(TIOCLINUX) && defined(__linux__)
	char mod;

	mod = 6;
	if (ioctl(0, TIOCLINUX, &mod) >= 0) {
		if ((mod & 2) || (mod & 8))
			qp->key |= MODIFIER_ALT;
		if (mod & 4)
			qp->key |= MODIFIER_CTRL;
	}
#endif
}

arr_kqueue *
read_queue(func)
	int (*func)(FILE *);
{
	ktree_t *node, *scan;
	arr_kqueue *qp;
	int fc, ch;

	node = tree_head;

	for (;;) {
		if ((fc = fcntl(fileno(stdin), F_GETFL)) >= 0)
			fcntl(fileno(stdin), F_SETFL, fc | O_NONBLOCK);

		/* If we don't receive a key, wait a bit and try again.  This is
		 * actually a hack to make multi-byte locales work.
		 */
		ch = func(stdin);
		if (ch <= 0) {
			fd_set fds;
			struct timeval timeout;
			
			FD_ZERO(&fds);
			FD_SET(fileno(stdin), &fds);
			
			timeout.tv_usec = 100000; /* 100ms - 1/10s */
			timeout.tv_sec = 0;

			select(fileno(stdin) + 1, &fds, NULL, &fds, &timeout);
			ch = func(stdin);
		}

		fcntl(fileno(stdin), F_SETFL, fc);

		if (ch <= 0)
			return queue_flush(node, (arr_kqueue *)NULL);

		if (!(scan = find_node(node, ch))) {
			if (!(qp = (arr_kqueue *) malloc(sizeof(arr_kqueue)))) {
				perror("Unable to allocate memory chunk");
				return NULL;
			}

			qp->key = mod_key(ch);
			read_modifiers(qp);
			qp->next = NULL;

			return queue_flush(node, qp);
		}

		if (!scan->child) {
			if (!(qp = (arr_kqueue *) malloc(sizeof(arr_kqueue)))) {
				perror("Unable to allocate memory chunk");
				return NULL;
			}

			qp->key = scan->symbol;
			read_modifiers(qp);
			qp->next = NULL;

			return qp;
		}
		
		node = scan;
	}

	return NULL;
}

ktree_t *
create_node(parent, k, symbol)
	ktree_t *parent;
	char k;
	int symbol;
{
	ktree_t *p;

	if ((p = (ktree_t *) calloc(sizeof(ktree_t), 1)) == NULL) {
		perror("Unable to allocate memory chunk");
		return NULL;
	}

	p->key = k;
	p->symbol = symbol;
	p->parent = parent;
	
	if (parent) {
		p->next = parent->child;
		parent->child = p;
	}
	
	return p;
}

int
ktree_init(void)
{
	kqueue_entry *kp_ptr, **kp_table, *kp_table_ptr;
	ktree_t *kt_ptr, *kt_tp;
	char *t_ptr;
	kqueue_entry_c *kb_ptr;
	int n;

	/*
	 * We need to allocate enough memory for the shift+tab entry, the ('z' 
	 * - 'a') for the alt keys, and the borked keys.
	 */

	if ((kp_table = (kqueue_entry **) malloc((('z' - 'a') +
				(sizeof(k_list) / sizeof(*k_list)) +
				(sizeof(k_bork) / sizeof(*k_bork))) *
				sizeof(kqueue_entry))) == NULL) {
		perror("Unable to allocate memory chunk");
		return -1;
	}

	kp_table_ptr = (kqueue_entry *) &kp_table[0];

	/* Grab the necessary strings from termcap:
	 */
	for (kp_ptr = &k_list[0]; kp_ptr->arr_code; ++kp_ptr) {
		kp_table_ptr->tc_code = tinfo_getstr(tc_entry, kp_ptr->tc_off);
		kp_table_ptr->arr_code = kp_ptr->arr_code;
		++kp_table_ptr;
	}

	/* Copy in the borked keys:
	 */
	for (kb_ptr = &k_bork[0]; kb_ptr->str; kb_ptr++) {
		kp_table_ptr->tc_code = (char *)kb_ptr->str;
		kp_table_ptr->arr_code = kb_ptr->arr_code;
		++kp_table_ptr;
	}

	/* Alt keys:
	 */
	for (n = ('z' - 'a'); (n >= 'a'); --n) {
		/* XXX this malloc(3) is never free()'d.  Do we *really* care?
		 */
		if ((kp_table_ptr->tc_code = (char *)malloc(3 * sizeof(char)))
				== NULL) {
			perror("Unable to allocate memory chunk");
			return -1;
		}

		kp_table_ptr->arr_code = (MODIFIER_ALT | n);
		kp_table_ptr->tc_code[0] = '\033';
		kp_table_ptr->tc_code[1] = n;
		kp_table_ptr->tc_code[2] = 0;
		++kp_table_ptr;
	}

	if (!(tree_head = create_node((ktree_t *)NULL, 0, 0)))
		return -1;

	for (kp_ptr = (kqueue_entry *)&kp_table[0]; kp_ptr->tc_code; kp_ptr++) {
		if (*kp_ptr->tc_code == NUL)
			continue;

		t_ptr = kp_ptr->tc_code;

		kt_ptr = tree_head; 
		while ((*t_ptr) && (kt_tp = find_node(kt_ptr, *t_ptr)) != NULL){
			kt_ptr = kt_tp;
			t_ptr++;
		}
		
		if (*t_ptr == NUL)
			continue;

		for (; *t_ptr != NUL; kt_ptr = kt_tp)
			kt_tp = create_node(kt_ptr, *t_ptr++, 0);

		kt_tp->symbol = kp_ptr->arr_code;
	}
	
	free(kp_table);

	return 0;
}
