\ henon.4th
\
\ Compute the Henon map, its fixed point(s), the Jacobian,
\   and eigenvalues and eigenvectors. Use analytic methods
\   and numerical methods.
\
\ Copyright (c) 2002, Krishna Myneni, 2-12-2002
\ Provided under the GNU General Public License
\ 
\ Requires:
\
\	matrix.4th
\	quadratic.4th
\
\ Revisions:
\	2002-03-18  added storage for evolution of initial conditions  km
\

(

The Henon map is a two-dimensional map which can exhibit chaos.
It is given by:

	x_n+1 = B*y_n - {x_n}^2 + A
	y_n+1 = x_n 

A and B are parameters, which for certain selections, give rise
to a chaotic sequence of values for x_n, y_n.

Here, we use A = 1.4, B = 0.3, which give chaos. The user
may experiment with other values of A and B as well. To observe 
the chaotic sequence, select an initial point such as 
x = 0.5, y = 0.5; then apply the map repeatedly.


Example:	0.5e 0.5e H 

	Applies the map once and returns the next x and y on the stack.
	The new x and y values may be printed by typing:

		fswap f. f.

-------
Example:	0.5e 0.5e 10 times

	applies the map 10 times and shows the new point at each iteration.
	To see the chaotic "attractor", iterate the map a large number of
	times and redirect the output to a file: 

		>file henon.dat 0.5e 0.5e 1000 times console

	The x, y pairs may then be plotted using a program such as xyplot.
	Plot the data as discrete points, rather than connected by line
	segments, to see the attractor properly.
 
-------
Example:	solve_fixed_point

	finds the point x, y at which the sequence remains stationary.
	For example,

		solve_fixed_point set_fixed_point
		xf f@ yf f@ 20 times

	shows that the map stays at the fixed point. However, typing

		xf f@ yf f@ 200 times

	shows that the sequence become chaotic once again. This is because the
	fixed point is "unstable". Small displacements about the actual fixed 
	point cause the sequence to move away rapidly from the fixed point.
	Since our solution of the fixed point is only to finite precision,
	we have in effect a displacement from the actual fixed point.  
	'solve_fixed_point' uses an analytic method specific to the Henon 
	map to determine the fixed point. A numerical method for finding 
	the fixed point may also be used:

		x y search_fixed_point

	where x and y are an initial starting point for the search
	routine. As the map is iterated, the same point must be seen
	successively 5 times, to within a specified tolerance, to be
	considered the fixed point. The search may take a long time,
	depending on the initial point which is specified.

-------
Example:	solve_fixed_point set_fixed_point
		xf f@ yf f@ evolve_ball

	A ball of points around the fixed point are followed after
	one iteration of the map. It can be seen that the ball of
	"initial conditions" transforms to an ellipse, contracting
	in one direction and expanding along another direction.
	After executing the above instructions, type

		>file ellipse.dat ellipse fmat. console

	to write the evolved x,y pairs to a file. Plot the data
	to observe the ellipse.

-------
Example:	solve_fixed_point set_fixed_point
		xf f@ yf f@ Jacobian
		eigenvalues
		eigenvectors
		e fmat.

	computes and prints the stable and unstable eigenvectors
	at the fixed point. The eigenvectors give the directions
	along which the fixed point is stable and unstable. First
	column is the stable direction, second column is unstable
	direction.
)


include matrix
include quadratic

fvariable A	1.4e A f!
fvariable B     0.3e B f!


: H ( x y -- xn yn | apply the Henon map to the input state)
	fover 2>r
	B f@ f*			\ B*y_n
	fswap fdup f* f- 	\ B*y_n - x_n^2
	A f@ f+ 2r> ;

\ Iterate the Henon map n times starting from the given point
\   (initial condition) and print the x and y values at each
\   iteration. The output data can be sent to a file for
\   plotting, e.g. '>file henon.dat .5e .8e 1000 times console'

: times ( x0 y0 n -- x_n y_n | iterate the map n times and show the results)
	0 ?do
	  fover f. 2 spaces fdup f. cr
	  H
	loop ;

\ Fixed point methods

fvariable xf
fvariable yf

: set_fixed_point ( x y -- )
	yf f! xf f! ;

\ ====== Analytical solution of fixed point for the Henon map

\ The equation to solve is -x^2 + (B-1)*x + A = 0

: solve_fixed_point ( -- x y )
	-1e  B f@ 1e f- A f@ solve_quadratic
	fdrop  \ assume real root and take only the positive root
	fdup f0< if fdrop fdrop else fswap fdrop fswap fdrop then fdup 
;

\ ======= Numerical method for finding the fixed point.

\ Starting with initial condition x0 and y0, iterate the Henon map
\   to find the unstable fixed point.

1e-4 fconstant tolerance

fvariable x 
fvariable y
variable seen

: some_where ( -- )   0 seen ! ;
: seen_again ( -- )   1 seen +! ;

: remember_this_point ( x y -- x y ) fdup y f! fover x f! ;

: distance_to_last ( x y -- x y dist)
	fover fover y f@ f- fdup f* fswap x f@ f- fdup f* f+ fsqrt ;

: same_point? ( x y -- flag | true if same as last point)
	distance_to_last tolerance f< ;

: search_fixed_point ( x0 y0 -- xf yf )
	some_where
	begin
	  remember_this_point	
	  H			\ apply the map once
	  same_point?
	  if seen_again else some_where then
	  seen @ 5 =
	until ;


\ ======= Analytical methods for determining stability of the fixed point

2 2 fmatrix DM

: Jacobian ( x y -- | compute the Jacobian at point x,y)
	fdrop          ( Jacobian for Henon Map does not depend on y or A)
	-2e f* 1 1 DM fmat!
	B f@   1 2 DM fmat!
	1e     2 1 DM fmat!
	0e     2 2 DM fmat!  ;

\ Compute the eigenvalues of the real 2x2 Jacobian matrix.
\ Assumes the eigenvalues are real, which they are for this example.

fvariable lambda1
fvariable lambda2

: eigenvalues ( -- | find eigenvalues of 2x2 Jacobian matrix)
	1e
	1 1 DM fmat@ 2 2 DM fmat@ f+ fnegate
	1 1 DM fmat@ 2 2 DM fmat@ f* 
	1 2 DM fmat@ 2 1 DM fmat@ f* f-
	solve_quadratic
	fdrop lambda2 f!
	fdrop lambda1 f! 

	\ order the eigenvalues so that lambda1 has smaller magnitude;
	\   for the fixed point, lambda1 will be the stable eigenvalue
	\   and lambda2 will be the unstable eigenvalue

	lambda1 f@ fabs lambda2 f@ fabs f>
	if lambda1 f@ lambda2 f@ lambda1 f! lambda2 f! then 
;	


2 2 fmatrix e  \ eigenvector matrix

: eigenvectors ( -- | use only after finding eigenvalues)

	\ Place the two eigenvectors in the columns of e

	\ Compute eigenvector for lambda1
	1 2 DM fmat@ 1 1 DM fmat@ lambda1 f@ f- f/ fnegate
	1 1 e fmat! 
	1e 2 1 e fmat!
	
	\ Compute eigenvector for lambda2
	1 2 DM fmat@ 1 1 DM fmat@ lambda2 f@ f- f/ fnegate
	1 2 e fmat!
	1e 2 2 e fmat!

	\ Normalize the vectors
	1 e fcol@ drop fdup f* fswap fdup f* f+ fsqrt 2>r
	1 e fcol@ drop 2r@ f/ fswap 2r> f/ fswap 2 1 e fcol!
	2 e fcol@ drop fdup f* fswap fdup f* f+ fsqrt 2>r
	2 e fcol@ drop 2r@ f/ fswap 2r> f/ fswap 2 2 e fcol! 
;


\ ======== Calculations on a ball of initial conditions

: dist ( x1 y1 x2 y2 -- d | distance between two points)
	frot f- fdup f* 2>r f- fdup f* 2r> f+ fsqrt ;


fvariable theta
1e-3 fconstant ball_radius  ( chosen specifically for this map )

360 2 fmatrix ellipse

\ The ellipse matrix may be saved to a two column ascii file using:
\
\	>file filename ellipse fmat. console

 
: evolve_ball ( x y -- | evolve a ball of initial conditions about a point)
	\ store the evolved points in the matrix ellipse
	361 1 do
	  fover fover
	  i 1- s>f deg>rad theta f!
	  fswap theta f@ fcos ball_radius f* f+	\ starting x
	  fswap theta f@ fsin ball_radius f* f+	\ starting y
	  H
	  2 i ellipse frow!	\ store the evolved point
	loop 
	fdrop fdrop ;


: major_axis? ( -- theta_major flength | determine major axis of ellipse)
	0 0e
	181 1 do
	  i ellipse frow@ drop
	  i 180 + ellipse frow@ drop
	  dist fover fover
	  f< if fswap fdrop rot drop i 1- -rot else fdrop then
	loop
	rot s>f deg>rad fswap ;


: minor_axis? ( -- theta_minor flength | determine minor axis of ellipse)
	0 1e100
	181 1 do
	  i ellipse frow@ drop
	  i 180 + ellipse frow@ drop
	  dist fover fover
	  f> if fswap fdrop rot drop i 1- -rot else fdrop then
	loop
	rot s>f deg>rad fswap ;  



