\ ANS Forth Complex Arithmetic Lexicon
\
\ --------------------------------------------------------------
\             Copyright (C) 1998  Julian V. Noble              \
\                                                              \
\ This library is free software; you can redistribute it       \
\ and/or modify it under the terms of the GNU Lesser General   \
\ Public License as published by the Free Software Foundation; \
\ either version 2.1 of the License, or at your option any     \
\ later version.                                               \
\                                                              \
\ This library is distributed in the hope that it will be      \
\ useful, but WITHOUT ANY WARRANTY; without even the implied   \
\ warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR      \
\ PURPOSE.  See the GNU Lesser General Public License for more \
\ details.                                                     \
\                                                              \
\ You should have received a copy of the GNU Lesser General    \
\ Public License along with this library; if not, write to the \
\ Free Software Foundation, Inc., 59 Temple Place, Suite 330,  \
\ Boston, MA 02111-1307 USA.                                   \
\ --------------------------------------------------------------

\ Environmental dependences:
\       1. requires FLOAT and FLOAT EXT wordsets
\
\       2. THIS VERSION ASSUMES INTEGRATED FP AND DATA STACK
\
\       3. does not construct a separate complex number stack

\ Complex numbers x+iy are stored on the stack as ( x y -- ).
\ Angles are in radians.
\ Polar representation measures angle from the positive x-axis.

\ All Standard words are in uppercase, non-Standard words in lowercase,
\ as a convenience to the user.

\ Version 0.8.2
\  5Mar03  * Release with changes under 0.8.1  Passes all our
\            tests, except for a few "exotic" signed zero
\            properties.
\ 14May03  * Adapted for use with kForth (K. Myneni):
\	     -- PRINCIPAL_ARG is set to TRUE by default, rather than 
\               being set to FALSE as in DNW's version. Conditional
\               tests for PRINCIPAL-ARG have been commented out.
\            -- Substituted "D=" for "0e F~" in the definition of
\               PSQRT;  WARNING: this is a kForth specific change!
\	     -- The Kahan versions of zsinh and ztanh are not used by
\               default as in DNW's version, because FSINH and FCOSH
\               are not currently implemented in kForth.
\            -- Appropriate mods were made for integrated fp/data stack
\               operation, including changes to stack diagrams.
\
\ Version 0.8.1
\ 21Feb03  * Rewrote PSQRT for stability on the branch cut.
\          * Added X+ and X-.  Used to make ZASINH, ZACOSH,
\            ZATANH, and ZACOTH valid on their cuts, labeled by
\            signed zero.
\          * Passed complex-test.fs on MacOS X, including signed
\            zero and branch cut tests.
\ 22Feb03  * Replaced Z. and ZS. with jvn's code for Krishna
\            Myneni's suggestion to factor out the sign of the
\            imaginary part.

\ Version 0.8.0
\ 15Dec02  * Started revision of jvn's complex.f.
\ 20Feb03  * Release.                

\ The basic modifications have been a few changes in
\ floating-point alignment and the completion of the definitions
\ of the inverse functions.

\ The definitions here coincide with Abramowitz and Stegun [1],
\ Kahan [2], and the OpenMath standard [3], which produce
\ principal branches given principal branches for square roots
\ and natural logs.  The formulas, or "principal expressions",
\ are selected from [3], with choices among equivalent, formally
\ correct possibilities based mainly on computational
\ conciseness, with a nod to numerical stability where the
\ authors mention it.  Those authors do not claim to analyze
\ numerical stability, but Kahan does, and we implement his
\ algorithms in Forth in a separate file.

\ The original Noble code uses the convention common among
\ physicists for branch cuts, with arguments between zero and
\ 2pi, especially for logs and noninteger powers.  The numerical
\ analysis community is pretty unanimous about using principal
\ branches instead.

\ Everybody seems to agree on the nontriviality of the branch
\ cuts for the inverse functions and to follow Abramowitz and
\ Stegun, who define them in terms of principal branches. 
\ In this code we include a PRINCIPAL-ARG switch to select
\ between the two common conventions for arg, log, and
\ nonintegral powers, but we use only principal arguments for
\ the inverse functions.

\ Kahan pays attention to signed zero, where available in IEEE
\ 754/854 implementations.  We address that a couple of ways in
\ this file.  One is to provide uncommentable versions of ZSINH
\ and ZTANH which respect the sign of zero.  The other is to
\ write the functions having branch cuts so that signed zero in
\ the appropriate x or y input produces correct values on the
\ cuts.

 1.570796326794896619231E FCONSTANT pi/2
 6.283185307179586476925E FCONSTANT 2pi

\ The PRINCIPAL-ARG flag controls conditional compilation
\ with/without the principal argument for FATAN2, ARG, >POLAR,
\ ZSQRT, ZLN, and Z^.  The inverse functions are always defined
\ with principal arguments, in accord with Abramowitz and
\ Stegun [1], Kahan [2], and OpenMath [3].

\  false CONSTANT PRINCIPAL-ARG  \ output  0 <= arg <  2pi
  true  CONSTANT PRINCIPAL-ARG  \ output -pi < arg <= pi
  IMMEDIATE

\ In ANS Forth FATAN2 is ambiguous.  The following assumes it
\ either gives -pi < arg < pi (or maybe <=) or 0 <= arg < 2pi. 
\ Then PARG is defined to force the principal arg for later use
\ in the forced principal arg function PLN, and FATAN2 is
\ redefined according to the PRINCIPAL-ARG flag for later use in
\ defining ARG.

-1E 0E ( y x) FATAN2 F0<

( arg<0)  DROP \ DUP
\ [IF]    \ FATAN2 principal
  : parg  ( x y -- princ.arg )  FSWAP FATAN2 ;
\ [ELSE]  \ FATAN2 not principal arg
\  : parg  ( x y -- princ.arg )
\      FDUP F0< >R FSWAP FATAN2
\      ( y<0) R> IF 2pi F- THEN ; 
\ [THEN]

\ PRINCIPAL-ARG [IF]
\  ( arg<0) 0= [IF]  \ FATAN2 not principal, redefine
\  : fatan2    FOVER ( y x y) F0< >R FATAN2
\      ( y<0) R> IF 2pi F- THEN ;
\  [THEN]
\ [ELSE]
\  ( arg<0) [IF]     \ FATAN2 principal, redefine
\  : fatan2    FOVER ( y x y) F0< >R FATAN2
\      ( y<0) R> IF 2pi F+ THEN ;
\  [THEN]
\ [THEN]


\ ----------- kForth Requirements-------------------
: FSINCOS ( f -- fsin fcos ) FDUP FSIN FSWAP FCOS ; 
: FLOAT+ DFLOAT+ ;
: FLOATS DFLOATS ;
\ ----------- end of kForth Requirements ------------

: f2/ 2e F/ ;
: f2* 2e F* ;

1.0E0  FCONSTANT  f1.0
0.0E0  FCONSTANT  f0.0

: f-rot    FROT   FROT  ;
: fnip     FSWAP  FDROP  ;
: ftuck    FSWAP  FOVER ;

: 1/f   F1.0  FSWAP  F/ ;
: f^2   FDUP  F*  ;

\ ---------------------------------------- LOAD, STORE
: z@  DUP  F@  ROT FLOAT+  F@ ;     ( adr -- z)
: z!  DUP  >R FLOAT+  F!  R> F! ;   ( f adr -- z)

: zvariable   CREATE   2 FLOATS  ALLOT  ;

: zconstant   FSWAP  CREATE  2 FLOATS  ?allot  
	      >R  R@  F!  R>  FLOAT+  F!  
	      DOES>  >R  R@  F@  R>  FLOAT+  F@  ;


\ ---------------------------------------- MANIPULATE STACK

: z.  FSWAP F.    ( x y -- )    \ emit complex #
  FDUP F0<
  DUP INVERT [CHAR] + AND   SWAP [CHAR] - AND   +
  EMIT ."  i " FABS F. ;


: z=0  f0.0 f0.0 ;                 ( -- 0 0)
: z=1  f1.0 f0.0 ;                 ( -- 1 0)
: z=i  z=1 FSWAP ;                 ( -- 0 1)
: zdrop  FDROP FDROP ;             ( x y --)
: zdup   FOVER FOVER ;             ( x y -- x y x y)

\ temporary storage for stuff from stack

CREATE noname   3 FLOATS  ALLOT     \ ALLOT z variable

: zswap    ( x y u v -- u v x y)
    [ noname ] LITERAL  F!  f-rot
    [ noname ] LITERAL  F@  f-rot  ;

: zover     ( x y u v -- x y u v x y )
    FROT    [ noname FLOAT+ ]  LITERAL  F!   ( -- x u v)
    FROT FDUP   [ noname    ]  LITERAL  F!   ( -- u v x)
    f-rot   [ noname FLOAT+ ]  LITERAL  F@   ( -- x u v y)
    f-rot   [ noname        ]  LITERAL  z@   ( -- x y u v x y)
;

: real    FDROP ;
: imag    fnip  ;
: conjg   FNEGATE ;


: znip     zswap  zdrop ;
: ztuck    zswap  zover ;

: z=       ( x y u v -- flag )
    FROT F= >R F= R> AND ;

: z*f      ( x y a -- x*a y*a)
    FROT  FOVER  F*  f-rot  F*  ;

: z/f      ( x y a -- x/a y/a)
    1/f   z*f  ;

: z*    ( x y u v -- x*u-y*v  x*v+y*u)
\ uses the algorithm
\       (x+iy)*(u+iv) = [(x+y)*u - y*(u+v)] + i[(x+y)*u + x*(v-u)]
\       requiring 3 multiplications and 5 additions
  
        zdup F+                         ( x y u v u+v)
        [ noname ] LITERAL  F!          ( x y u v)
        FOVER F-                        ( x y u v-u)
        [ noname FLOAT+ ] LITERAL F!    ( x y u)
        FROT FDUP                       ( y u x x)
        [ noname FLOAT+ ] LITERAL F@    ( y u x x v-u)
        F*
        [ noname FLOAT+ ] LITERAL F!    ( y u x)
        FROT FDUP                       ( u x y y)
        [ noname ] LITERAL F@           ( u x y y u+v)
        F*
        [ noname ] LITERAL F!           ( u x y)
        F+  F* FDUP                     ( u*[x+y] u*[x+y])
        [ noname ] LITERAL F@ F-        ( u*[x+y] x*u-y*v)
        FSWAP
        [ noname FLOAT+ ] LITERAL F@    ( x*u-y*v u*[x+y] x*[v-u])
        F+ ;                            ( x*u-y*v x*v+y*u)

: z+   FROT F+  f-rot F+ FSWAP ;  ( a b x y -- a+x b+y)

: znegate  FSWAP FNEGATE FSWAP FNEGATE ;

: z-  znegate  z+ ;

\ to avoid unneeded calculations on the other part that could
\ raise gratuitous overflow or underflow signals and changes in
\ the sign of zero (Kahan)
: x+  ( x y a -- x+a y )  frot f+ fswap ;
: x-  ( x y a -- x-a y )  fnegate frot f+ fswap ;
\ : y+  ( x y a -- x y+a )  f+ ;
\ : y-  ( x y a -- x y-a )  f- ;

: |z|^2   f^2  FSWAP  f^2  F+  ;  ( z -- f)

\ writing |z| and 1/z as shown reduces overflow probability
: |z|   ( x y -- |z|)
    FABS  FSWAP  FABS
    zdup  FMAX
    FDUP F0= IF
      FDROP zdrop 0E
    ELSE  
      f-rot  FMIN     ( max min)
      FOVER  F/  f^2  1e0  F+  FSQRT  F*  
    THEN ;

: 1/z   fnegate  zdup  |z|  1/f  FDUP  [ noname ] LITERAL F!
        z*f  [ noname ] LITERAL  F@  z*f  ;

: z/    1/z  z* ;
: z2/   f2/  FSWAP  f2/  FSWAP  ;
: z2*   f2*  FSWAP  f2*  FSWAP  ;

: arg   ( x y -- arg[x+iy] )  FSWAP fatan2 ;

: >polar  ( x+iy -- r phi )  zdup  |z|  f-rot  arg  ;
: polar>  ( r phi -- x+iy )  FSINCOS FROT  z*f   FSWAP  ;

: i*      FNEGATE FSWAP ;  ( x+iy -- -y+ix)
: (-i)*   FSWAP FNEGATE ;  ( x+iy -- y-ix)

: pln   ( z -- ln[z].prin )
    zdup parg f-rot |z| FLN FSWAP ;

: zln   ( z -- ln[|z|]+iarg[z] )
    >polar   FSWAP  \ FDUP  F0=  ABORT" Can't take ZLN of 0"  
    FLN   FSWAP ;

: zexp   ( z -- exp[z] )   FSINCOS  FSWAP FROT  FEXP  z*f ;

: z^2   zdup  z*  ;
: z^3   zdup  z^2  z* ;
: z^4   z^2  z^2  ;

: z^n      ( z n -- z^n )    \ raise z to integer power
\ Use Z^ instead for n > 50 or so.
       >R  z=1   zswap  R>
       BEGIN   DUP  0>  WHILE  >R
               R@  1 AND   IF ztuck  z*  zswap THEN z^2
               R>  2/
       REPEAT  DROP  zdrop ;

: z^   ( x y u v --  [x+iy]^[u+iv] )  zswap zln  z* zexp  ;

\ WARNING: The following def of PSQRT is kForth specific
\   in its use of D= to compare *exactly* two fp numbers on
\   the stack. The D= can be replaced by "0e F~" for generality.

: psqrt  ( z -- sqrt[z].prin )
(
Kahan's algorithm without overflow/underflow avoidance and
without treating infinity.  But it should handle signed zero
properly.
)
    zdup [ noname FLOAT+ ] LITERAL ( y a) F!
         [ noname ] LITERAL ( x a) F!
    |z| [ noname ] LITERAL F@
        fabs f+ f2/ fsqrt                 ( rho=sqrt[[|z|+|x|]/2])
    fdup f0= >R
        [ noname FLOAT+ ] LITERAL F@      ( rho y)
    R> 0= IF  \ rho <> 0
      fover f/ f2/                        ( rho eta=[y/rho]/2)
      [ noname ] LITERAL F@
      f0< IF  \ x < 0
        fabs fswap                        ( |eta| rho)
        [ noname FLOAT+ ] LITERAL F@
        FDUP F0<  >R  -0e  D= ( 0e F~)  R> OR
        IF \ y < 0
          FNEGATE THEN                    ( |eta| cps[rho,y])
      THEN
    THEN ;

\ PRINCIPAL-ARG [IF]
: zsqrt   psqrt ;
\ [ELSE]
\ : zsqrt   ( x y -- a b )     \ (a+ib)^2 = x+iy
\     zdup                               ( -- z z)
\     |z|^2                              ( -- z |z|^2 )
\     FDUP  F0=   IF   FDROP EXIT  THEN  ( -- z=0 )
\     FSQRT FROT  FROT  F0<  >R          ( -- |z| x )  ( -- sgn[y])
\     ftuck                              ( -- x |z| x )
\     F-  F2/                            ( -- x [|z|-x]/2 )
\     ftuck  F+                          ( -- [|z|-x]/2 [|z|+x]/2 )
\     FSQRT  R>  IF  FNEGATE  THEN       ( -- [|z|-x]/2  a )
\     FSWAP  FSQRT   ;                   ( -- a b)
\ [THEN]                   

\ Complex trigonometric functions

\ All stack patterns are ( z -- func[z] ).

: zsinh    zexp   zdup   1/z   z-  z2/  ;
\ This version preserves signed zero.
\ : zsinh
\    FSINCOS [ noname ] LITERAL ( cos[y]) F!
\        [ noname FLOAT+ ] LITERAL ( sin[y]) F!
\    FDUP FSINH [ noname ] LITERAL F@ F*   ( x sh[x]cos[y])
\    FSWAP FCOSH
\        [ noname FLOAT+ ] LITERAL F@ F*   ( sh[x]cos[y] ch[x]sin[y])
\ ;

: zcosh    zexp   zdup   1/z   z+  z2/  ;


: ztanh    zexp  z^2    i*   zdup   f1.0 F-   zswap   f1.0 F+   z/  ;
\ This version, based on Kahan, preserves signed zero.
\ : ztanh
(
            [1 + tan^2[y]] cosh[x] sinh[x] + i tan[y]
  tanh[z] = -----------------------------------------
                 1 + [1 + tan^2[y]] sinh^2[x]
)
\    FTAN FDUP f^2 1E F+                 ( f: x t=tan[y] b=1+t^2)
\        [ noname ] LITERAL F!
\    FSWAP FDUP FSINH                    ( f: t x sh[x])
\        [ noname FLOAT+ ] LITERAL F!
\    FCOSH
\        [ noname FLOAT+ ] LITERAL F@    ( f: t ch[x] sh[x])
\    F* [ noname ] LITERAL F@ F* FSWAP   ( f: c=ch[x]sh[x]b t)
\    [ noname ] LITERAL F@
\        [ noname FLOAT+ ] LITERAL F@
\        f^2 F* 1E F+                    ( f: c t 1+b*sh^2[x])
\    z/f ;

: zcoth    ztanh  1/z ;

: zsin     i*  zsinh  (-i)* ;
: zcos     i*  zcosh  ;
: ztan     i*  ztanh  (-i)* ;
: zcot     i*  zcoth  i* ;


\ Complex inverse trigonometric functions
\ In the following, we use phrases like "1E x+" instead of
\ "z=1 z+", for stability on branch cuts involving signed zero.
\ This follows a suggestion by Kahan [2], and it actually makes
\ a difference in every one of the functions.

: zasinh   ( z -- ln[z+sqrt[[z+i][z-i]] )
\ This is more stable than the version with z^2+1.
     zdup 1E F+   zover 1E F-   z* psqrt z+ pln ;

: zacosh   ( z -- 2ln[sqrt[[z+1]/2]+sqrt[[z-1]/2] )
    zdup  1E x- z2/ psqrt
    zswap 1E x+ z2/ psqrt
    z+ pln z2* ;

: zatanh   ( z -- [ln[1+z]-ln[1-z]]/2 )
    zdup  1E x+ pln
    zswap 1E x- znegate pln
    z- z2/ ;

: zacoth  ( z -- [ln[-1-z]-ln[1-z]]/2 )
    znegate zdup 1E x- pln
    zswap 1E x+ pln
    z- z2/ ;

: zasin   ( z -- -iln[iz+sqrt[1-z^2]] )    i* zasinh (-i)* ;
: zacos   ( z -- pi/2-asin[z] )     pi/2 0E zswap zasin z- ;
: zatan   ( z -- [ln[1+iz]-ln[1-iz]]/2i )  i* zatanh (-i)* ;
: zacot   ( z -- [ln[[z+i]/[z-i]]/2i )  (-i)* zacoth (-i)* ;


\ ------------------------------------------ for use with ftran2xx.f

: cmplx   ( x 0 y 0 -- x y)  FDROP  FNIP  ;
\ ------------------------------------------

\ Rudimentary testing: ( answers verified by hand calculation -- km)
\
\ 	zvariable c1		( define complex variable c1)
\	1e 0e c1 z!		( store 1+i0 in c1)
\	z=1 c1 z!		( equivalent to previous line)
\	c1 z@ z.		( prints 1 + i 0)
\
\	2e 3e c1 z!		( store 2+i3 in c1)
\	c1 z@ conjg z.		( 2 - i 3)
\	c1 z@ znegate z.	( -2 - i 3)
\	c1 z@ i* z.		( -3 + i 2)
\	c1 z@ (-i)* z.		( 3 - i 2)
\	c1 z@ z^2 z.		( -5 + i 12) 
\	c1 z@ zsqrt z.		( 1.67415 + i 0.895977)
\	c1 z@ |z|^2 f.		( 13)
\	c1 z@ |z| f.		( 3.60555)
\	c1 z@ 7 z^n z.		( 6554 + i 4449)
\	c1 z@ 2e z*f z.		( 4 + i 6)
\	c1 z@ 2e z/f z.		( 1 + i 1.5)
\	c1 z@ 1/z z.		( 0.153846 - i 0.230769)
\	c1 z@ arg f.		( 0.982794)
\	c1 z@ >polar f. f.	( 0.982794 3.60555)
\	c1 z@ >polar polar> z.	( 2 + i 3)
\	c1 z@ zexp z.		( -7.31511 + i 1.04274)
\	c1 z@ zln z.		( 1.28247 + i 0.982794)
\
\	zvariable c2		
\	2e 3e c1 z!		
\	1e 5e c2 z!		
\	c1 z@ c2 z@ z+ z.	( add c1 and c2, prints 3 + i 8)
\	c1 z@ c2 z@ z- z.	( 1 - i 2)
\	c1 z@ c2 z@ z* z.	( -13 + i 13)
\	c1 z@ c2 z@ z/ z.	( 0.653846 - i 0.269231)
\

\ --------------------------------- REFERENCES
(
1. M. Abramowitz and I. Stegun, Handbook of Mathematical
   Functions with Formulas, Graphs, and Mathematical Tables, US
   Government Printing Office, 10th Printing December 1972,
   Secs. 4.4, 4.6.

2. William Kahan, "Branch cuts for complex elementary
   functions", The State of the Art in Numerical Analysis, A.
   Iserles and M.J.D. Powell, eds., Clarendon Press, Oxford,
   1987, pp. 165-211.

3. Robert M. Corless, James H. Davenport, David J. Jeffrey,
   Stephen M. Watt, "'According to Abramowitz and Stegun' or
   arcoth needn't be uncouth", ACM SIGSAM Bulletin, June, 2000,
   pp. 58-65.
)
