#!/usr/bin/perl -w
#
# This script may not work with irssi older than 0.7.98.CVS 20011201.

use strict;
use vars qw($VERSION %IRSSI);

$VERSION = "2.3";
%IRSSI = (
    authors     => 'Jakub Jankowski, Erkki Seppl',
    contact     => 'shasta@atn.pl, flux@inside.org',
    name        => 'friends',
    description => 'Maintains list of people you know.',
    license     => 'GNU GPLv2 or later',
    url         => 'http://irssi.atn.pl/friends/',
);

use Irssi 20011201.0100 ();
use Irssi::Irc;

# friends.pl
my $friends_version = $VERSION . " (20020304)";

# release note, if any
my $release_note = "/QUEUE FLUSH contains some experimental code.";

##############################################
# These variables are adjustable with /set
# but here are some 'safe' defaults:

# do you want to process CTCP queries?
my $friends_use_ctcp = 1;

# space-separated list of allowed (implemented ;) CTCP commands
my $friends_ctcp_commands = "OP VOICE LIMIT KEY INVITE PASS";

# do you want to learn new users?
my $friends_learn = 0;

# do you want to autovoice already opped nicks?
my $friends_voice_opped = 0;

# which flags do you want to use? (case *sensitive*)
my $friends_allowed_flags = "aoviklLFdrm";

# which flags do you want to add automatically with /addfriend? (case *sensitive*)
my $friends_default_flags = "";

# path to friendlist
my $friends_file = "$ENV{HOME}/.irssi/friends";

# do you want to save friendlist every time irssi's setup is saved
my $friends_autosave = 0;

# do you want to show friend's flags while he joins a channel?
my $friends_flags_on_join = 0;

# maximum size of operationQueue
my $friends_max_queue_size = 20;

# default debug level
# higher value => more messages
# 0 means off, 8 is max
my $friends_debug_level = 0;

# do you want to use dedicated debug window?
my $friends_use_debug_window = 1;

# min delaytime
my $default_delay_min = 10;

# max delaytime
my $default_delay_max = 60;

###############################################################

# registering themes
Irssi::theme_register([
	'friends_empty',		'Your friendlist is empty. Add items with /ADDFRIEND',
	'friends_notenoughargs',	'Not enough arguments. Usage: $0',
	'friends_badargs',		'Bad arguments. Usage: $0',
	'friends_nosuch',		'No such friend %R$0%n',
	'friends_endof',		'End of $0 $1',
	'friends_badhandle',		'Wrong handle: %R$0%n. $1',
	'friends_notuniqhandle',	'Handle %R$0%n already exists, choose another one',
	'friends_version',		'friends.pl\'s version: {hilight $0} [$1]',
	'friends_file_written',		'friendlist written on: {hilight $0}',
	'friends_file_version',		'friendlist written with: {hilight $0} [$1]',
	'friends_filetooold',		'Friendfile too old, loading aborted',
	'friends_loaded',		'Loaded {hilight $0} friends from $1',
	'friends_saved',		'Saved {hilight $0} friends to $1',
	'friends_duplicate',		'Skipping %R$0%n [duplicate?]',
	'friends_checking',		'Checking {hilight $0} took {hilight $1} secs [on $2]',
	'friends_line_head',		'[$[!-3]0] Handle: %R$1%n, flags: %C$2%n [password: $3]',
	'friends_line_hosts',		'$[-6]9 Hosts: $0',
	'friends_line_chan',		'$[-6]9 Channel {hilight $0}: Flags: %c$1%n, Delay: $2',
	'friends_line_comment',		'$[-6]9 Comment: $0',
	'friends_line_currentnick',	'$[-6]9 Current nick: {nick $0}',
	'friends_joined',		'{nick $0} is a friend, handle: %R$1%n, global flags: %C$2%n, flags for {hilight $3}: %C$4%n',
	'friends_queue_empty',		'Operation queue is empty',
	'friends_queue_line1',		'[$[!-2]0] Operation: %R$1%n secs left before {hilight $2}',
	'friends_queue_line2',		'     (Server: {hilight $0}, Channel: {hilight $1}, Nicklist: $2)',
	'friends_queue_nosuch',		'No such entry in operation queue ($0)',
	'friends_queue_removed',	'$0 queues: {hilight $1} [$2]',
	'friends_friendlist',		'{hilight Friendlist} [$0]:',
	'friends_friendlist_count',	'Listed {hilight $0} friend$1',
	'friends_findfriends',		'Online friends on channel {hilight $0} [on $1]:',
	'friends_added',		'Added %R$0%n to friendlist',
	'friends_removed',		'Removed %R$0%n from friendlist',
	'friends_comment_added',	'Added comment line to %R$0%n ($1)',
	'friends_comment_removed',	'Removed comment line from %R$0%n',
	'friends_host_added',		'Added {hilight $1} to %R$0%n',
	'friends_host_removed',		'Removed {hilight $1} from %R$0%n',
	'friends_host_exists',		'Hostmask {hilight $1} overlaps with one of the already added to %R$0%n',
	'friends_host_notexists',	'%R$0%n does not have {hilight $1} in hostlist',
	'friends_chanrec_removed',	'Removed {hilight $1} record from %R$0%n',
	'friends_chanrec_notexists',	'%R$0%n does not have {hilight $1} record',
	'friends_changed_handle',	'Changed {hilight $0} to %R$1%n',
	'friends_changed_delay',	'Changed %R$0%n\'s delay value on {hilight $1} to %c$2%n',
	'friends_chflagexec',		'Executing %c$0%n for %R$1%n ($2)',
	'friends_currentflags',		'Current {channel $2} flags for %R$1%n are: %c$0%n',
	'friends_chpassexec',		'Altered password for %R$0%n',
	'friends_ctcprequest',		'%R$0%n asks for {hilight $1} on {hilight $2}',
	'friends_ctcppass',		'Password for %R$0%n altered by $1',
	'friends_ctcpfail',		'Failed CTCP {hilight $0} from %R$1%n. $2',
	'friends_optree_header',	'Opping tree:',
	'friends_optree_line1',		'%R$0%n has opped these:',
	'friends_optree_line2',		'{hilight $0} times: $1',
	'friends_general',		'$0',
	'friends_notice',		'[%RN%n] $0',
	'friends_debug',		'DEBUG(%R$0%n): $1'
]);

# Initialize vars
my @friends = ();
my @operationQueue = ();
my $timerHandle = undef;
my $friends_file_version;
my $friends_file_written;
my $debug_win_name = "friends-debug";

# void print_version($what)
# print's version of script/userlist
sub print_version {
	my ($what) = @_;
	$what = lc($what);

	if ($what eq "filever") {
		if ($friends_file_version) {
			my ($verbal, $numeric) = $friends_file_version =~ /^(.+)\ \(([0-9]+)\)$/;
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_file_version', $verbal, $numeric);
		} else {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_empty');
		}
	} elsif ($what eq "filewritten" && $friends_file_written) {
		my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($friends_file_written);
		my $written = sprintf("%4d%02d%02d %02d:%02d:%02d", ($year+1900), ($mon+1), $mday, $hour, $min, $sec);
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_file_written', $written);
	} else {
		my ($verbal, $numerical) = $friends_version =~ /^(.+)\ \(([0-9]+)\)$/;
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_version', $verbal, $numerical);
	}
}

# void print_releasenote()
# suprisingly, prints a release note ;^)
sub print_releasenote {
	foreach my $line (split(/\n/, $release_note)) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notice', $line);
	}
}

# bool globMatcher ($what, $regex)
# glob matching
sub globMatcher ($$) {
	my ($what, $regex) = @_;

	debug("globMatcher(): what='$what', regex='$regex'", 7);

	$regex =~ s/\\/\\\\/g;
	$regex =~ s/\./\\\./g;
	$regex =~ s/\*/\.\*/g;
	$regex =~ s/\!/\\\!/g;
	# Lam's suggestion: from ircd's point of view, '?' means
	# ,,any, but always one, character''
	$regex =~ s/\?/\./g;
	$regex =~ s/\+/\\\+/g;
	$regex =~ s/\^/\\\^/g;
	$regex =~ s/\$/\\\$/g;
	$regex =~ s/\[/\\\[/g;
	$regex =~ s/\]/\\\]/g;
	$regex =~ s/\(/\\\(/g;
	$regex =~ s/\)/\\\)/g;
	$regex =~ s/\|/\\\|/g;
	my $matches = $what =~ /^$regex$/i;

	debug("globMatcher(): after quoting: regex='$regex' (".(($matches) ? "MATCHED" : "Not matched" ).")", 7);

	return $matches;
}

# void debug($text, $level)
# prints debug message if friends_debug_level >= $level
sub debug {
	my ($text, $level) = @_;
	my $debug_level = Irssi::settings_get_int('friends_debug_level');
	my $win = undef;

	return unless ($debug_level >= $level);

	if (Irssi::settings_get_bool('friends_use_debug_window')) {
		if (!($win = Irssi::window_find_name($debug_win_name))) {
			$win = Irssi::Windowitem::window_create("DEBUG", 1);
			$win->set_name($debug_win_name);
			$win->printformat(MSGLEVEL_CRAP, 'friends_debug', 0, "This window has been created for debugging purposes");
			$win->printformat(MSGLEVEL_CRAP, 'friends_debug', 0, "of friends.pl script, according to your settings.");
			$win->printformat(MSGLEVEL_CRAP, 'friends_debug', 0, "If you don't want this behaviour, RTFM! :)");
		}
	} else {
		$win = Irssi::active_win();
	}
	$win->printformat(MSGLEVEL_CRAP, 'friends_debug', $level, $text);
}

# str friends_crypt($plain)
# returns crypt()ed $plain, using random salt;
# or "" if $plain is empty
sub friends_crypt {
	my ($plain) = @_;
	if ($plain eq "") {
		debug("friends_crypt(): got empty password", 2);
		return;
	}
	return crypt("$plain", (join '', ('.', '/', 0..9, 'A'..'Z', 'a'..'z')[rand 64, rand 64]));
}

# bool friend_passwdok($idx, $pwd)
# returns 1 if password is ok, 0 if isn't
sub friends_passwdok {
	my ($idx, $pwd) = @_;
	if (crypt("$pwd", $friends[$idx]->{password}) eq $friends[$idx]->{password}) {
		debug("friends_passwdok(): Password for $friends[$idx]->{handle} is OK", 3);
		return 1;
	}
	debug("friends_passwdok(): Password for $friends[$idx]->{handle} not matched", 3);
	return 0;
}

# arr get_friends_channels($idx)
# returns list of $friends[$idx] channels
sub get_friends_channels {
	my ($idx) = @_;
	return keys(%{$friends[$idx]->{channels}});
}

# arr get_friends_hosts($idx)
# returns list of $friends[$idx] hosts
sub get_friends_hosts {
	my ($idx) = @_;
	return keys(%{$friends[$idx]->{hosts}});
}

# str get_friends_flags($idx[, $chan])
# returns list of $chan flags for $idx
# $chan can be 'global' or undef
# case insensitive about the $chan
sub get_friends_flags {
	my ($idx, $chan) = @_;
	$chan = lc($chan);
	if ($chan eq "" || $chan eq "global") {
		return $friends[$idx]->{globflags};
	} else {
		foreach my $friendschan (get_friends_channels($idx)) {
			if ($chan eq lc($friendschan)) {
				return $friends[$idx]->{channels}->{$friendschan}->{flags};
			}
		}
	}
	debug("get_friends_flags() failed for idx=$idx, chan=$chan", 5);
	return;
}

# str get_friends_delay($idx[, $chan])
# returns $chan delay for $idx
# returns "" if $chan is 'global' or undef
# case insensitive about the $chan
sub get_friends_delay {
	my ($idx, $chan) = @_;
	$chan = lc($chan);
	if ($chan && $chan ne "global") {
		foreach my $friendschan (get_friends_channels($idx)) {
			if ($chan eq lc($friendschan)) {
				return $friends[$idx]->{channels}->{$friendschan}->{delay};
			}
		}
	}
	debug("get_friends_delay() failed for idx=$idx, chan=$chan", 5);
	return;
}

# struct friend new_friend($handle, $hoststr, $globflags, $chanflagstr, $password, $comment)
# hoststr is: *!foo@host1 *!bar@host2 *!?baz@host3
# chanstr is: #chan1,flags,delay #chan2,flags,delay
sub new_friend {
	# $friend's structure:
	# $friend->{handle} = handle
	# $friend->{hosts}->{*!foo@host1} = 1
	# $friend->{hosts}->{*!bar@host2} = 1
	# $friend->{hosts}->{*!?baz@host3} = 1
	# $friend->{globflags} = oikl [global-flags]
	# $friend->{channels}->{#chan1}->{exist} = 1
	# $friend->{channels}->{#chan1}->{flags} = dk
	# $friend->{channels}->{#chan1}->{delay} = 5
	# $friend->{channels}->{#chan2}->{exist} = 1
	# $friend->{channels}->{#chan2}->{flags} = ao
	# $friend->{channels}->{#chan2}->{delay} = 
	# $friend->{password} = crypt(password)
	# $friend->{comment} = This is a string that must not contain any %
	# $friend->{friends} = []

	my $friend = {};
	$friend->{handle} = $_[0];
	$friend->{globflags} = $_[2];
	$friend->{password} = $_[4];
	$friend->{comment} = $_[5];
	$friend->{friends} = [];

	foreach my $host (split(/ +/, $_[1])) {
		$friend->{hosts}->{$host} = 1;
	}

	foreach my $cfd (split(/ +/, $_[3])) {
		# $cfd format: #foobar,oikl,15 (channelname,flags,delay)
		my ($channel, $flags, $delay) = split(",", $cfd, 3);
		$friend->{channels}->{$channel}->{exist} = 1;
		$friend->{channels}->{$channel}->{flags} = $flags;
		$friend->{channels}->{$channel}->{delay} = $delay;
	}

	return $friend;
}

# bool is_allowed_flag($flag)
# checks if $flag is in range of friends_allowed_flags
sub is_allowed_flag {
	my ($flag) = @_;
	my $allowed = Irssi::settings_get_str('friends_allowed_flags');
	return 1 if ($allowed =~ /\Q$flag\E/);

	debug("is_allowed_flag(): Flag $flag is out of allowed flags range ($allowed)", 5);
	return 0;
}

# bool is_ctcp_command($command)
# check if $command is one of the implemented ctcp commands
sub is_ctcp_command {
	my ($command) = @_;
	my $ctcp_commands = uc(Irssi::settings_get_str('friends_ctcp_commands'));
	$command = uc($command);
	foreach my $allowed (split(/[,\ ]+/, $ctcp_commands)) {
		return 1 if ($command eq $allowed);
	}
	debug("is_ctcp_command(): Command $command isn't one of the implemented (ctcp_commands)", 4);
	return 0;
}

# int get_idx($chan, $nick, $userhost)
# returns idx of the friend or -1 if not a friend
sub get_idx {
	my ($chan, $nick, $userhost) = @_;
	$chan = lc($chan);
	# browse through all friends
	for (my $idx = 0; $idx < @friends; ++$idx) {
		if (friend_has_host($idx, $nick.'!'.$userhost, 1) &&
			(get_friends_flags($idx, undef) || friend_has_chanrec($idx, $chan))) {
			debug("get_idx(): matched $nick!$userhost/$chan to idx=$idx", 6);
			return $idx;
		}
	}
	debug("get_idx(): failed on '$nick!$userhost/$chan'", 6);
	return -1;
}

# int get_idxbyhand($handle)
# returns $idx of friend with $handle or -1 if no such handle
# case insensitive
sub get_idxbyhand {
	my ($handle) = @_;
	$handle = lc($handle);
	for (my $idx = 0; $idx < @friends; ++$idx) {
		return $idx if (lc($friends[$idx]->{handle}) eq $handle);
	}
	debug("get_idxbyhand() failed on '$handle'", 6);
	return -1;
}

# int get_idxbyhost($nick, $userhost)
# returns $idx of friend with first matching $nick!$userhost
# or -1 if no such friend
# can be dangerous if used improperly :)
sub get_idxbyhost {
	my ($nick, $userhost) = @_;
	for (my $idx = 0; $idx < @friends; ++$idx) {
		foreach my $hostmask (get_friends_hosts($idx)) {
			return $idx if (globMatcher(($nick."!".$userhost), $hostmask));
		}
	}
	debug("get_idxbyhost() failed on '$nick!$userhost'", 4);
	return -1;
}

# bool friend_has_host($idx, $host, $mode)
# checks wheter any of $friend[$idx]'s hosts can be matched with $host
# case sensitive
# mode 1 = $host, $hostmask
# mode 2 = $hostmask, $host
sub friend_has_host {
	my ($idx, $host, $mode) = @_;
	my $success = 0;

	$mode = 1 unless ($mode);

	foreach my $hostmask (get_friends_hosts($idx)) {
		$success = 1 if ($mode == 1 && globMatcher($host, $hostmask));
		$success = 1 if ($mode == 2 && globMatcher($hostmask, $host));
		if ($success) {
			debug("friend_has_host(): Matched: $host and $hostmask", 6);
			return 1;
		}
	}
	return 0;
}

# void add_host($idx, $host)
# adds $host to $friends[$idx]->{hosts}
sub add_host {
	my ($idx, $host) = @_;
	$friends[$idx]->{hosts}->{$host} = 1;
	debug("add_host(): Added $host to $friends[$idx]->{handle}", 5);
}

# bool del_host($idx, $host)
# deletes $host from $friends[$idx]->{hosts}
sub del_host {
	my ($idx, $host) = @_;
	my $deleted = 0;
	foreach my $hostmask (get_friends_hosts($idx)) {
		if ($hostmask eq $host) {
			delete $friends[$idx]->{hosts}->{$hostmask};
			debug("del_host(): Deleted $host from $friends[$idx]->{handle}", 5);
			$deleted = 1;
		}
	}
	return $deleted;
}

# bool friend_has_chanrec($idx, $chan)
# checks wheter $friend[$idx] has a $chan record
# case insensitive
sub friend_has_chanrec {
	my ($idx, $chan) = @_;
	$chan = lc($chan);
	foreach my $friendschan (get_friends_channels($idx)) {
		if ($chan eq lc($friendschan)) {
			debug("friend_has_chanrec(): $friends[$idx]->{handle} has channel record for $chan", 7);
			return 1;
		}
	}
	debug("friend_has_chanrec(): $friends[$idx]->{handle} doesn't have channel record for $chan", 7);
	return 0;
}

# void add_chanrec($idx, $chan)
# adds an empty $chan record to $friends[$idx]
# case sensitive
sub add_chanrec {
	my ($idx, $chan) = @_;
	$friends[$idx]->{channels}->{$chan}->{exist} = 1;
	debug("add_chanrec(): Added $chan record to $friends[$idx]->{handle}", 5);
}

# bool del_chanrec($idx, $chan)
# deletes $chan record from $friends[$idx]
# case *in*sensitive
sub del_chanrec {
	my ($idx, $chan) = @_;
	my $deleted = 0;
	foreach my $friendschan (get_friends_channels($idx)) {
		if (lc($chan) eq lc($friendschan)) {
			delete $friends[$idx]->{channels}->{$friendschan};
			debug("del_chanrec(): Deleted $friendschan record from $friends[$idx]->{handle}", 5);
			$deleted = 1;
		}
	}
	return $deleted;
}

# struct friend del_friend($idx)
# removes friend no. $idx from $friends
# returns the removed $friend object
sub del_friend {
	my ($idx) = @_;
	my $result = -1;
	$result = splice(@friends, $idx, 1) if ($idx < scalar(@friends));
	debug("del_friend(): Deleted $result->{handle}", 4);
	return $result
}

# bool is_unique_handle($handle)
# checks if the $handle is unique for the whole friendlist
# returns 1 if there's no such $handle
# returns 0 if there is one.
sub is_unique_handle {
	my ($handle) = @_;
	my $idx = get_idxbyhand($handle);
	if ($idx > -1) {
		debug("is_unique_handle() failed on $handle", 2);
		return 0;
	}
	return 1;
}

# str choose_handle($proposed)
# tries to choose a handle, closest to the $proposed one
sub choose_handle {
	my ($proposed) = @_;
	my $counter = 0;
	my $handle = $proposed;

	# do this until we have an unique handle
	while (!is_unique_handle($handle)) {
		if (($handle !~ /([0-9]+)$/) && !$counter) {
			# first, if handle doesn't end with a digit, append '0'
			# (but only in first step)
			$handle .= "0";
		} elsif ($counter < 5) {
			# later, increase the trailing number by one
			# do that 4 times
			my ($number) = $handle =~ /([0-9]+)$/;
			++$number;
			$handle =~ s/([0-9]+)$/$number/;
		} elsif ($counter == 5) {
			# then, if it didn't helped, make $handle = $proposed."_"
			$handle = $proposed . "_";
		} elsif ($counter < 10) {
			# if still unsuccessful, append "_" to the handle
			# do that 4 times
			$handle .= "_";
		} else {
			# if THAT didn't help -- make some silly handle
			# and exit the loop
			$handle = $proposed.'_'.(join '', (0..9, 'a'..'z')[rand 36, rand 36, rand 36, rand 36]);
			last;
		}
		++$counter;
	}

	debug("choose_handle() returning '$handle' in step $counter", 2);
	# return our glorious handle ;-)
	return $handle;
}

# bool friend_has_flag($idx, $flag[, $chan])
# returns true if $friends[$idx] has $flag for $chan
# (checks global flags, if $chan is 'global' or undef)
# returns false if hasn't or $flag is not in allowed_flags
# case sensitive about the FLAG
# case insensitive about the chan.
sub friend_has_flag {
	my ($idx, $flag, $chan) = @_;

	# return false if $flag is out of friends_allowed_flags range
	return 0 unless (is_allowed_flag($flag));

	$chan = "global" unless ($chan);

	if (get_friends_flags($idx, $chan) =~ /\Q$flag\E/) {
		debug("friend_has_flag(): $friends[$idx]->{handle} has $chan flag '$flag'", 6);
		return 1;
	}

	debug("friend_has_flag(): $friends[$idx]->{handle} doesn't have $chan flag '$flag'", 6);
	return 0;
}

# bool friend_is_wrapper($idx, $chan, $goodflag, $badflag)
# something to replace friend_is_* subs
# true on: ($channel +$goodflag OR global +$goodflag) AND ($badflag == "" OR NOT $channel +$badflag))
sub friend_is_wrapper {
	my ($idx, $chan, $goodflag, $badflag) = @_;
	if ((friend_has_flag($idx, $goodflag, $chan) ||
		 friend_has_flag($idx, $goodflag, undef)) && 
		($badflag eq "" || !friend_has_flag($idx, $badflag, $chan))) {
		debug("friend_is_wrapper('$idx', '$chan', '$goodflag', '$badflag') returning true", 7);
		return 1;
	}
	debug("friend_is_wrapper('$idx', '$chan', '$goodflag', '$badflag') returning false", 7);
	return 0;
}

# bool add_flag($idx, $flag[, $chan])
# adds $flag to $idx's $chan flags
# $chan can be 'global' or undef
# case insensitive about the $chan -- chooses the proper case.
# returns 1 on success
sub add_flag {
	my ($idx, $flag, $chan) = @_;
	$chan = lc($chan);
	if ($chan eq "" || $chan eq "global") {
		$friends[$idx]->{globflags} .= $flag;
		return 1;
	} else {
		foreach my $friendschan (get_friends_channels($idx)) {
			if ($chan eq lc($friendschan)) {
				$friends[$idx]->{channels}->{$friendschan}->{flags} .= $flag;
				return 1;
			}
		}
	}
	debug("add_flag() failed for idx=$idx, flag=$flag, chan=$chan", 3);
	return 0;
}

# bool del_flag($idx, $flag[, $chan])
# removes $flag from $idx's $chan flags
# $chan can be 'global' or undef
# case insensitive about the $chan -- chooses the proper case.
sub del_flag {
	my ($idx, $flag, $chan) = @_;
	$chan = lc($chan);
	if ($chan eq "" || $chan eq "global") {
		$friends[$idx]->{globflags} =~ s/\Q$flag\E//g;
		return 1;
	} else {
		foreach my $friendschan (get_friends_channels($idx)) {
			if ($chan eq lc($friendschan)) {
				$friends[$idx]->{channels}->{$friendschan}->{flags} =~ s/\Q$flag\E//i;
				return 1;
			}
		}
	}
	debug("del_flag() failed for idx=$idx, flag=$flag, chan=$chan", 3);
	return 0;
}

# bool change_delay($idx, $delay, $chan)
# alters $idx's delay time for $chan
# fails if $chan is 'global' or undef
sub change_delay {
	my ($idx, $delay, $chan) = @_;
	$chan = lc($chan);
	if ($chan && $chan ne "global") {
		foreach my $friendschan (get_friends_channels($idx)) {
			if ($chan eq lc($friendschan)) {
				$friends[$idx]->{channels}->{$friendschan}->{delay} = $delay;
				debug("change_delay() set delay=$delay for idx=$idx, chan=$chan", 4);
				return 1;
			}
		}
	}
	debug("change_delay() failed for idx=$idx, delay=$delay, chan=$chan", 3);
	return 0;
}

# void list_friend($idx[, $nick])
# prints an info line about certain friend.
# if you want to improve the look of the script, you should
# change /format friends_*, probably.
sub list_friend {
	my ($idx, $nick) = @_;
	my $globflags = get_friends_flags($idx, undef);

	Irssi::printformat(MSGLEVEL_CRAP, 'friends_line_head',
		$idx,
		$friends[$idx]->{handle},
		(($globflags) ? "$globflags" : "[none]"),
		(($friends[$idx]->{password}) ? "yes" : "no"));

	Irssi::printformat(MSGLEVEL_CRAP, 'friends_line_hosts',
		join(", ", get_friends_hosts($idx)) );

	foreach my $chan (get_friends_channels($idx)) {
		my $flags = get_friends_flags($idx, $chan);
		my $delay = get_friends_delay($idx, $chan);
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_line_chan', 
			$chan,
			(($flags) ? "$flags" : "[none]"),
			(($delay) ? "$delay" : "random"));
	}

	if ($friends[$idx]->{comment}) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_line_comment', $friends[$idx]->{comment});
	}

	if ($nick) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_line_currentnick', $nick);
	}
}

# void add_operation($server, "#channel", "op|voice|deop|devoice", timeout, "nick1", "nick2", ...)
# adds a delayed operation
sub add_operation {
	my ($server, $channel, $operation, $timeout, @nicks) = @_;

	# my dear queue, don't grow too big, mmkay? ;^)
	my $maxsize = Irssi::settings_get_int('friends_max_queue_size');
	$maxsize = $friends_max_queue_size unless ($maxsize > 0);
	if (@operationQueue >= $maxsize) {
		debug("add_operation(): operationQueue too big, skipping", 3);
		return;
	}

	push(@operationQueue,
	{
		server=>$server,		# server object
		left=>$timeout,			# seconds left
		nicks=>[ @nicks ],		# array of nicks
		channel=>$channel,		# channel name
		operation=>$operation	# operation ("op", "voice" and so on)
	});

	debug("add_operation(): Added operation to queue, left=$timeout, chan=$channel, operation=$operation", 4);

	$timerHandle = Irssi::timeout_add(1000, 'timer_handler', 0) unless (defined $timerHandle);
}

# void timer_handler()
# handles delay timer
sub timer_handler {
	my @ops = ();

	# splice out expired timeouts. if they are expired, move them to
	# local ops-queue. this allows creating new operations to the queue
	# in the operation. (we're not (yet) doing that)

	for (my $c = 0; $c < @operationQueue;) {
		if ($operationQueue[$c]->{left} <= 0) {
			push(@ops, splice(@operationQueue, $c, 1));
		} else {
			++$c;
		}
	}

	for (my $c = 0; $c < @ops; ++$c) {
		my $op = $ops[$c];
		my $channel = $op->{server}->channel_find($op->{channel});

		# check if $channel is still active (you might've parted)
		if ($channel) {
			my @operationNicks = ();
			foreach my $nickStr (@{$op->{nicks}}) {
				my $nick = $channel->nick_find($nickStr);
				# check if there's still such nick (it might've quit/parted)
				if ($nick) {
					if ($op->{operation} eq "op" && !$nick->{op}) {
						push(@operationNicks, $nick->{nick});
					}
					if ($op->{operation} eq "voice" && !$nick->{voice} &&
						(!$nick->{op} || Irssi::settings_get_bool('friends_voice_opped'))) {
						push(@operationNicks, $nick->{nick});
					}
					if ($op->{operation} eq "deop" && $nick->{op}) {
						push(@operationNicks, $nick->{nick});
					}
					if ($op->{operation} eq "devoice" && $nick->{voice}) {
						push(@operationNicks, $nick->{nick});
					}
				}
			}
			# final stage: issue desired command if we're a chanop
			$channel->command($op->{operation}." ".join(" ", @operationNicks)) if ($channel->{chanop});
		}
	}

	# decrement timeouts.
	for (my $c = 0; $c < @operationQueue; ++$c) {
		--$operationQueue[$c]->{left};
		debug("timer_handler(): Decreasing timeout: $operationQueue[$c]->{left} sec left", 8);
	}

	# if operation queue is empty, remove timer.
	if (!@operationQueue && $timerHandle) {
		Irssi::timeout_remove($timerHandle);
		$timerHandle = undef;
	}
}

# void load_friends()
# loads friends from file
sub load_friends {
	my $friendfile = Irssi::settings_get_str('friends_file');
	if (-e $friendfile) {
		@friends = ();
		local *F;
		open(F, "<$friendfile");
		local $/ = "\n";
		while (<F>) {
			my ($handle, $hosts, $globflags, $chanstr, $password, $comment);
			chop;

			# dealing with comments
			if (/^\#/) {
				# script version
				if (/^\# version = (.+)/) {
					$friends_file_version = $1;
				}
				# timestamp
				if (/^\# written = ([0-9]+)/) {
					$friends_file_written = $1;
				}
				next;
			}

			# split by '%'
			my @fields = split("%", $_);
			foreach my $field (@fields) {
				if ($field =~ /^handle=(.*)$/) {
					$handle = $1;
				} elsif ($field =~ /^hosts=(.*)$/) {
					$hosts = $1;
				} elsif ($field =~ /^globflags=(.*)$/) {
					$globflags = $1;
				} elsif ($field =~ /^chanflags=(.*)$/) {
					$chanstr = $1;
				} elsif ($field =~ /^password=(.*)$/) {
					$password = $1;
				} elsif ($field =~ /^comment=(.*)$/) {
					$comment = $1;
				}
			}

			# handle cannot start with a digit
			# skip friend if it does
			next if ($handle =~ /^[0-9]/);

			# if all fields were processed, and $handle is unique,
			# make a friend and add it to $friends
			debug("load_friends(): calling is_unique_handle($handle)", 5);
			if (is_unique_handle($handle)) {
				push(@friends, new_friend($handle, $hosts, $globflags, $chanstr, $password, $comment));
				debug("load_friends(): Loaded $handle from file", 4);
			} else {
				debug("load_friends(): Duplicate $handle in friends file", 1);
				Irssi::printformat(MSGLEVEL_CRAP, 'friends_duplicate', $handle);
			}
		}

		close(F);

		# check friendlist's version
		my ($verbal, $numerical) = $friends_file_version =~ /^(.+)\ \(([0-9]+)\)$/;
		debug("load_friends(): checking version gave: $verbal $numerical", 5);
		if ($numerical < 20011218) {
			# refuse to load friendlist, if it's too old.
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_filetooold');
			# clear the friendlist
			@friends = ();
			return;
		}

		# if everything's ok -- print a message
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_loaded', scalar(@friends), $friendfile);

	} else {
		# whoops, no such file? bad perms?
		Irssi::print("Cannot load $friendfile");
	}
}

# void cmd_loadfriends($data, $server, $channel)
# handles /loadfriends
sub cmd_loadfriends {
	load_friends();
}

# void save_friends($auto)
# saving friends to file
sub save_friends {
	my ($auto) = @_;
	my $friendfile = Irssi::settings_get_str('friends_file');

	local *F;
	open(F, ">$friendfile") or die "Couldn't open $friendfile for writing";

	# write script's version and update corresponding variable
	$friends_file_version = $friends_version;
	print(F "# version = $friends_file_version\n");
	# write current unixtime and update corresponding variable
	$friends_file_written = time();
	print(F "# written = $friends_file_written\n");

	# go through all entries
	for (my $idx = 0; $idx < @friends; ++$idx) {
		# get friend's channels, corresponding flags and delay values
		# then put them as c,f,d fields into @chanstr
		my @chanstr = ();
		foreach my $chan (get_friends_channels($idx)) {
			$chan =~ s/\%//g;
			push(@chanstr, $chan.",".(get_friends_flags($idx, $chan)).",".
				(get_friends_delay($idx, $chan)));
		}

		debug("save_friends(): Writing $friends[$idx]->{handle} to file", 4);

		# write the actual line
		print(F join("%",
			"handle=".$friends[$idx]->{handle},
			"hosts=".(join(" ", get_friends_hosts($idx))),
			"globflags=".(get_friends_flags($idx, undef)),
			"chanflags=".(join(" ", @chanstr)),
			"password=".$friends[$idx]->{password},
			"comment=".$friends[$idx]->{comment},
			"\n"));
	}
	# close file, print message
	close(F);
	Irssi::printformat(MSGLEVEL_CRAP, 'friends_saved', scalar(@friends), $friendfile) unless ($auto);
}

# void cmd_savefriends($data, $server, $channel)
# handles /savefriends
sub cmd_savefriends {
	eval {
		save_friends(0);
	};
	Irssi::print("Saving friendlist failed: $?") if ($?);
}

# void event_setup_saved($config, $auto)
# calls save_friends to save friendslist while saving irssi's setup
# (if friends_autosave is turned on)
sub event_setup_saved {
	my ($config, $auto) = @_;
	return unless (Irssi::settings_get_bool('friends_autosave'));

	eval {
		save_friends($auto);
	};
	Irssi::print("Saving friendlist failed: $?") if ($?);
}

# void event_setup_reread($config)
# calls load_friends() while setup is re-readed
# (if friends_autosave is turned on)
sub event_setup_reread {
	load_friends() if (Irssi::settings_get_bool('friends_autosave'));
}

# int calculate_delay($idx, $chan)
# calculates delay
sub calculate_delay {
	my ($idx, $chan) = @_;
	my $delay = get_friends_delay($idx, $chan);
	my $min = Irssi::settings_get_int('friends_delay_min');
	my $max = Irssi::settings_get_int('friends_delay_max');

	# lazy man's sanity checks :-P
	$min = $default_delay_min if $min < 0;
	$max = $default_delay_max if $min > $max;

	$delay = int(rand ($max - $min)) + $min unless ($delay =~ /^[0-9]+$/);

	return $delay;
}

# void check_friends($server, $channelstr, $options, @nickstocheck)
# checks the given nicklist, channelname and server against the friendlist
sub check_friends {
	my ($server, $channelName, $options, @nicks) = @_;
	my $channel = $server->channel_find($channelName);
	my $delay = 30;
	my %opList = ();
	my %voiceList = ();

	debug("check_friends(): $server->{address}, $channelName, nickcount=".scalar(@nicks), 5);

	# server and channel -- a must.
	return unless ($server && $channelName);

	# get settings
	my $voice_opped = Irssi::settings_get_bool('friends_voice_opped');
	my $allowed = Irssi::settings_get_str('friends_allowed_flags');

	# for each nick from the given list
	foreach my $nick (@nicks) {
		# check if he's a friend on a given channel
		if ((my $idx = get_idx($channelName, $nick->{nick}, $nick->{host})) > -1) {

			# notify about the join if "showjoins" is set
			if ($options =~ /showjoins/) {
				my $globflags = get_friends_flags($idx, undef);
				my $chanflags = get_friends_flags($idx, $channelName);

				my $win = $server->window_item_find($channelName);
				$win = Irssi::active_win() unless ($win);
				$win->printformat(MSGLEVEL_CRAP, 'friends_joined',
					$nick->{nick},
					$friends[$idx]->{handle},
					($globflags) ? $globflags : "[none]",
					$channelName,
					($chanflags) ? $chanflags : "[none]");
			}

			# notice1: password doesn't matter in this loop
			# notice2: channel flags take precedence over the global ones

			# handle auto-(op|voice)
			if (friend_is_wrapper($idx, $channelName, "a", undef)) {
				# add $nick to opList{delay} if he is a valid op
				# and isn't opped already
				# 'valid op' means: (chanflag +o OR globflag +o) AND NOT chanflag +d
				if (friend_is_wrapper($idx, $channelName, "o", "d") && !$nick->{op}) {
					# calculate delay, add to $opList{$delay}
					$delay = calculate_delay($idx, $channelName);
					$opList{$delay}->{$nick->{nick}} = 1;
				}
				# add $nick to voiceList{delay} if he is a valid voice
				# and isn't voiced already
				if (friend_is_wrapper($idx, $channelName, "v", undef) && !$nick->{voice} &&
					(!$nick->{op} || $voice_opped)) {
					# calculate delay, add to $voiceList{$delay}
					$delay = calculate_delay($idx, $channelName);
					$voiceList{$delay}->{$nick->{nick}} = 1;
				}
			}
		}
	}

	# opping
	foreach my $delay (keys %opList) {
		debug("check_friends(): adding operation: $server->{address}, $channelName, op, $delay", 3);
		add_operation($server, $channelName, "op", $delay, keys %{$opList{$delay}});
	}
	# voicing
	foreach my $delay (keys %voiceList) {
		debug("check_friends(): adding operation: $server->{address}, $channelName, voice, $delay", 3);
		add_operation($server, $channelName, "voice", $delay, keys %{$voiceList{$delay}});
	}

	debug("check_friends(): calling timer_handler()", 4);
	timer_handler();
}

# void event_modechange($server, $data, $nick)
# handles modechanges and learning
sub event_modechange {
	my ($server, $data, $nick) = @_;
	my ($channel, $modeStr, $nickStr) = $data =~ /^([^ ]+) ([^ ]+) (.*)$/;
	my @modeargs = split(" ", $nickStr);
	my $ptr = 0;
	my $mode = undef;
	my $gotOpped = 0;
	my $learnFriends = Irssi::settings_get_bool('friends_learn');
	my $opperInfo = undef;
	my $opperIdx = -1;
	my $learnFromOpper = 0;
	my $channelInfo = $server->channel_find($channel);
	my $myNick = $server->{nick};

	debug("event_modechange(): Got mode change on $server->{address}: $nick/$channel $modeStr $nickStr", 4);

	if ($channelInfo && ($nick ne $myNick)) {
		$opperInfo = $channelInfo->nick_find($nick);
		$opperIdx = get_idx($channel, $opperInfo->{nick}, $opperInfo->{host}) if ($opperInfo);
	}

	# learn if it ISN'T ME who opped, learning is enabled, 
	# we know the opper, and we're allowed to learn from him
	if ($learnFriends && $opperIdx != -1 &&
		(friend_is_wrapper($opperIdx, $channel, "F", undef))) {
		$learnFromOpper = 1;
	}

	# process the mode string
	foreach my $char (split(//, $modeStr)) {
		if ($char eq "+") {
			$mode = "+";
		} elsif ($char eq "-") {
			$mode = "-";
		} elsif (lc($char) eq "o") {

			if ($mode eq "+") {
				# op
				if ($modeargs[$ptr] eq $myNick) {
					$gotOpped = 1;
					debug("event_modechange(): Got opped by $nick on $channel/$server->{address}", 4);

				} elsif ($learnFromOpper && ($nick ne $modeargs[$ptr])) {
					# handle the learning stuff.
					my $nickInfo = $channelInfo->nick_find($modeargs[$ptr]);
					my $friend;
					my $friendIdx = -1;

					$friendIdx = get_idx($channel, $nickInfo->{nick}, $nickInfo->{host}) if ($nickInfo);
					debug("event_modechange(): Checked ".$nickInfo->{nick}."!".$nickInfo->{host}." (opped by $nick), got idx=$friendIdx", 4);

					if ($friendIdx == -1) {
						# we got someone not known before
						# choose a handle for him and add him to our friendlist with +L $channel
						$friend = new_friend(
							choose_handle($modeargs[$ptr]),	# handle
							"*!".$nickInfo->{host}, 		# hostmask
							undef,							# globflags
							$channel.",L,",					# channel,chanflags,chandelay
							undef,							# password
							"Learnt (opped by $friends[$opperIdx]->{handle} on $channel\@$server->{tag})"	# comment
						);
						push(@friends, $friend);
						debug("event_modechange(): Added $friend->{handle} to userlist (learnt)", 2);
					} else {
						# we know him already
						$friend = $friends[$friendIdx];
						debug("event_modechange(): opped nick is known as $friend->{handle}", 4);
					}

					if ($friendIdx == -1 || get_friends_flags($friendIdx, $channel) eq "L") {
						# add him to the opper's friendlist
						# ($opperIdx != -1, we've checked that with $learnFromOpper earlier)
						debug("event_modechange(): pushing $friend->{handle} to $friends[$opperIdx]->{handle}'s friendlist", 5);
						push(@{$friends[$opperIdx]->{friends}}, $friend);
					} else {
						debug("event_modechange(): $friend->{handle} has more than 'L' for $channel, leave him alone", 5);
					}
				}
			} elsif ($mode eq "-" && ($nick ne $myNick) && ($nick ne $modeargs[$ptr])) {
				# deop
				if (my $victim = $channelInfo->nick_find($modeargs[$ptr])) {
					if ((my $victimIdx = get_idx($channel, $victim->{nick}, $victim->{host})) > -1) {
						# if a +r'ed person was deopped by an non-master person, perform a reop
						if (friend_is_wrapper($victimIdx, $channel, "r", "d") &&
							!friend_is_wrapper($opperIdx, $channel, "m", undef)) {
							add_operation($server, $channel, "op", calculate_delay($victimIdx, $channel), $victim->{nick})
						}
					}
				}
			}
			# increase pointer, 'o' mode has argument, *always*
			$ptr++;
		} elsif ($char =~ /[beIqdhvk]/ || ($char eq "l" && $mode eq "+")) {
			# increase pointer, these modes have arguments as well
			$ptr++;
		}
	}

	if ($gotOpped && $channelInfo) {
		debug("event_modechange(): calling check_friends($server->{address}, $channel, undef, channel->nicks)", 5);
		check_friends($server, $channel, undef, $channelInfo->nicks());
	}
}

# void event_massjoin($channel, $nicklist)
# handles join event
sub event_massjoin {
	my ($channel, $nicksList) = @_;
	my @nicks = @{$nicksList};
	my $server = $channel->{'server'};
	my $channelName = $channel->{name};
	my $begin = time;

	my $options = "showjoins|" if Irssi::settings_get_bool("friends_flags_on_join");

	debug("event_massjoin(): calling check_friends($server->{address}, $channelName, '$options', nicks)", 5);
	check_friends($server, $channelName, $options, @nicks);

	if ((my $duration = time - $begin) >= 2) {
		# if checking took more than 2 seconds -- print a message about it
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_checking', $channelName, $duration, $server->{address});
	}
}

# void event_nicklist_changed($channel, $nick, $oldnick)
# some kind of nick-tracking
# alters operationQueue if someone from there has changed nick
sub event_nicklist_changed {
	my ($channel, $nick, $oldnick) = @_;

	debug("event_nicklist_changed(): Got nick change on $channel->{name}/$channel->{server}->{address} from $oldnick to $nick->{nick}", 5);

	# nicknames are case insensitive
	return if (lc($oldnick) eq lc($nick->{nick}));

	# cycle through all operation queues
	for (my $c = 0; $c < @operationQueue; ++$c) {
		# temporary array
		my @nickarr = ();
		# is there any nick in this queue that needs altering?
		my $found = 0;

		# skip if server doesn't match
		next unless ($operationQueue[$c]->{server}->{address} eq $channel->{server}->{address});

		# cycle through all nicks in single operation queue
		foreach my $opnick (@{$operationQueue[$c]->{nicks}}) {
			# if $oldnick was in the queue
			if (lc($oldnick) eq lc($opnick)) {
				# ... replace it with the new one
				push(@nickarr, $nick->{nick});
				debug("event_nicklist_changed(): Nick change impacts the queue. Altering. ($oldnick -> $nick->{nick})", 3);
				$found = 1;
			} else {
				push(@nickarr, $opnick);
			}
		}

		# replace $opQ[$c]->{nicks} with our new nicklist if any nick needed updating
		$operationQueue[$c]->{nicks} = [ @nickarr ] if ($found);
	}
}

# void event_server_disconnected($server, $anything)
# removes all queues related to $server from @operationQueue
sub event_server_disconnected {
	my ($server, $anything) = @_;
	my @removed = ();

	debug("event_server_disconnected(): $server->{address} got disconnected, removing queues", 6);

	# cycle through all operation queues
	for (my $c = 0; $c < @operationQueue;) {
		if ($operationQueue[$c]->{server}->{address} eq $server->{address}) {
			push(@removed, splice(@operationQueue, $c, 1));
		} else {
			++$c;
		}
	}

	for (my $c = 0; $c < @removed; ++$c) {
		debug("event_server_disconnected(): removed queue related to $removed[$c]->{server}->{address} ($removed[$c]->{left} secs before execution)", 4);
	}

	# if operation queue is empty, remove the timer.
	if (scalar(@removed) && !@operationQueue && $timerHandle) {
		debug("event_server_disconnected(): operationQueue is now empty, removing the timer", 3);
		Irssi::timeout_remove($timerHandle);
		$timerHandle = undef;
	}
}

# void cmd_queue($data, $server, $channel)
# handles /QUEUE
# parses arguments, calls appropriate subroutines
sub cmd_queue {
	my ($what, $args) = split(/ +/, $_[0], 2);
	my $usage = "Usage: /QUEUE <show|flush|purge>";
	$what = lc($what);

	debug("cmd_queue(): got what='$what' args='$args'", 4);

	if ($what eq "show") {
		debug("cmd_queue(): calling queue_show()", 5);
		queue_show();
	} elsif ($what eq "flush") {
		debug("cmd_queue(): calling queue_flush()", 5);
		queue_flush($args);
	} elsif ($what eq "purge") {
		debug("cmd_queue(): calling queue_purge()", 5);
		queue_purge($args);
	} else {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}
}

# bool queue_flush_expand(%what)
# "... and few lines of The Magic Code. Now. Your poison is ready."
sub queue_flush_expand {
	my ($flush) = @_;
	my $result = 0;

	foreach my $s (keys(%{$flush})) {
		# is this server active?
		my $server = Irssi::server_find_tag($s);
		next unless $server;

		foreach my $c (keys(%{$flush->{$s}})) {
			# is this channel active?
			my $channel = $server->channel_find($c);
			next unless $channel;

			foreach my $o (sort keys(%{$flush->{$s}->{$c}})) {
				my @nicklist = ();
				foreach my $nickStr (sort keys(%{$flush->{$s}->{$c}->{$o}})) {
					# is this nick still here?
					if (my $nick = $channel->nick_find($nickStr)) {
						push(@nicklist, $nick->{nick});
					}
				}

				if (my $nickstr = join(" ", @nicklist)) {
					debug("queue_flush_expand() sending command '/$o $nickstr' to $server->{address}", 1);
					$channel->command($o." ".$nickstr);
					$result = 1;
				}
			}
		}
	}
	return $result;
}

# void queue_show()
# handles /QUEUE SHOW
# prints @operationQueue's contents
sub queue_show {
	if (!@operationQueue) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_queue_empty');
		return;
	}

	# cycle through all operation queues
	for (my $c = 0; $c < @operationQueue; ++$c) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_queue_line1', 
			$c,
			$operationQueue[$c]->{left},
			$operationQueue[$c]->{operation}
		);
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_queue_line2', 
			$operationQueue[$c]->{server}->{address},
			$operationQueue[$c]->{channel},
			join(", ", @{$operationQueue[$c]->{nicks}})
		);
	}
}

# void queue_flush($data)
# handles /QUEUE FLUSH <number|all>
# flushes given/all queue(s)
sub queue_flush {
	my ($data) = @_;
	my $usage = "/QUEUE FLUSH <number|all>";
	my @flushqueue = ();
	my $flushdata = {};

	if (!@operationQueue) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_queue_empty');
		return;
	}

	if ($data eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	if (lc($data) =~ /^all/) {
		@flushqueue = @operationQueue;
		@operationQueue = ();
	} elsif ($data =~ /^[0-9,]+$/) {
		foreach my $num (split(/,/, $data)) {
			if ($num >= @operationQueue) {
				Irssi::printformat(MSGLEVEL_CRAP, 'friends_queue_nosuch', $num);
				next;
			}
			push(@flushqueue, splice(@operationQueue, $num, 1));
		}
	} else {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_badargs', $usage);
		return;
	}

	if (@flushqueue) {
		# don't ask... ;^)
		foreach my $q (@flushqueue) {
			my $s = $q->{server}->{tag};
			my $c = $q->{channel};
			my $o = $q->{operation};
			foreach my $n (@{$q->{nicks}}) {
				$flushdata->{$s}->{$c}->{$o}->{$n} = 1 unless ($o eq "voice" && 
					exists $flushdata->{$s}->{$c}->{op}->{$n} && 
					!Irssi::settings_get_bool('friends_voice_opped'));
			}
		}
		debug("queue_flush(): calling queue_flush_expand()", 3);
		my $result = ((queue_flush_expand($flushdata)) ? "seems ok" : "looks like nothing done");
		debug("queue_flush(): flushed queues: $data (result: $result)", 3);
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_queue_removed', "Flushed", $data, $result);
	}

	if (!@operationQueue && $timerHandle) {
		Irssi::timeout_remove($timerHandle);
		$timerHandle = undef;
	}
}

# void queue_purge($data)
# handles /QUEUE PURGE <number|all>
# removes given/all queue(s)
sub queue_purge {
	my ($data) = @_;
	my $usage = "/QUEUE PURGE <number|all>";
	my $result = "";

	if (!@operationQueue) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_queue_empty');
		return;
	}

	if ($data ne "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	if (lc($data) =~ /^all/) {
		@operationQueue = ();
		$result = "OK";
		debug("queue_purge(): purged all queues", 4);
	} elsif ($data =~ /^[0-9,]+$/) {
		foreach my $num (split(/,/, $data)) {
			if ($num >= @operationQueue) {
				Irssi::printformat(MSGLEVEL_CRAP, 'friends_queue_nosuch', $num);
				next;
			}
			if (splice(@operationQueue, $num, 1)) {
				$result = "OK";
				debug("queue_flush(): purged queue $num", 4);
			}
		}
	} else {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_badargs', $usage);
		return;
	}

	if (!@operationQueue && $timerHandle) {
		Irssi::timeout_remove($timerHandle);
		$timerHandle = undef;
	}

	Irssi::printformat(MSGLEVEL_CRAP, 'friends_queue_removed', "Purged", $data, $result) if ($result);
}

# void friends_chflags($idx, $string[, $chan])
# parses the $string and calls add_flag() or del_flag()
sub friends_chflags {
	my ($idx, $string, $chan) = @_;
	my $mode = undef;
	my $char;

	$chan = "global" if ($chan eq "" || lc($chan) eq "global");

	foreach my $char (split(//, $string)) {
		if ($char eq "+") {
			$mode = "+";
		} elsif ($char eq "-") {
			$mode = "-";
		} elsif ($mode) {
			if ($mode eq "+" && is_allowed_flag($char)) {
				# ADDING flags
				# add chan record, if needed
				add_chanrec($idx, $chan) if ($chan ne "global" && !friend_has_chanrec($idx, $chan));
				if (!friend_has_flag($idx, $char, $chan)) {
					# add this flag if he doesn't have it yet
					add_flag($idx, $char, $chan);
					debug("friends_chflags(): added $char to $friends[$idx]->{handle}'s $chan flags", 4);
				}
			} elsif ($mode eq "-") {
				# REMOVING flags
				if ($chan eq "global" || friend_has_chanrec($idx, $chan)) {
					del_flag($idx, $char, $chan);
					debug("friends_chflags(): deleted $char from $friends[$idx]->{handle}'s $chan flags", 4);
				}
			}
		}
	}
}

# void cmd_chflags($data, $server, $channel)
# handles /chflags <handle> <+-flags> [#channel]
sub cmd_chflags {
	my ($handle, $flags, @chans) = split(/ +/, $_[0]);
	my $usage = "/CHFLAGS <handle> <flags_spec> [#channel1] [#channel2] ...";

	# strip %'s
	$handle =~ s/\%//g;

	# not enough args
	if ($handle eq "" || $flags eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	# bad args
	# if the 'flags' part doesn't start with + or -
	if ($flags !~ /^[\+\-]/) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_badargs', $usage);
		return;
	}

	# get idx, yell and return if it isn't valid
	my $idx = get_idxbyhand($handle);
	if ($idx == -1) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $handle);
		return;
	}

	# if #channel wasn't specified -- we'll deal with global flags
	push(@chans, "global") unless (@chans);

	# go through all channels specified
	foreach my $chan (@chans) {
		# strip %'s
		$chan =~ s/\%//g;

		debug("cmd_chflags(): calling friends_chflags($idx, $flags, '$chan')", 4);

		# 'executing +foo-bar for someone (where)'
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_chflagexec', $flags, $friends[$idx]->{handle}, $chan);
		# make changes
		friends_chflags($idx, $flags, $chan);

		my $flagstr = get_friends_flags($idx, $chan);
		# 'current $chan flags for someone are: +blah/[none]'
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_currentflags', (($flagstr) ? $flagstr : "[none]"), $friends[$idx]->{handle}, $chan);
	}
}

# void cmd_chhandle($data, $server, $channel)
# handles /chhandle <oldhandle> <newhandle>
sub cmd_chhandle {
	my ($oldhandle, $newhandle) = split(/ +/, $_[0]);
	my $usage = "/CHHANDLE <oldhandle> <newhandle>";

	# strip %'s
	$newhandle =~ s/\%//g;

	# not enough args
	if ($oldhandle eq "" || $newhandle eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	# handle cannot start with a digit
	if ($newhandle =~ /^[0-9]/) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_badhandle', $newhandle, 
			"Handle may not start with a digit");
		return;
	}

	# check if $newhandle is unique
	# if not, print appropriate message and return
	if (!is_unique_handle($newhandle)) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notuniqhandle', $newhandle);
		return;
	}

	# get idx, yell and return if it's not valid
	my $idx = get_idxbyhand($oldhandle);
	if ($idx == -1) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $oldhandle);
		return;
	}

	# ok, everything seems fine now, let's change the handle.
	$friends[$idx]->{handle} = $newhandle;
	# ... and print a message
	Irssi::printformat(MSGLEVEL_CRAP, 'friends_changed_handle', $oldhandle, $newhandle);
}

# void cmd_chpass($data, $server, $channel)
# handles /chpass <handle> [pass]
# if pass is empty, removes password
# otherwise, crypts it and sets as current one
sub cmd_chpass {
	my ($handle, $pass) = split(/ +/, $_[0]);
	my $usage = "/CHPASS <handle> [newpassword]";

	# not enough args
	if ($handle eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	# get idx, yell and return if it's not valid
	my $idx = get_idxbyhand($handle);
	if ($idx == -1) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $handle);
		return;
	}

	# crypt and set password. then print a message
	$friends[$idx]->{password} = friends_crypt("$pass");
	Irssi::printformat(MSGLEVEL_CRAP, 'friends_chpassexec', $friends[$idx]->{handle});
}

# void cmd_chdelay($data, $server, $channel)
# handles /chdelay <handle> <delay> <#channel>
# use delay=0 to get instant opping
# use delay>0 to get fixed opping delay
# use delay='random' or delay='none' to remove fixed delay (make it random)
sub cmd_chdelay {
	my ($handle, $delay, $chan) = split(/ +/, $_[0]);
	my $usage = "/CHDELAY <handle> <delay> <#channel>";
	my $value;

	# strip %'s
	$chan =~ s/\%//g;

	# not enough args
	if ($handle eq "" || $delay eq "" || $chan eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	# if $chan doesn't start with one of the [!&#+]
	if ($chan !~ /^[\!\&\#\+]/) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_badargs', $usage);
		return;
	}

	# check validness of $delay
	if ($delay =~ /^[0-9]+$/) {
		# numeric value
		$value = $delay;
	} elsif (lc($delay) eq "remove" || lc($delay) eq "none") {
		# 'remove' or 'none'
		$value = undef
	} else {
		# badargs, return
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_badargs', $usage);
		return;
	}

	# get idx, yell and return if it's not valid
	my $idx = get_idxbyhand($handle);
	if ($idx == -1) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $handle);
		return;
	}

	# check if $idx has got $chan record.
	# add one if needed
	add_chanrec($idx, $chan) unless (friend_has_chanrec($idx, $chan));

	# finally, set it, and print a message
	change_delay($idx, $value, $chan);
	Irssi::printformat(MSGLEVEL_CRAP, 'friends_changed_delay', $friends[$idx]->{handle},
		$chan, (($value) ? $value : "[random]"));
}

# void cmd_comment($data, $server, $channel)
# handles /comment <handle> [comment]
# if comment is empty, removes it
# otherwise, sets it as the current one
sub cmd_comment {
	my ($handle, $comment) = split(" ", $_[0], 2);
	my $usage = "/COMMENT <handle> [comment]";

	# not enough args
	if ($handle eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	# get idx, yell and return if it's not valid
	my $idx = get_idxbyhand($handle);
	if ($idx == -1) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $handle);
		return;
	}

	# remove trailing spaces (just-in-case ;) and %'s
	$comment =~ s/[\ ]+$//;
	$comment =~ s/\%//g;

	# finally, set it, and print a message
	$friends[$idx]->{comment} = $comment;

	if ($comment) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_comment_added', $friends[$idx]->{handle}, $comment);
	} else {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_comment_removed', $friends[$idx]->{handle});
	}
}

# void cmd_listfriend($data, $server, $chanel)
# handles /listfriends [what]
# 'what' can be either handle, channel name, 1,2,5,15-style, host mask or empty.
sub cmd_listfriends {
	if (@friends == 0) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_empty');
	} else {
		my ($data) = @_;
		my $counter = 0;
		# remove whitespaces
		$data =~ s/[\ ]+//g;

		if ($data =~ /^[\!\&\#\+]/) {
			# deal with channel
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_friendlist', "channel " . $data);
			for (my $idx = 0; $idx < @friends; ++$idx) {
				if (friend_has_chanrec($idx, $data)) {
					list_friend($idx);
					$counter++;
				}
			}
		} elsif ($data =~ /^[0-9,]+$/) {
			# deal with 1,2,5,15 style
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_friendlist', $data);
			foreach my $idx (split(/,/, $data)) {
				if ($idx < @friends) {
					list_friend($idx);
					$counter++;
				}
			}
		} elsif ($data =~ /^.*!.*@.*$/) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_friendlist', "matching " . $data);
			foreach (my $idx = 0; $idx < @friends; ++$idx) {
				if (friend_has_host($idx, $data, 2)) {
					list_friend($idx);
					$counter++;
				}
			}
		} else {
			if ((my $idx = get_idxbyhand($data)) > -1) {
				# deal with handle
				Irssi::printformat(MSGLEVEL_CRAP, 'friends_friendlist', $data);
				list_friend($idx);
				$counter++;
			} else {
				# deal with every entry
				Irssi::printformat(MSGLEVEL_CRAP, 'friends_friendlist', "all");
				for (my $idx = 0; $idx < @friends; ++$idx) {
					list_friend($idx);
					$counter++;
				}
			}
		}
		if ($counter) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_friendlist_count', $counter, (($counter > 1) ? "s" : ""));
		}
	}
}

# void cmd_addfriend($data, $server, $channel)
# handles /addfriend <handle> <hostmask> [flags]
# if 'flags' is empty, uses friends_default_flags instead
sub cmd_addfriend {
	my ($handle, $host, $flags) = split(/ +/, $_[0]);
	my $usage = "/ADDFRIEND <handle> <hostmask> [flags]";

	# strip %'s
	$handle =~ s/\%//g;
	$host =~ s/\%//g;

	# not enough args
	if ($handle eq "" || $host eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	# handle cannot start with a digit
	if ($handle =~ /^[0-9]/) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_badhandle', $handle, "Handle may not start with a digit");
		return;
	}

	# check must be unique
	if (!is_unique_handle($handle)) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notuniqhandle', $handle);
		return;
	}

	# add friend.
	debug("cmd_addfriend(): calling new_friend($handle, $host, undef, undef, undef, undef)", 4);
	push(@friends, new_friend($handle, $host, undef, undef, undef, undef));
	Irssi::printformat(MSGLEVEL_CRAP, 'friends_added', $handle);

	# check 'flags' parameter, add default flags if empty.
	$flags = Irssi::settings_get_str('friends_default_flags') unless ($flags);

	# add flags and print them if needed
	if ($flags) {
		# check if $flags start with a '+'. if not, prepend one.
		$flags = "+".$flags unless ($flags =~ /^\+/);

		# our new friend should have $idx=(scalar(@friends)-1) now, so we'll use it.
		my $idx = scalar(@friends) - 1;

		debug("cmd_addfriend(): calling friends_chflags($idx, $flags, global)", 4);
		friends_chflags($idx, $flags, "global");
		$flags = get_friends_flags($idx, undef);
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_currentflags', $flags, $handle, "global") if ($flags);
	}
}

# void cmd_delfriend($data, $server, $channel)
# handles /delfriend <handle|number>
sub cmd_delfriend {
	my ($who) = split(/ +/, $_[0]);
	my $usage = "/DELFRIEND <handle|number>";
	my $idx = -1;

	# strip %'s
	$who =~ s/\%//g;

	# not enough args
	if ($who eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	if ($who =~ /^[0-9]+$/) {
		# query in numeric format, check if it's valid, yell if it's not
		if ($who > -1 && $who < @friends) {
			$idx = $who;
		} else {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $who);
			return;
		}
	} else {
		# query in a verbal format
		# get idx by handle, yell and return if it's not valid
		if (($idx = get_idxbyhand($who)) == -1) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $who);
			return;
		}
	}

	# found a valid idx, now delete a friend
	if ((my $deleted = del_friend($idx)) > -1) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_removed', $deleted->{handle});
	} else {
		Irssi::print("Whoops. Bug in /delfriend? Report it to shasta\@atn.pl please");
	}
}

# void cmd_addhost($data, $server, $channel)
# handles /addhost <handle> <hostmask1> [hostmask2] ...
# hostmask may not overlap with any of the current ones
sub cmd_addhost {
	my ($handle, @hosts) = split(/ +/, $_[0]);
	my $usage = "/ADDHOST <handle> <hostmask1> [hostmask2] [hostmask3] ...";

	# not enough args
	if ($handle eq "" || !@hosts) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	# get idx, yell and return if it's not valid
	my $idx = get_idxbyhand($handle);
	if ($idx == -1) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $handle);
		return;
	}

	foreach my $host (@hosts) {
		# strip %'s
		$host =~ s/\%//g;

		if (!friend_has_host($idx, $host, 2)) {
			add_host($idx, $host);
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_host_added', $friends[$idx]->{handle}, $host);
		} else {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_host_exists', $friends[$idx]->{handle}, $host);
		}
	}
}

# void cmd_delhost($data, $server, $channel)
# handles /delhost <handle> <hostmask>
# hostmask should be EXACTLY the same as one in $friends[$idx]->{hosts}
sub cmd_delhost {
	my ($handle, $host) = split(/ +/, $_[0]);
	my $usage = "/DELHOST <handle> <hostmask>";

	# strip %'s
	$host =~ s/\%//g;

	# not enough args
	if ($handle eq "" || $host eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	# get idx, yell and return if it's not valid
	my $idx = get_idxbyhand($handle);
	if ($idx == -1) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $handle);
		return;
	}

	# delete host, print appropriate message
	if (del_host($idx, $host)) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_host_removed', $friends[$idx]->{handle}, $host);
	} else {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_host_notexists', $friends[$idx]->{handle}, $host);
	}
}

# void cmd_delchanrec($data, $server, $channel)
# handles /delchanrec <handle> <#channel>
sub cmd_delchanrec {
	my ($handle, $chan) = split(/ +/, $_[0]);
	my $usage = "/DELCHANREC <handle> <#channel>";

	# strip %'s
	$chan =~ s/\%//g;

	# not enough args
	if ($handle eq "" || $chan eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	# get idx, yell and return if it's not valid
	my $idx = get_idxbyhand($handle);
	if ($idx == -1) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $handle);
		return;
	}

	# delete chanrec, print appropriate message
	if (del_chanrec($idx, $chan)) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_chanrec_removed', $friends[$idx]->{handle}, $chan);
	} else {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_chanrec_notexists', $friends[$idx]->{handle}, $chan);
	}
}

# void cmd_findfriends($data, $server, $channel)
# handles /findfriends
# prints all online friends
sub cmd_findfriends {
	# cycle through all channels
	foreach my $channel (Irssi::channels()) {
		my $myNick = $channel->{server}->{nick};

		Irssi::printformat(MSGLEVEL_CRAP, 'friends_findfriends', $channel->{name}, $channel->{server}->{address});

		# cycle through all nicks on current channel
		foreach my $nick ($channel->nicks()) {
			# but skip our own
			next if ($nick->{nick} eq $myNick);

			# check if $nick is a friend of us
			if ((my $idx = get_idx($channel->{name}, $nick->{nick}, $nick->{host})) > -1) {
				# if yes, show his entry
				list_friend($idx, $nick->{nick});
			}
		}
	}
}

# void cmd_isfriend($data, $server, $channel)
# handles /isfriend <nick>
sub cmd_isfriend {
	my ($data, $server, $channel) = @_;
	my $usage = "/ISFRIEND <nick>";

	# remove trailing spaces
	$data =~ s/[\ ]+$//;

	# not enough args
	if ($data eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_notenoughargs', $usage);
		return;
	}

	# no server item in current window
	if (!$server) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_general', "No server item in current window");
		return;
	}

	# redirect userhost reply to event_isfriend_userhost()
	# caution: This works only with Irssi 0.7.98.CVS (20011117) and newer
	$server->redirect_event("userhost", 1, $data, 0, undef, {
				"event 302" => "redir userhost_friends"});
	debug("cmd_isfriend(): Redirect set up: event 302 -> userhost_friends(), $data", 3);
	# send our query
	$server->send_raw("USERHOST :$data");
	debug("cmd_isfriend(): USERHOST :$data sent to $server->{address}", 3);
}

# void event_isfriend_userhost($server, $reply, $servername?)
# handles redirected USERHOST replies
# (part of /isfriend)
sub event_isfriend_userhost {
	my ($mynick, $reply) = split(/ +/, $_[1]);
	my ($nick, $user, $host) = $reply =~ /^:?([^\*=]*)\*?=.(.*)@(.*)/;
	my $friend_matched = 0;

	debug("event_isfriend_userhost(): Got reply: $reply", 3);

	# try matching ONLY if the response is positive
	if ($nick && $user && $host) {
		# cycle through all friendlist's entries
		for (my $idx = 0; $idx < @friends; ++$idx) {
			# check if received nick!user@host matches any of our friends
			if (friend_has_host($idx, ($nick.'!'.$user.'@'.$host), 1)) {
				# if so, give informations about him
				list_friend($idx, $nick);
				$friend_matched = 1;
			}
			# you can uncomment this to stop matching
			# after the first match:
			#last if ($friend_matched);
		}
	} else {
		debug("event_isfriend_userhost(): reply was empty, skipping", 3);
	}

	# print a message
	if ($friend_matched) {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_endof', "/isfriend", $nick);
	} else {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_nosuch', $nick);
	}
}

# void flush_learnt()
# cycles through all users and removes every chanrec with flag L
# then, if no other stuff left (specific delay, other chanrecs,
# global flags, password maybe) -- deletes user.
# clears the opping tree too
sub flush_learnt {
	# cycle through the whole friendlist
	for (my $idx = 0; $idx < @friends; ++$idx) {
		my $was_learnt = 0;

		# foreach friend, clear his opping tree
		$friends[$idx]->{friends} = [];

		# now go through all friend's channel entries
		foreach my $chan (get_friends_channels($idx)) {
			# if 'L' is the only flag for this chan
			if (get_friends_flags($idx, $chan) eq "L") {
				# remove channel record and print a message
				$was_learnt = del_chanrec($idx, $chan);
				Irssi::printformat(MSGLEVEL_CRAP, 'friends_chanrec_removed', $friends[$idx]->{handle}, $chan);
			}
		}

		# delete friend, if he has no global flags,
		# neither password, nor chanrecs, and he was learnt.
		if ($was_learnt && !get_friends_flags($idx, undef) &&
			!get_friends_channels($idx) && !$friends[$idx]->{password}) {
			if ((my $deleted = del_friend($idx)) > -1) {
				# print a message
				Irssi::printformat(MSGLEVEL_CRAP, 'friends_removed', $deleted->{handle});
			}
		}
	}
}

# void opping_tree()
# prints the Opping Tree
sub opping_tree {
	Irssi::printformat(MSGLEVEL_CRAP, 'friends_optree_header');

	# cycle through the whole friendlist
	for (my $idx = 0; $idx < @friends; ++$idx) {
		# get friend's friends
		my @friendFriends = @{$friends[$idx]->{friends}};
		if (@friendFriends) {
			# print info about our friend
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_optree_line1', $friends[$idx]->{handle});
			my %masks;
			# get all masks opped by him
			foreach my $friend (@friendFriends) {
				foreach my $host (keys(%{$friend->{hosts}})) {
					$masks{$host}++;
					last;
				}
			}
			# print them, along with the opcount
			foreach my $friend (sort keys %masks) {
				Irssi::printformat(MSGLEVEL_CRAP, 'friends_optree_line2', $masks{$friend}, $friend);
			}
		}
	}
}

# void cmd_oppingtree($data, $server, $channel)
# handles /oppingtree
sub cmd_oppingtree {
	opping_tree();
}

# void cmd_flushlearnt(data, $server, $channel)
# handles /flushlearnt
sub cmd_flushlearnt {
	flush_learnt();
}

# void event_ctcpmsg($server, $args, $sender, $senderhsot, $target)
# handles ctcp requests
sub event_ctcpmsg {
	my ($server, $args, $sender, $userhost, $target) = @_;
	my $idx = -1;

	debug("event_ctcpmsg(): Got CTCP from $sender!$userhost for $target/$server->{address} - $args", 6);

	# return, if ctcp is not for us
	my $myNick = $server->{nick};
	return if (lc($target) ne lc($myNick));

	# return, if we don't process ctcp requests
	return unless (Irssi::settings_get_bool('friends_use_ctcp'));

	my @cmdargs = split(/ +/, $args);

	# prepare arguments:
	# get 1st arg, uppercase it
	my $command = uc($cmdargs[0]);
	# get 2nd arg
	my $channelName = $cmdargs[1];
	# get 3rd arg
	my $password = $cmdargs[2];

	# check if $command is one of friends_ctcp_commands. return if it isn't
	debug("event_ctcpmsg(): calling is_ctcp_command($command)", 5);
	return unless (is_ctcp_command($command));

	my $ctcp_sigstop = 1;

	if ($command eq "PASS") {
		# PASS is weird a bit, it doesn't require friend to have a chanrec or flags at all
		debug("event_ctcpmsg(): calling get_idxbyhost($sender, $userhost)", 5);
		$idx = get_idxbyhost($sender, $userhost);
	} else {
		# Ok, it's NOT a password request.
		# get idx
		debug("event_ctcpmsg(): calling get_idx($channelName, $sender, $userhost)", 5);
		$idx = get_idx($channelName, $sender, $userhost);
	}

	# if get_idx* failed, return.
	if ($idx == -1) {
		my $reason = "Not a friend" . (($command ne "PASS") ? " for $channelName" : "");
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $sender.'!'.$userhost, $reason);
		Irssi::signal_stop();
		return;
	}

	# we'll use handle instead of $sender!$userhost in messages
	my $handle = $friends[$idx]->{handle};

	# check if $channelName was supplied.
	# (first argument, should be always given)
	if ($channelName eq "") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Not enough arguments");
		Irssi::signal_stop();
		return;
	}

	# get channel object. if not found -- yell, stop the signal, and return
	debug("event_ctcpmsg(): calling server->channel_find($channelName)", 5);
	my $channel = $server->channel_find($channelName);
	if (!$channel && $command ne "PASS") {
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Not on channel $channelName");
		Irssi::signal_stop();
		return;
	}

	debug("event_ctcpmsg(): CTCP command from $handle: $command $channelName $password", 5);

	# /ctcp nick OP #channel password
	if ($command eq "OP") {
		if (!friend_is_wrapper($idx, $channelName, "o", "d")) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Not enough flags");
			Irssi::signal_stop();
			return;
		}
		if (!friends_passwdok($idx, $password)) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Bad password");
			Irssi::signal_stop();
			return;
		}

		# process allowed opping
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcprequest', $handle, $command, $channelName);
		debug("event_ctcpmsg(): calling channel->command('op $sender')", 5);
		$channel->command("op $sender");

	# /ctcp nick VOICE #channel password
	} elsif ($command eq "VOICE") {
		if (!friend_is_wrapper($idx, $channelName, "v", undef)) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Not enough flags");
			Irssi::signal_stop();
			return;
		}
		if (!friends_passwdok($idx, $password)) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Bad password");
			Irssi::signal_stop();
			return;
		}

		# process allowed voicing
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcprequest', $handle, $command, $channelName);
		debug("event_ctcpmsg(): calling channel->command('voice $sender')", 5);
		$channel->command("voice $sender");

	# /ctcp nick INVITE #channel password
	} elsif ($command eq "INVITE") {
		if (!friend_is_wrapper($idx, $channelName, "i", undef)) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Not enough flags");
			Irssi::signal_stop();
			return;
		}
		if (!friends_passwdok($idx, $password)) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Bad password");
			Irssi::signal_stop();
			return;
		}

		# process allowed invite
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcprequest', $handle, $command, $channelName);
		debug("event_ctcpmsg(): calling channel->command('invite $sender')", 5);
		$channel->command("invite $sender");

	# /ctcp nick KEY #channel password
	} elsif ($command eq "KEY") {
		if (!friend_is_wrapper($idx, $channelName, "k", undef)) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Not enough flags");
			Irssi::signal_stop();
			return;
		}
		if (!friends_passwdok($idx, $password)) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Bad password");
			Irssi::signal_stop();
			return;
		}

		# process allowed key giving
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcprequest', $handle, $command, $channelName);
		if ($channel->{key}) {
			# give a key if channel is +k'ed
			debug("event_ctcpmsg(): calling server->command('/^notice $sender key for $channelName is: $channel->{key}')", 5);
			$server->command("/^NOTICE $sender Key for $channelName is: $channel->{key}");
		}

	# /ctcp nick LIMIT #channel password
	} elsif ($command eq "LIMIT") {
		if (!friend_is_wrapper($idx, $channelName, "l", undef)) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Not enough flags");
			Irssi::signal_stop();
			return;
		}
		if (!friends_passwdok($idx, $password)) {
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Bad password");
			Irssi::signal_stop();
			return;
		}

		# process allowed limit raising
		Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcprequest', $handle, $command, $channelName);
		my @nicks = $channel->nicks();
		if ($channel->{limit} && $channel->{limit} <= scalar(@nicks)) {
			# raise the limit if it's needed
			debug("event_ctcpmsg(): calling server->command('MODE $channelName +l ".(scalar(@nicks) + 1)."')", 5);
			$server->command("MODE $channelName +l " . (scalar(@nicks) + 1));
		}

	# /ctcp nick PASS pass [newpass]
	} elsif ($command eq "PASS") {
		# if someone has password already set - we can only *change* it
		if ($friends[$idx]->{password}) {
			# if cmdargs[1] ($channelName, that is) is a valid password (current)
			if (!friends_passwdok($idx, $channelName)) {
				Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcpfail', $command, $handle, "Bad password");
				Irssi::signal_stop();
				return;
			}
			# and $cmdargs[2] ($password, that is) contains something ...
			if ($password) {
				# ... process allowed password change.
				# in this case, old password is in $channelName
				# and new password is in $password
				debug("event_ctcpmsg(): changing password for $handle", 4);
				$friends[$idx]->{password} = friends_crypt("$password");
				Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcppass', $handle, $sender."!".$userhost);
				# send a quiet notice to sender
				$server->command("/^NOTICE $sender Password changed to: $password");
			} else {
				# in this case, notify sender about his current password quietly
				debug("event_ctcpmsg(): Password already set for $handle while trying to change", 4);
				$server->command("/^NOTICE $sender You already have a password set");
			}
		# if $idx doesn't have a password, we will *set* it
		} else {
			# in this case, new password is in $channelName
			# and $password is unused
			debug("event_ctcpmsg(): setting password for $handle", 5);
			$friends[$idx]->{password} = friends_crypt("$channelName");
			Irssi::printformat(MSGLEVEL_CRAP, 'friends_ctcppass', $handle, $sender.'!'.$userhost);
			# send a quiet notice to sender
			$server->command("/^NOTICE $sender Password set to: $channelName");
		}
	}

	# stop the signal if we processed the request
	if ($ctcp_sigstop) {
		debug("event_ctcpmsg(): stopping signal", 4);
		Irssi::signal_stop();
	}
}

# void cmd_friendsversion($data, $server, $channel)
# handles /friendsversion
# prints script's and friendlist's version
sub cmd_friendsversion() {
	print_version("script");
	print_version("filever");
	print_version("filewritten");
}

Irssi::settings_add_int("misc", "friends_delay_min", $default_delay_min);
Irssi::settings_add_int("misc", "friends_delay_max", $default_delay_max);
Irssi::settings_add_int("misc", "friends_max_queue_size", $friends_max_queue_size);
Irssi::settings_add_int("misc", "friends_debug_level", $friends_debug_level);
Irssi::settings_add_bool("misc", "friends_learn", $friends_learn);
Irssi::settings_add_bool("misc", "friends_voice_opped", $friends_voice_opped);
Irssi::settings_add_bool("misc", "friends_use_ctcp", $friends_use_ctcp);
Irssi::settings_add_bool("misc", "friends_autosave", $friends_autosave);
Irssi::settings_add_bool("misc", "friends_flags_on_join", $friends_flags_on_join);
Irssi::settings_add_bool("misc", "friends_use_debug_window", $friends_use_debug_window);
Irssi::settings_add_str("misc", "friends_ctcp_commands", $friends_ctcp_commands);
Irssi::settings_add_str("misc", "friends_allowed_flags", $friends_allowed_flags);
Irssi::settings_add_str("misc", "friends_default_flags", $friends_default_flags);
Irssi::settings_add_str("misc", "friends_file", $friends_file);
Irssi::command_bind("addfriend", "cmd_addfriend");
Irssi::command_bind("delfriend", "cmd_delfriend");
Irssi::command_bind("addhost", "cmd_addhost");
Irssi::command_bind("delhost", "cmd_delhost");
Irssi::command_bind("delchanrec", "cmd_delchanrec");
Irssi::command_bind("chhandle", "cmd_chhandle");
Irssi::command_bind("chdelay", "cmd_chdelay");
Irssi::command_bind("loadfriends", "cmd_loadfriends");
Irssi::command_bind("savefriends", "cmd_savefriends");
Irssi::command_bind("listfriends", "cmd_listfriends");
Irssi::command_bind("findfriends", "cmd_findfriends");
Irssi::command_bind("isfriend", "cmd_isfriend");
Irssi::command_bind("chflags", "cmd_chflags");
Irssi::command_bind("chpass", "cmd_chpass");
Irssi::command_bind("comment", "cmd_comment");
Irssi::command_bind("oppingtree", "cmd_oppingtree");
Irssi::command_bind("queue", "cmd_queue");
Irssi::command_bind("queue show", "queue_show");
Irssi::command_bind("queue flush", "queue_flush");
Irssi::command_bind("queue purge", "queue_purge");
Irssi::command_bind("flushlearnt", "cmd_flushlearnt");
Irssi::command_bind("friendsversion", "cmd_friendsversion");
Irssi::signal_add_last("massjoin", "event_massjoin");
Irssi::signal_add_last("event mode", "event_modechange");
Irssi::signal_add("default ctcp msg", "event_ctcpmsg");
Irssi::signal_add("redir userhost_friends", "event_isfriend_userhost");
Irssi::signal_add("setup saved", "event_setup_saved");
Irssi::signal_add("setup reread", "event_setup_reread");
Irssi::signal_add("nicklist changed", "event_nicklist_changed");
# these two could be bound to separate subroutines,
# but they are both bound to event_server_disconnected for now
Irssi::signal_add("server disconnected", "event_server_disconnected");
Irssi::signal_add("server connect failed", "event_server_disconnected");

print_releasenote() if ($release_note);
debug("main(): loading friendlist...", 2);
load_friends();
debug("main(): friends.pl $friends_version init complete", 2);
