/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasutils.h - utility functions.
 */
#ifndef __GOO_CANVAS_UTILS_H__
#define __GOO_CANVAS_UTILS_H__

#include <gtk/gtk.h>

G_BEGIN_DECLS


/*
 * Enum types.
 */
#define GOO_TYPE_CANVAS_POINTER_EVENTS   (goo_canvas_pointer_events_get_type ())
GType goo_canvas_pointer_events_get_type   (void) G_GNUC_CONST;

/**
 * GooCanvasPointerEvents
 * @GOO_CANVAS_EVENTS_VISIBLE_MASK: a mask indicating that the item only
 *  receives events when it is visible.
 * @GOO_CANVAS_EVENTS_PAINTED_MASK: a mask indicating that the item only
 *  receives events when the specified parts of it are painted.
 * @GOO_CANVAS_EVENTS_FILL_MASK: a mask indicating that the filled part of
 *  the item receives events.
 * @GOO_CANVAS_EVENTS_STROKE_MASK: a mask indicating that the stroked part
 *  of the item receives events.
 * @GOO_CANVAS_EVENTS_NONE: the item doesn't receive events at all.
 * @GOO_CANVAS_EVENTS_VISIBLE_PAINTED: the item receives events in its
 *  painted areas when it is visible (the default).
 * @GOO_CANVAS_EVENTS_VISIBLE_FILL: the item's interior receives events
 *  when it is visible.
 * @GOO_CANVAS_EVENTS_VISIBLE_STROKE: the item's perimeter receives
 *  events when it is visible.
 * @GOO_CANVAS_EVENTS_VISIBLE: the item receives events when it is visible,
 *  whether it is painted or not.
 * @GOO_CANVAS_EVENTS_PAINTED: the item receives events in its painted areas,
 *  whether it is visible or not.
 * @GOO_CANVAS_EVENTS_FILL: the item's interior receives events, whether it
 *  is visible or painted or not.
 * @GOO_CANVAS_EVENTS_STROKE: the item's perimeter receives events, whether
 *  it is visible or painted or not.
 * @GOO_CANVAS_EVENTS_ALL: the item's perimeter and interior receive events,
 *  whether it is visible or painted or not.
 *
 * Specifies when an item receives pointer events such as mouse clicks.
 */
typedef enum
{
  /* If the item must be visible to receive events. */
  GOO_CANVAS_EVENTS_VISIBLE_MASK	= 1 << 0,

  /* If the fill or stroke must be painted to receive events. */
  GOO_CANVAS_EVENTS_PAINTED_MASK	= 1 << 1,

  /* If the fill should receive events. */
  GOO_CANVAS_EVENTS_FILL_MASK		= 1 << 2,

  /* If the stroke should receive events. */
  GOO_CANVAS_EVENTS_STROKE_MASK		= 1 << 3,


  GOO_CANVAS_EVENTS_NONE		= 0,

  GOO_CANVAS_EVENTS_VISIBLE_PAINTED	= GOO_CANVAS_EVENTS_VISIBLE_MASK
					  | GOO_CANVAS_EVENTS_PAINTED_MASK
					  | GOO_CANVAS_EVENTS_FILL_MASK
					  | GOO_CANVAS_EVENTS_STROKE_MASK,
  GOO_CANVAS_EVENTS_VISIBLE_FILL	= GOO_CANVAS_EVENTS_VISIBLE_MASK
					  | GOO_CANVAS_EVENTS_FILL_MASK,
  GOO_CANVAS_EVENTS_VISIBLE_STROKE	= GOO_CANVAS_EVENTS_VISIBLE_MASK
					  | GOO_CANVAS_EVENTS_STROKE_MASK,
  GOO_CANVAS_EVENTS_VISIBLE		= GOO_CANVAS_EVENTS_VISIBLE_MASK
					  | GOO_CANVAS_EVENTS_FILL_MASK
					  | GOO_CANVAS_EVENTS_STROKE_MASK,

  GOO_CANVAS_EVENTS_PAINTED		= GOO_CANVAS_EVENTS_PAINTED_MASK
					  | GOO_CANVAS_EVENTS_FILL_MASK
					  | GOO_CANVAS_EVENTS_STROKE_MASK,
  GOO_CANVAS_EVENTS_FILL		= GOO_CANVAS_EVENTS_FILL_MASK,
  GOO_CANVAS_EVENTS_STROKE		= GOO_CANVAS_EVENTS_STROKE_MASK,
  GOO_CANVAS_EVENTS_ALL			= GOO_CANVAS_EVENTS_FILL_MASK
					  | GOO_CANVAS_EVENTS_STROKE_MASK
} GooCanvasPointerEvents;


#define GOO_TYPE_CANVAS_ITEM_VISIBILITY   (goo_canvas_item_visibility_get_type ())
GType goo_canvas_item_visibility_get_type   (void) G_GNUC_CONST;

/**
 * GooCanvasItemVisibility
 * @GOO_CANVAS_ITEM_VISIBLE: the item is visible.
 * @GOO_CANVAS_ITEM_VISIBLE_ABOVE_THRESHOLD: the item is visible when the
 *  canvas scale setting is greater than or equal to the item's visibility
 *  threshold setting.
 * @GOO_CANVAS_ITEM_INVISIBLE: the item is invisible.
 *
 * The #GooCanvasItemVisibility enumeration is used to specify when a canvas
 * item is visible.
 */
typedef enum
{
  GOO_CANVAS_ITEM_VISIBLE,
  GOO_CANVAS_ITEM_VISIBLE_ABOVE_THRESHOLD,
  GOO_CANVAS_ITEM_INVISIBLE
} GooCanvasItemVisibility;


/*
 * Cairo utilities.
 */
typedef struct _GooCanvasLineDash GooCanvasLineDash;

/**
 * GooCanvasLineDash
 * @ref_count: the reference count of the struct.
 * @num_dashes: the number of dashes and gaps between them.
 * @dashes: the sizes of each dash and gap.
 * @dash_offset: the start offset into the dash pattern.
 *
 * #GooCanvasLineDash specifies a dash pattern to be used when drawing items.
 */
struct _GooCanvasLineDash
{
  int ref_count;
  int num_dashes;
  double *dashes;
  double dash_offset;
};

#define GOO_TYPE_CANVAS_LINE_DASH   (goo_canvas_line_dash_get_type ())

GType              goo_canvas_line_dash_get_type (void) G_GNUC_CONST;
GooCanvasLineDash* goo_canvas_line_dash_new   (gint               num_dashes,
					       ...);
GooCanvasLineDash* goo_canvas_line_dash_newv  (gint               num_dashes,
                                               double            *dashes);
GooCanvasLineDash* goo_canvas_line_dash_ref   (GooCanvasLineDash *dash);
void               goo_canvas_line_dash_unref (GooCanvasLineDash *dash);


#define GOO_TYPE_CAIRO_MATRIX	   (goo_cairo_matrix_get_type())

GType              goo_cairo_matrix_get_type  (void) G_GNUC_CONST;
cairo_matrix_t*    goo_cairo_matrix_copy      (cairo_matrix_t    *matrix);


#define GOO_TYPE_CAIRO_PATTERN	   (goo_cairo_pattern_get_type ())
GType              goo_cairo_pattern_get_type (void) G_GNUC_CONST;


G_END_DECLS

#endif /* __GOO_CANVAS_UTILS_H__ */
