/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasmodelsimple.c - simple canvas model.
 */

/**
 * SECTION:goocanvasmodelsimple
 * @Title: GooCanvasModelSimple
 * @Short_Description: a simple implementation of the canvas model interface.
 *
 * #GooCanvasModelSimple is a simple implementation of the #GooCanvasModel
 * interface.
 *
 * Typically the canvas model is created, items are added, and then the model
 * is set in the #GooCanvasView:
 *
 * <informalexample><programlisting>
 *  GooCanvasModelSimple *canvas_model;
 *  GooCanvasItem *root, *item;
 *  
 *  /&ast; Create a simple model. &ast;/
 *  canvas_model = goo_canvas_model_simple_new&nbsp;();
 *  
 *  /&ast; Get the root item of the model. &ast;/
 *  root = goo_canvas_model_get_root_item (GOO_CANVAS_MODEL (canvas_model));
 *  
 *  /&ast; Create a rectangle item. &ast;/
 *  item = goo_canvas_rect_new (root, 100, 100, 50, 50, NULL);
 *  
 *  /&ast; Pass the model to the GooCanvasView. &ast;/
 *  goo_canvas_view_set_model (GOO_CANVAS_VIEW (canvas),
 *                             GOO_CANVAS_MODEL (canvas_model));
 *   
 *  /&ast; Unref the model, since the GooCanvasView owns it now. &ast;/
 *  g_object_unref (canvas_model);
 * </programlisting></informalexample>
 */
#include <config.h>
#include <string.h>
#include <gtk/gtk.h>
#include "goocanvasmodelsimple.h"
#include "goocanvasgroup.h"
#include "goocanvasitem.h"
#include "goocanvasutils.h"


static void canvas_model_interface_init      (GooCanvasModelIface *iface);
static void goo_canvas_model_simple_finalize (GObject *object);
static GooCanvasItem* goo_canvas_model_simple_get_root_item (GooCanvasModel *model);


G_DEFINE_TYPE_WITH_CODE (GooCanvasModelSimple, goo_canvas_model_simple,
			 G_TYPE_OBJECT,
			 G_IMPLEMENT_INTERFACE (GOO_TYPE_CANVAS_MODEL,
						canvas_model_interface_init))

static void
goo_canvas_model_simple_class_init (GooCanvasModelSimpleClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass*) klass;

  gobject_class->finalize = goo_canvas_model_simple_finalize;
}


static void
canvas_model_interface_init (GooCanvasModelIface *iface)
{
  iface->get_root_item = goo_canvas_model_simple_get_root_item;
}


static void
goo_canvas_model_simple_init (GooCanvasModelSimple *model)
{
  model->root_group = goo_canvas_group_new (NULL);

  goo_canvas_group_set_model (GOO_CANVAS_GROUP (model->root_group),
			      GOO_CANVAS_MODEL (model));
}


/**
 * goo_canvas_model_simple_new:
 * 
 * Creates a new simple canvas model.
 * 
 * Returns: a new #GooCanvasModelSimple.
 **/
GooCanvasModelSimple*
goo_canvas_model_simple_new (void)
{
  return GOO_CANVAS_MODEL_SIMPLE (g_object_new (GOO_TYPE_CANVAS_MODEL_SIMPLE, NULL));
}


static void
goo_canvas_model_simple_finalize (GObject *object)
{
  GooCanvasModelSimple *model = (GooCanvasModelSimple*) object;

  g_object_unref (model->root_group);

  G_OBJECT_CLASS (goo_canvas_model_simple_parent_class)->finalize (object);
}


static GooCanvasItem*
goo_canvas_model_simple_get_root_item (GooCanvasModel *model)
{
  return GOO_CANVAS_MODEL_SIMPLE (model)->root_group;
}
