/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/*
    Gpredict: Real-time satellite tracking and orbit prediction program

    Copyright (C)  2001-2006  Alexandru Csete, OZ9AEC.

    Authors: Alexandru Csete <csete@users.sourceforge.net>

    Comments, questions and bugreports should be submitted via
    http://sourceforge.net/projects/groundstation/
    More details can be found at the project home page:

            http://groundstation.sourceforge.net/
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, visit http://www.fsf.org/
*/
/** \file    main.c
 *  \ingroup main
 *  \bief    Main program file.
 *
 * Add some more text.
 *
 * \bug Change to glib getopt in 2.6
 *
 */
#include <stdlib.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#ifdef HAVE_GETOPT_H
#  include <getopt.h>
#endif
#include "sat-log.h"
#include "first-time.h"
#include "compat.h"
#include "gui.h"
#include "mod-mgr.h"
#include "tle-lookup.h"
#include "sat-cfg.h"
#include "sat-debugger.h"

/** \brief Main application widget. */
GtkWidget *app;


const gchar *dummy = N_("just to have a pot");


/* private funtion prototypes */
static void   gpredict_app_create     (void);
static gint   gpredict_app_delete     (GtkWidget *, GdkEvent *, gpointer);
static void   gpredict_app_destroy    (GtkWidget *, gpointer);
static void   gpredict_sig_handler    (int sig);



int
main (int argc, char *argv[])
{
	guint  error;


#ifdef ENABLE_NLS
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (PACKAGE, "UTF-8");
	textdomain (PACKAGE);
#endif
	gtk_set_locale ();
	gtk_init (&argc, &argv);

	/* start logger first, so that we can catch error messages if any */


	/* check that user settings are ok */
	error = first_time_check_run ();
	if (error) {
		sat_log_log (SAT_LOG_LEVEL_ERROR,
			     _("%s: User config check failed (code %d). This is fatal.\n"\
			       "A possible solution would be to remove the .gpredict2 data dir\n"\
			       "in your home directory"),
			     __FUNCTION__, error);

		return 1;
	}

	/* initialise sub-systems */
	if (tle_lookup_init (NULL) != TLE_LOOKUP_INIT_OK) {
		sat_log_log (SAT_LOG_LEVEL_ERROR,
			     _("%s: TLE check failed! This is fatal."),
			     __FUNCTION__);

		return -1;
	}

	sat_cfg_load ();

	/* create application */
	gpredict_app_create ();
	//	gtk_container_add (GTK_CONTAINER (app), gpredict_gui_create ());
	gtk_widget_show_all (app);

	//sat_debugger_run ();

	gtk_main ();

	return 0;
}



/** \brief Create main application window.
 *  \return A new top level window as as GtkWidget.
 *
 * This function creates and initialises the main application window.
 * This function does not create any contents; that part is done in the
 * gpredict_gui package.
 *
 */
static void
gpredict_app_create ()
{
	GtkWidget *window;    /* application window */
	gchar     *title;     /* window title */
	gchar     *icon;      /* icon file name */
	
	/* create window title and file name for window icon  */
	title = g_strdup (_("GPREDICT"));
	icon = icon_file_name ("gpredict-icon.png");

	/* ceate window, add title and icon */
	app = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_default_size (GTK_WINDOW (app),
				     sat_cfg_get_int (SAT_CFG_INT_WINDOW_WIDTH),
				     sat_cfg_get_int (SAT_CFG_INT_WINDOW_HEIGHT));

	gtk_window_set_title (GTK_WINDOW (app), title);

	gtk_container_add (GTK_CONTAINER (app),
					   gui_create (app));

	if (g_file_test (icon, G_FILE_TEST_EXISTS)) {
		gtk_window_set_icon_from_file (GTK_WINDOW (app),
					       icon,
					       NULL);
	}

	g_free (title);
	g_free (icon);

	/* connect delete and destroy signals */
	g_signal_connect (G_OBJECT (app), "delete_event",
			  G_CALLBACK (gpredict_app_delete), NULL);    
	g_signal_connect (G_OBJECT (app), "destroy",
			  G_CALLBACK (gpredict_app_destroy), NULL);

	/* register UNIX signals as well so that we 
	   have a chance to clean up external resources.
	*/
	signal (SIGTERM, (void *) gpredict_sig_handler);
	signal (SIGINT,  (void *) gpredict_sig_handler);
	signal (SIGABRT, (void *) gpredict_sig_handler);

}



/** \brief Handle terminate signals.
 *  \param sig The signal that has been received.
 *
 * This function is used to handle termination signals received by the program.
 * The currently caught signals are SIGTERM, SIGINT and SIGABRT. When one of these
 * signals is received, the function sends an error message to logger and tries
 * to make a clean exit.
 */
static void
gpredict_sig_handler (int sig)
{
/* 	satlog_log (SAT_LOG_ERROR, "Received signal: %d\n", sig); */
/* 	satlog_log (SAT_LOG_ERROR, "Trying clean exit...\n"); */

	gtk_widget_destroy (app);
}


/** \brief Handle delete events.
 *  \param widget The widget which received the delete event signal.
 *  \param event  Data structure describing the event.
 *  \param data   User data (NULL).
 *  \param return Always FALSE to indicate that the app should be destroyed.
 *
 * This function handles the delete event received by the main application
 * window (eg. when the window is closed by the WM). This function simply
 * returns FALSE indicating that the main application window should be
 * destroyed by emiting the destroy signal.
 *
 */
static gint
gpredict_app_delete      (GtkWidget *widget,
			  GdkEvent  *event,
			  gpointer   data)
{
	return FALSE;
}



/** \brief Handle destroy signals.
 *  \param widget The widget which received the signal.
 *  \param data   User data (NULL).
 *
 * This function is called when the main application window receives the
 * destroy signal, ie. it is destroyed. This function signals all daemons
 * and other threads to stop and exits the Gtk+ main loop.
 *
 */
static void
gpredict_app_destroy    (GtkWidget *widget,
			 gpointer   data)
{
	gint w,h;

	/* GUI timers are stopped automatically */

	/* stop timeouts */

	/* stop other sub-systems */
	tle_lookup_close ();

	/* configuration data */
	mod_mgr_save_state ();

/* not good, have to use configure event instead (see API doc) */
/*	gtk_window_get_size (GTK_WINDOW (app), &w, &h);
	sat_cfg_set_int (SAT_CFG_INT_WINDOW_WIDTH, w);
	sat_cfg_set_int (SAT_CFG_INT_WINDOW_HEIGHT, h);	
*/
	sat_cfg_save ();
	sat_cfg_close ();

	/* exit Gtk+ */
	gtk_main_quit ();
}
