/***************************************

    This is part of frox: A simple transparent FTP proxy
    Copyright (C) 2000 James Hollingshead

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
  
     data.c -- Code for handling the data connection and picking fds to select.

  ***************************************/


#include <sys/ioctl.h>
#include <fcntl.h>
#include <syslog.h>

#include "data.h"
#include "control.h"
#include "cache.h"
#include "vscan.h"
#include "transdata.h"

void client_data_connect(void);
void server_data_connect(void);

void writebuf2client(void);
void writebuf2server(void);
void forwarddata2client(void);
void forwarddata2server(void);

void closecd(void);
void closesd(void);

/* ------------------------------------------------------------- **
** Setup fd sets for reading/writing. We always listen to the client's
** control stream, and other fds are selected on depending on state. 
** ------------------------------------------------------------- */
int setup_fds(fd_set * reads, fd_set * writes)
{
	int n;

	FD_ZERO(reads);
	FD_ZERO(writes);

	FD_SET(info->client_control.fd, reads);
	n=info->client_control.fd;

	if (info->server_control.fd != -1) {
		FD_SET(info->server_control.fd, reads);
		if(info->server_control.fd>n) n=info->server_control.fd;
	}

	if (info->server_data.fd != -1) {
		if (sstr_len(info->client_data.buf) == 0
		    && info->state == DOWNLOAD)
			FD_SET(info->server_data.fd, reads);

		if (sstr_len(info->server_data.buf) != 0)
			FD_SET(info->server_data.fd, writes);
		if(info->server_data.fd>n) n=info->server_data.fd;
	}
	if (info->client_data.fd != -1) {
		if (sstr_len(info->server_data.buf) == 0
		    && info->state == UPLOAD)
			FD_SET(info->client_data.fd, reads);
		if (sstr_len(info->client_data.buf) != 0)
			FD_SET(info->client_data.fd, writes);
		if(info->client_data.fd>n) n=info->client_data.fd;
	}
	if (info->listen != -1) {
		FD_SET(info->listen, reads);
		if(info->listen > n) n=info->listen;
	}
	return(n);
}

/* ------------------------------------------------------------- **
** Check the fd_sets, and do data connection forwarding if any.
** ------------------------------------------------------------- */
void do_dataforward(fd_set * reads, fd_set * writes)
{
	if (info->listen != -1 && FD_ISSET(info->listen, reads)) {
		if (info->mode == ACTIVE) server_data_connect();
		else client_data_connect();
	}

	if (info->server_data.fd != -1)
		if (FD_ISSET(info->server_data.fd, reads))
			forwarddata2client();
	if (info->server_data.fd != -1)
		if (FD_ISSET(info->server_data.fd, writes))
			writebuf2server();

	if (info->client_data.fd != -1)
		if (FD_ISSET(info->client_data.fd, reads))
			forwarddata2server();
	if (info->client_data.fd != -1)
		if (FD_ISSET(info->client_data.fd, writes))
			writebuf2client();

}

/* ------------------------------------------------------------- **
**  Client just connected to the data line
** ------------------------------------------------------------- */
void client_data_connect()
{
	int len = sizeof(info->client_data.address);

	debug("  Client has connected to proxy data line\n");
	info->client_data.fd = accept(info->listen,
				      (struct sockaddr *) &info->
				      client_data.address, &len);

	if (config.sameaddress) {
		if (info->client_data.address.sin_addr.s_addr !=
		    info->client_control.address.sin_addr.s_addr) {
			write_log(ATTACK,
				  "Blocked %s from connecting to data line",
				  addr2name(info->client_data.address.
					    sin_addr));
			close(info->client_data.fd);
			info->client_data.fd = -1;
			return;
		}
	}
	il_free();  /* Remove the ipchains entry for intercepting this
		     * data connection.*/

	sstr_empty(info->client_data.buf);

	/* Cache code could have already connected us.*/
	if (info->server_data.fd == -1) connect_server_data();

}

/* ------------------------------------------------------------- **
**  Server just connected to the data line
** ------------------------------------------------------------- */
void server_data_connect()
{
	int len = sizeof(info->server_data.address);

	debug("  Server has connected to proxy data line\n");
	info->server_data.fd = accept(info->listen,
				      (struct sockaddr *) &info->
				      server_data.address, &len);
	if (config.sameaddress) {
		if (info->server_data.address.sin_addr.s_addr !=
		    info->server_control.address.sin_addr.s_addr) {
			write_log(ATTACK,
				  "Blocked %s from connecting to data line",
				  addr2name(info->server_data.address.
					    sin_addr));
			close(info->server_data.fd);
			info->server_data.fd = -1;
			return;
		}
	}

	sstr_empty(info->server_data.buf);
	connect_client_data();
}

int connect_server_data()
{
	info->server_data.fd = 
		connect_to_socket(info->server_data.address, PASV);
	if (info->server_data.fd == -1) {
		write_log(ERROR, "Failed to contact server data port");
		close(info->client_data.fd);
		info->client_data.fd = -1;
		return(-1);
	}
	sstr_empty(info->server_data.buf);
	fcntl(info->server_data.fd, F_SETFL, O_NONBLOCK);
	return(0);
}

int connect_client_data()
{
	/*FIXME should use server_data.address.sin_addr, but 20 as port*/
	if (config.transdata)
		info->client_data.fd =
			transp_connect(info->client_data.address,
				       info->server_data.address);
	else
		info->client_data.fd =
			connect_to_socket(info->client_data.address, ACTV);
	if (info->client_data.fd == -1) {
		write_log(ERROR, "Failed to contact client data port");
		close(info->server_data.fd);
		info->server_data.fd = -1;
		return(-1);
	}
	sstr_empty(info->client_data.buf);
	fcntl(info->client_data.fd, F_SETFL, O_NONBLOCK);
	return(0);
}

/* ------------------------------------------------------------- **
** Forward as much data as possible from client-->server, and store
** the rest in server_data.buf.
** ------------------------------------------------------------- */
void forwarddata2server()
{
	int i;

	i = sstr_append_read(info->client_data.fd, info->server_data.buf, 0);
	if (i < 1) { /*Socket close or error*/
		closecd();
	}

	writebuf2server();
}

/* ------------------------------------------------------------- **
** Forward as much data as possible from server-->client, and store
** the rest in client_data.buf.
**
** The vscan_inc_data(), cache_inc_data() order is because http
** caching uses cache_inc_data to strip http headers which must be
** done before data reaches the virus scanner, while local caching
** uses it to write file data to file which musn't be done until the
** vscan code has emptied the buffer. Without caching cache_inc_data
** has no effect so it makes no difference. Ugly I know, but I could
** think of no better way...
** ------------------------------------------------------------- */
void forwarddata2client()
{
	sstr_append_read(info->server_data.fd, info->client_data.buf, 0);

	/*FIXME - why did I do it like this! cf forwarddata2server() */
	if (sstr_len(info->client_data.buf) < 1) {  /*Socket close or error*/
		if(!vscan_switchover()) closesd();
		return;
	}

	if(config.cachemod && *config.cachemod=='h') { /*http caching*/
		cache_inc_data(info->client_data.buf);
		vscan_inc(info->client_data.buf);
	} else { /*local caching*/
		vscan_inc(info->client_data.buf);
		cache_inc_data(info->client_data.buf);
	}
	writebuf2client();
}

/* ------------------------------------------------------------- **
**  Flush as much buffer as we can
** ------------------------------------------------------------- */
void writebuf2client()
{
	int i;

	i = sstr_write(info->client_data.fd, info->client_data.buf, 0);
	if (i == -1) {
		if(errno == EAGAIN) return;
		if(errno != EPIPE) debug_perr("writebuf2client()");
		closecd();
		return;
	}
	sstr_split(info->client_data.buf, NULL, 0, i);

	if (sstr_len(info->client_data.buf) == 0 && info->server_data.fd == -1)
		closecd();
	debug(".");
}

/* ------------------------------------------------------------- **
**  Flush as much buffer as we can
** ------------------------------------------------------------- */
void writebuf2server()
{
	int i;

	i = sstr_write(info->server_data.fd, info->server_data.buf, 0);
	if (i == -1) {
		if(errno == EAGAIN) return;
		if(errno != EPIPE) debug_perr("writebuf2server()");
		closesd();
		return;
	}
	sstr_split(info->server_data.buf, NULL, 0, i);

	if (sstr_len(info->server_data.buf) == 0 && info->client_data.fd == -1)
		closesd();
	debug(".");
}

/* ------------------------------------------------------------- **
**  Close data socket, and if there is nothing left to flush close
** the other one too.
** ------------------------------------------------------------- */
void closecd(void)
{
	debug("Closing client data connection\n");
	info->state = NEITHER;
	close(info->client_data.fd);
	info->client_data.fd = -1;
	if (sstr_len(info->server_data.buf) == 0) {
		debug("Closing server data connection\n");
		close(info->server_data.fd);
		info->server_data.fd = -1;
		vscan_end();
		cache_close_data();
	}
}

/* ------------------------------------------------------------- **
** Close data socket, and if there is nothing left to flush close the
** other one too.
** ------------------------------------------------------------- */
void closesd(void)
{
	debug("Closing server data connection\n");

	close(info->server_data.fd);
	vscan_end();
	cache_close_data();

	info->state = NEITHER;
	info->server_data.fd = -1;
	if (sstr_len(info->client_data.buf) == 0) {
		debug("Closing client data connection\n");
		close(info->client_data.fd);
		info->client_data.fd = -1;
	}
}
