/***************************************************************************
 *  The FreeMedForms project is a set of free, open source medical         *
 *  applications.                                                          *
 *  (C) 2008-2012 by Eric MAEKER, MD (France) <eric.maeker@gmail.com>      *
 *  All rights reserved.                                                   *
 *                                                                         *
 *  This program is free software: you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, either version 3 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program (COPYING.FREEMEDFORMS file).                   *
 *  If not, see <http://www.gnu.org/licenses/>.                            *
 ***************************************************************************/
/***************************************************************************
 *   Main developers : Eric MAEKER, <eric.maeker@gmail.com>                *
 *   Contributors :                                                        *
 *       NAME <MAIL@ADDRESS.COM>                                           *
 ***************************************************************************/
/**
 \class Utils::QButtonLineEdit
 \brief This widget can present two QToolButtons inside a QlineEdit: one left, the other on the right.
    QButtonLineEdit inherits QLineEdit
    After instanciating it, define the ToolButtons for the right and the left, or only one.
    The tooltip of the leftButton is shown in gray inside the LineEdit when the user has not typed
    anything in.
    You can delay the textChanged() signal using setDelayedSignals().
    All action pixmaps of the left and right buttons must be 16x16.
 */

#include "qbuttonlineedit.h"

#include <utils/log.h>

#include <QApplication>
#include <QTextDocument>
#include <QAction>
#include <QPainter>
#include <QPaintEvent>
#include <QKeyEvent>
#include <QStyle>
#include <QToolButton>
#include <QTimer>

#include <QDebug>

using namespace Utils;

static inline QString cleanString(const QString &s)
{
    QString ret = s;
    if (Qt::mightBeRichText(ret)) {
        ret.replace(QRegExp("<[^>]*>"), "");
        ret = ret.trimmed();
    }
    return ret;
}

namespace Utils {
namespace Internal {
class QButtonLineEditPrivate
{
public:
    QButtonLineEditPrivate(QButtonLineEdit *parent) :
        _leftButton(0), _rightButton(0), _delayed(false),
        _rightPadding(0), _leftPadding(0),
        q(parent)
    {}

    QString constructStyleSheet(const QString &extraStyleSheet)
    {
        QStringList css;

        // Create padding for buttons
        css << QString("padding-left:%1px").arg(_leftPadding);
        css << QString("padding-right:%1px").arg(_rightPadding);

        // Analyse extraStyleSheet (remove all paddings)
        if (!extraStyleSheet.isEmpty()) {
            foreach(const QString &c, extraStyleSheet.split(";", QString::SkipEmptyParts)) {
                if (!c.startsWith("paddin",Qt::CaseInsensitive))
                    css << c;
            }
        }
        return QString("%2;").arg(css.join(";"));
    }

    void setSpecificStyleSheet()
    {
        q->setStyleSheet(constructStyleSheet(_extraCss));
    }

    // Connects leftbutton
    void prepareConnections()
    {
        if (_leftButton)
            q->connect(_leftButton, SIGNAL(triggered(QAction*)), q, SLOT(leftTrig(QAction*)));
    }

    // Updates placeHolderText of the LineEdit to the text of the current QAction that was last called.
    void updatePlaceholderText()
    {
        if (_leftButton && _leftButton->defaultAction()) {
            QString t = QString("%1 %2")
                    .arg(cleanString(_leftButton->defaultAction()->text()))
                    .arg(QApplication::translate("Utils::QButtonLineEdit", "(press Alt up/down cursor to cycle)"));
            q->setPlaceholderText(t);

            QString e;
            if (!_extraToolTipContext.isEmpty() && !_extraToolTipTr.isEmpty()) {
                e = QString("%1<br />").arg(QApplication::translate(_extraToolTipContext.toUtf8(), _extraToolTipTr.toUtf8()));
            }

            if (!_extraToolTipNonTr.isEmpty())
                e += QString("%1<br />").arg(_extraToolTipNonTr);

            t = t.replace(" ", "&nbsp;");
            if (!e.contains("<br />"))
                t = QString("<p>%1%2</p>").arg(e, t.replace(" ", "&nbsp;"));
            else
                t.prepend(e);

            q->setToolTip(t);
        }
    }

public:
    QToolButton *_leftButton;
    QToolButton *_rightButton;
    QString _extraToolTipContext, _extraToolTipTr, _extraToolTipNonTr, _extraCss;
    QTimer *_timer;
    bool _delayed;
    int _rightPadding, _leftPadding;

private:
    QButtonLineEdit *q;
};
} // Internal
} // Utils

QButtonLineEdit::QButtonLineEdit(QWidget *parent) :
    QLineEdit(parent),
    d(new Internal::QButtonLineEditPrivate(this))
{
    static int handle = 0;
    handle++;
    if (objectName().isNull())
        setObjectName("QButtonLineEdit_" + QString::number(handle));
    d->_timer = new QTimer(this);
    d->_timer->setSingleShot(true);
}

QButtonLineEdit::~QButtonLineEdit()
{
    if (d)
        delete d;
    d = 0;
}

/** Delay textChanged signal emition */
void QButtonLineEdit::setDelayedSignals(bool state)
{
    d->_delayed = state;
    d->_timer->stop();
    if (state) {
        blockSignals(true);
        connect(d->_timer, SIGNAL(timeout()), this, SLOT(emitTextChangedSignal()));
    } else {
        blockSignals(false);
        disconnect(d->_timer, SIGNAL(timeout()), this, SLOT(emitTextChangedSignal()));
    }
}

/**
 \brief Define the left button for the line edit.
 QButtonLineEdit takes ownership of the button. \n
 Text of selected action is shown in gray inside the line edit when it is empty. \n
 For now pixmaps of QAction must be sized (16x16). \n
 */
void QButtonLineEdit::setLeftButton(QToolButton *button)
{
    if (!button)
        return;
    button->setParent(this);
    d->_leftButton = button;
    d->_leftButton->setStyleSheet("border:none;padding: 0 0 0 2px;");
    d->_leftButton->setCursor(Qt::ArrowCursor);

    int frameWidth = style()->pixelMetric(QStyle::PM_DefaultFrameWidth);
    d->_leftPadding = button->sizeHint().width() + frameWidth + 1;

    QSize msz = minimumSizeHint();
    setMinimumSize(qMax(msz.width(), button->sizeHint().height() + frameWidth * 2 + 2),
                   qMax(msz.height(), button->sizeHint().height() + frameWidth * 2 + 2));

    // set text to button toolTip
    d->updatePlaceholderText();
    d->prepareConnections();
    clearFocus();
    d->setSpecificStyleSheet();
}

/**
 \brief Define the right button for the line edit.
 QButtonLineEdit takes ownership of the button. \n
 For now pixmaps of QAction must be sized (16x16). \n
*/
void QButtonLineEdit::setRightButton(QToolButton * button)
{
    if (!button)
        return;
    button->setParent(this);
    d->_rightButton = button;
    d->_rightButton->setStyleSheet("border:none;padding: 0 0 0 0px;");
    d->_rightButton->setCursor(Qt::ArrowCursor);

    int frameWidth = style()->pixelMetric(QStyle::PM_DefaultFrameWidth);
    d->_rightPadding = button->sizeHint().width() + frameWidth + 1;

    QSize msz = minimumSizeHint();
    setMinimumSize(qMax(msz.width(), button->sizeHint().height() + frameWidth * 2 + 2),
                   qMax(msz.height(), button->sizeHint().height() + frameWidth * 2 + 2));
    d->prepareConnections();
    clearFocus();
    d->setSpecificStyleSheet();
}

void QButtonLineEdit::resizeEvent(QResizeEvent *)
{
    if (d->_leftButton) {
        QSize sz = d->_leftButton->sizeHint();
        int frameWidth = style()->pixelMetric(QStyle::PM_DefaultFrameWidth);
        d->_leftButton->move(rect().left() + frameWidth, (rect().bottom() + 1 - sz.height()) / 2);
    }
    if (d->_rightButton) {
        QSize sz = d->_rightButton->sizeHint();
        int frameWidth = style()->pixelMetric(QStyle::PM_DefaultFrameWidth);
        d->_rightButton->move(rect().right() - frameWidth - sz.width(), (rect().bottom() + 2 - sz.height()) / 2);
    }
}

void QButtonLineEdit::emitTextChangedSignal()
{
    blockSignals(false);
    Q_EMIT textChanged(text());
//    Q_EMIT cursorPositionChanged(int, int);
//    Q_EMIT editingFinished();
    Q_EMIT returnPressed();
//    Q_EMIT selectionChanged();
//    Q_EMIT textEdited(QString);
    blockSignals(true);
}

/** \brief slot for triggered left button */
void QButtonLineEdit::leftTrig(QAction *action)
{
    d->_leftButton->setDefaultAction(action);
    d->updatePlaceholderText();
    clearFocus();
}

void QButtonLineEdit::keyPressEvent(QKeyEvent *event)
{
    if (d->_delayed)
        d->_timer->stop();

    if (event->modifiers() & Qt::AltModifier) {
        if (!d->_leftButton)
            return;
        const QList<QAction *> &list = d->_leftButton->actions();

        if (list.count()) {
            int actual = list.indexOf(d->_leftButton->defaultAction());
            QAction *a = 0;

            if (event->key()==Qt::Key_Down) {
                ++actual;
                if (actual>=list.count())
                    actual = 0;
                a = list.at(actual);
            } else if (event->key()==Qt::Key_Up) {
                --actual;
                if (actual<0)
                    actual = list.count() - 1;
                a = list.at(actual);
            }
            if (a) {
                a->trigger();
                leftTrig(a);
                setFocus();
                QHelpEvent *e = new QHelpEvent(QEvent::ToolTip, pos(), mapToGlobal(pos()));
                QLineEdit::event(e);
                return;
            }
        }
    } else if (d->_delayed) {
        if (event->key()==Qt::Key_Enter) {
            blockSignals(false);
            Q_EMIT returnPressed();
            blockSignals(true);
        } else {
            const QString &t = text();
            if (t.isEmpty()) {
                blockSignals(false);
                Q_EMIT returnPressed();
                Q_EMIT textChanged("");
                blockSignals(true);
            } else {
                int delay = 300 - t.length()*t.length()*10;
                if (delay < 0)
                    delay = 0;
                d->_timer->start(delay);
            }
        }
    }
    QLineEdit::keyPressEvent(event);
}

/** \brief Define rounded border for the QLineEdit */
void QButtonLineEdit::setRoundedCorners()
{
    // FIXME: Buggy
    setStyleSheet(QString("QLineEdit#%1, QFrame#%1 {"
                          "border-style: groove;"
                          "border-width: 1px;"
                          "border-radius: 6px;"
                          "}").arg(objectName()));
}

void QButtonLineEdit::setTranslatableExtraToolTip(const QString &trContext, const QString &translatable)
{
    d->_extraToolTipContext = trContext;
    d->_extraToolTipTr = translatable;
}

void QButtonLineEdit::setExtraToolTip(const QString &nonTranslatable)
{
    d->_extraToolTipNonTr = nonTranslatable;
}

void QButtonLineEdit::setExtraStyleSheet(const QString &extraCss)
{
    d->_extraCss = extraCss;
    if (d->_extraCss.contains("{")) {
        int b = d->_extraCss.indexOf("{") + 1;
        int e = d->_extraCss.indexOf("}", b);
        if (e>0) {
            d->_extraCss = d->_extraCss.mid(b, e-b);
        } else {
            LOG_ERROR("Wrong styleSheet -> {} " + extraCss);
            d->_extraCss.clear();
        }
    }
    d->setSpecificStyleSheet();
}

void QButtonLineEdit::clearExtraStyleSheet()
{
    d->_extraCss.clear();
    d->setSpecificStyleSheet();
}

void QButtonLineEdit::changeEvent(QEvent *e)
{
    QWidget::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        d->updatePlaceholderText();
        break;
    default:
        break;
    }
}
