/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/


/******************************************************************
*
*    file:        knotenergy.c
*
*    Contents:  Functions calculating knot energy and its gradient
*
*                  Has both equilibrium charge distribution
*                  (for use with length constraint) and
*                  uniform charge disitribution.
*/

#include "include.h"

/* for accessing knot energy exponent as adjustable parameter */
int exponent_param;  /* parameter number */
#define KNOTPOWER_NAME "knot_power"  /* name in datafile */

static int ke_power_flag; /* set if even power */
static int ke_power;        /* half of even integer power */

#define CHARGE_ATTR_NAME "node_charge"
int charge_attr = -1;

/***************************************************************
*
*  function: knot_power_init()
*
*  purpose: initialization for all knot_energies which have variable power
*/

void knot_power_init(mode,mi)
int mode; /* energy or gradient */
struct method_instance *mi;
{ REAL hpower;

  exponent_param = lookup_global(KNOTPOWER_NAME);
  if ( exponent_param < 0 ) /* missing, so add */
    { exponent_param = add_global(KNOTPOWER_NAME);
      globals[exponent_param].value.real = 2.0;  /* default */
      globals[exponent_param].flags |=  ORDINARY_PARAM | SURFACE_PARAMETER;
    }
  hpower = globals[exponent_param].value.real/2; /* since rr is squared */
  if ( hpower == (REAL)(int)hpower  &&  hpower >= 0.)
  { ke_power_flag = 1;
    ke_power = (int)hpower; 
  }
  else ke_power_flag = 0;

  /* see if charge attribute */
   charge_attr = find_attribute(VERTEX,CHARGE_ATTR_NAME);
}


/**************************************************************
*
*  function: knot_energy()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others.
*
*  input: info about vertex is in qinfo structure.
*
*/

REAL knot_energy(v_info)
struct qinfo *v_info;
{ REAL *x = get_coord(v_info->id);
  REAL energy = 0.0;  /* for this vertex */
  vertex_id v_id;  /* other vertex */
  int i;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL rr;
  REAL hpower,p;
  REAL vcharge = charge_attr >= 0 ? 
        *((REAL*)get_extra(v_info->id,charge_attr)) : 1.;

  hpower = globals[exponent_param].value.real/2; /* since rr already square */
  FOR_ALL_VERTICES(v_id)
  { REAL *y = get_coord(v_id);
    REAL charges;
    if ( v_id <= v_info->id ) continue; /* each pair once */
    charges = charge_attr >= 0 ? 
                  *((REAL*)get_extra(v_id,charge_attr))*vcharge : 1.;
    for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
    rr = SDIM_dot(d,d);
    if ( ke_power_flag )
      for ( p = rr, i = 1 ; i < ke_power ; i++ ) p *= rr;
    else p = pow(rr,hpower);
    energy += charges/p;  /* inverse power potential */
  }
  return 2*energy; /* since each pair once */
}


/**************************************************************
*
*  function: knot_energy_gradient()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others.
*
*  input: info about vertex is in global qinfo structure.
*
*/

REAL knot_energy_gradient(v_info)
struct qinfo *v_info;
{ REAL *x = get_coord(v_info->id);
  REAL energy = 0.0;  /* for this vertex */
  vertex_id v_id;
  int i;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL rr,p,fact;
  REAL power,hpower;
  REAL vcharge = charge_attr >= 0 ? 
        *((REAL*)get_extra(v_info->id,charge_attr)) : 1.;

  power = globals[exponent_param].value.real;
  hpower = power/2;
  FOR_ALL_VERTICES(v_id)
  { REAL *y = get_coord(v_id);
    REAL charges;
    if ( equal_id(v_info->id,v_id) ) continue;
    charges = charge_attr >= 0 ? 
              *((REAL*)get_extra(v_id,charge_attr))*vcharge : 1.;
    for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
    rr = SDIM_dot(d,d);
    if ( ke_power_flag )
      for ( p = rr, i = 1 ; i < ke_power ; i++ ) p *= rr;
    else p = pow(rr,hpower);
    energy += charges/p;
    fact = power/(p*rr);
    for ( i = 0 ; i < SDIM ; i++ ) 
      v_info->grad[0][i] -= 2*charges*fact*d[i];
  }
  return energy;
}

/*******************************************************
*
*     function: knot_energy_hessian()
*
*     purpose: calculate knot energy hessian
*/


REAL knot_energy_hessian(v_info)
struct qinfo *v_info;
{
  REAL *x = get_coord(v_info->id);
  REAL energy = 0.0;  /* for this vertex */
  vertex_id w_id;
  int i,j;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL rr,p,fact;
  REAL power,hpower;
  MAT2D(hessvw,MAXCOORD,MAXCOORD);
  MAT2D(hessvv,MAXCOORD,MAXCOORD);
  REAL modulus;
  REAL vcharge = charge_attr >= 0 ? 
        *((REAL*)get_extra(v_info->id,charge_attr)) : 1.;

  modulus = METH_INST[v_info->method].modulus
                      *GEN_QUANTS[METH_INST[v_info->method].quant].modulus;
  /* need modulus due to direct insertion into hessian */

  for(i=0;i<SDIM;i++) for(j=0;j<SDIM;j++) hessvv[i][j]=0.;

  power = globals[exponent_param].value.real;
  hpower = power/2;
  FOR_ALL_VERTICES(w_id)
  { REAL *y = get_coord(w_id);
    REAL charges;
    if ( equal_id(v_info->id,w_id) ) continue;
    charges = charge_attr >= 0 ? 
              *((REAL*)get_extra(w_id,charge_attr))*vcharge : 1.;
    for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
    rr = SDIM_dot(d,d);
    if ( ke_power_flag )
      for ( p = rr, i = 1 ; i < ke_power ; i++ ) p *= rr;
    else p = pow(rr,hpower);
    energy += charges/p;
    fact = power/(p*rr);
    for ( i = 0 ; i < SDIM ; i++ ) 
      v_info->grad[0][i] -= 2*charges*fact*d[i];
    for ( i=0; i<SDIM; i++ )
      for ( j=0; j<SDIM; j++ )
      {
        register REAL hessij = fact*((power+2)/rr * d[i]*d[j] - (i==j));
        hessij *= modulus*charges;
        hessvw[i][j] = - hessij;
        hessvv[i][j] += 2*hessij;
      }
    fill_mixed_entry(v_info->id, w_id, hessvw);
  }
  fill_self_entry(v_info->id, hessvv);
  return modulus*energy;
}

/********************************************************************/
/*  This set of routines is for the gradient^2 of the last energy */
/*  assuming the points are constrained to the unit sphere */

static REAL *charge_grads = NULL;
static int cg_nverts = 0;

/***************************************************************
*
*  function: charge_gradient_init()
*
*  purpose: initialization for charge_gradient() and 
*              charge_gradient_gradient().
*/
void charge_gradient_init(mode,mi)
int mode; /* energy or gradient */
struct method_instance *mi;
{
    REAL power;
    vertex_id u_id,v_id;  /* two vertices */
    int nv;

    knot_power_init(mode,mi);
    power = globals[exponent_param].value.real;

    if ((nv = web.skel[VERTEX].max_ord+1) != cg_nverts)
    {
         if (charge_grads) myfree((char*)charge_grads);
         charge_grads = (REAL*)mycalloc(nv, MAXCOORD*sizeof(REAL));
         cg_nverts = nv;
	 }

    FOR_ALL_VERTICES(u_id)
    {
      int i;
      REAL *x = get_coord(u_id);
      REAL *w = charge_grads+MAXCOORD*ordinal(u_id);

      for ( i = 0 ; i < SDIM ; i++ ) w[i] = 0.;
      FOR_ALL_VERTICES(v_id)
      {
          REAL *y = get_coord(v_id);
          REAL d[MAXCOORD];         /* difference vector */
          REAL rr;

          if (u_id==v_id) continue;
          for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
          rr = SDIM_dot(d,d);
          for ( i = 0 ; i < SDIM ; i++ ) w[i] -= d[i]/pow(rr,(power+2)/2);
      }
   }
}

/**************************************************************
*
*  function: charge_gradient()
*  
*  purpose: calculates force^2 on one vertex from charges at all others
*
*  input: info about vertex is in qinfo structure.
*
*/

REAL charge_gradient(v_info)
struct qinfo *v_info;
{
  REAL *x = get_coord(v_info->id);
  REAL *w = charge_grads+MAXCOORD*ordinal(v_info->id);
  REAL wx = SDIM_dot(w,x);
  return SDIM_dot(w,w) - wx*wx;
}


/**************************************************************
*
*  function: charge_gradient_gradient()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others.
*
*  input: info about vertex is in global qinfo structure.
*
*/

REAL charge_gradient_gradient(v_info)
struct qinfo *v_info;
{ REAL *xi = get_coord(v_info->id);
  vertex_id v_id;
  int i;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL *wi,*wj; /* force vectors at vertices */
  REAL yi[MAXCOORD], yd[MAXCOORD]; /* more forces */
  REAL rr,p,ddd,power;

  power = globals[exponent_param].value.real;
  wi = charge_grads+MAXCOORD*ordinal(v_info->id);
  ddd = SDIM_dot(wi,xi);
  for ( i = 0 ; i < SDIM ; i++ ) 
      v_info->grad[0][i] = -2*ddd*wi[i]; /* diagonal term in gradient */
  for ( i = 0 ; i < SDIM ; i++ )  yi[i] = wi[i] - ddd*xi[i];
  FOR_ALL_VERTICES(v_id)
  { REAL *xj = get_coord(v_id);
    if ( equal_id(v_info->id,v_id) ) continue;
    for ( i = 0 ; i < SDIM ; i++ ) d[i] = xi[i] - xj[i];
    rr = SDIM_dot(d,d);
    wj = charge_grads+MAXCOORD*ordinal(v_id);
    ddd = SDIM_dot(wj,xj);
    for ( i = 0 ; i < SDIM ; i++ ) yd[i] = yi[i] - wj[i] + ddd*xj[i];
    ddd = SDIM_dot(yd,d);
    p = pow(rr,(power+2)/2);  /* power of distance */
    for ( i = 0 ; i < SDIM ; i++ )
      v_info->grad[0][i] += (-2*yd[i] + 2*(power+2)*ddd*d[i]/rr)/p;
  }
  return SDIM_dot(yi,yi);
}

/* Next set of routines is for uniform charge distribution */

/***************************************************************
*
*  function: uniform_knot_energy_init()
*
*  purpose: initialization for knot_energy() and 
*              knot_energy_gradient().
*/

void uniform_knot_energy_init(mode,mi)
int mode; /* energy or gradient */
struct method_instance *mi;
{
  edge_id e_id;
  vertex_id v_id;

  knot_power_init(mode,mi);

  FOR_ALL_VERTICES(v_id)
  {
     set_vertex_star(v_id,0.);
     get_vertex_evalence(v_id);
  }
  FOR_ALL_EDGES(e_id)
  {
     calc_edge(e_id);
     add_vertex_star(get_edge_tailv(e_id),get_edge_length(e_id)/2);
     add_vertex_star(get_edge_headv(e_id),get_edge_length(e_id)/2);
  }
}

/**************************************************************
*
*  function: uniform_knot_energy()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others. Charge at vertex is length of star.
*
*  input: info about vertex is in global qinfo structure.
*
*/

REAL uniform_knot_energy(v_info)
struct qinfo *v_info;
{ REAL *x = get_coord(v_info->id);
  REAL energy = 0.0;  /* for this vertex */
  vertex_id v_id;  /* other vertex */
  int i;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL rr;
  REAL hpower;
  REAL ti;  /* length weights of vertices, ti for home vertex */
  REAL p;

  hpower = globals[exponent_param].value.real/2; /* since rr already square */
  ti = get_vertex_star(v_info->id);
  FOR_ALL_VERTICES(v_id)
  { REAL *y = get_coord(v_id);
    if ( v_id <= v_info->id ) continue; /* each pair once */
      for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
    rr = SDIM_dot(d,d);
    if ( ke_power_flag )
      for ( p = rr, i = 1 ; i < ke_power ; i++ ) p *= rr;
    else p = pow(rr,hpower);
    energy += ti*get_vertex_star(v_id)/p;
  }
  return 2*energy; /* since each pair once */
}


/**************************************************************
*
*  function: uniform_knot_energy_gradient()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others.
*
*  input: info about vertex is in global qinfo structure.
*
*/

REAL uniform_knot_energy_gradient(v_info)
struct qinfo *v_info;
{ REAL *x = get_coord(v_info->id);
  REAL energy = 0.0;  /* for this vertex */
  vertex_id v_id;
  int i;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL rr,p;
  REAL power,hpower;
  REAL ti,tj;  /* length weights of vertices, ti for home vertex */
  edge_id e_id;
  REAL left[MAXCOORD],right[MAXCOORD]; /* edge vectors */
  REAL sum,sumleft,sumright;
  REAL *xleft,*xright;
  REAL leftmag,rightmag;

  power = globals[exponent_param].value.real;
  hpower = power/2;
  for ( i = 0 ; i < SDIM ; i++ ) 
      v_info->grad[0][i] = 0.0; /* initialize gradient */
  ti = get_vertex_star(v_info->id);
  e_id = get_vertex_edge(v_info->id);
  xright = get_coord(get_edge_headv(e_id));
  e_id = get_next_tail_edge(e_id);  /* assuming exactly two edges per vertex */
  xleft = get_coord(get_edge_headv(e_id));
  sum = sumleft = sumright = 0.0;
  FOR_ALL_VERTICES(v_id)
  { REAL *y = get_coord(v_id);
    REAL fact;

    tj = get_vertex_star(v_id);

    for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
    rr = SDIM_dot(d,d);
    if (v_id != v_info->id)
    {
      if ( ke_power_flag )
        for ( p = rr, i = 1 ; i < ke_power ; i++ ) p *= rr;
      else p = pow(rr,hpower);
      fact = ti*tj/p;
      energy += fact;
      fact *= 2*power/rr;
      for ( i = 0 ; i < SDIM ; i++ ) 
        v_info->grad[0][i] -= fact*d[i];
      sum += tj/p;
    }

    for ( i = 0 ; i < SDIM ; i++ ) d[i] = xleft[i] - y[i];
    rr = SDIM_dot(d,d);
    if ( rr > 1e-18) /* don't do self */
     { 
       if ( ke_power_flag )
         for ( p = rr, i = 1 ; i < ke_power ; i++ ) p *= rr;
       else p = pow(rr,hpower);
       sumleft += tj/p;
     }

    for ( i = 0 ; i < SDIM ; i++ ) d[i] = xright[i] - y[i];
    rr = SDIM_dot(d,d);
    if ( rr > 1e-18) /* don't do self */
     {
        if ( ke_power_flag )
          for ( p = rr, i = 1 ; i < ke_power ; i++ ) p *= rr;
        else p = pow(rr,hpower);
        sumright += tj/p;
     }
  }
  for ( i = 0 ; i < SDIM ; i++ )
  { left[i] = x[i] - xleft[i];
    right[i] = x[i] - xright[i];
  }
  leftmag = sqrt(SDIM_dot(left,left));
  rightmag = sqrt(SDIM_dot(right,right));
  for ( i = 0 ; i < SDIM ; i++ )
      v_info->grad[0][i] += ((left[i]/leftmag + right[i]/rightmag)*sum
                     + left[i]/leftmag*sumleft + right[i]/rightmag*sumright);
  return energy;
}

/******************************************************************
*
*  function: uniform_normalization()
*
*  purpose: calculates internal knot energy to normalize
*              singular divergence of integral.
*/

REAL uniform_normalization(v_info)
struct qinfo *v_info;
{
  vertex_id v_id;
  edge_id e_id,ee_id;
  REAL ti,tj;
  REAL dist=0.,energy=0.;
  REAL power;

  e_id = get_vertex_edge(v_info->id);
  if ( !valid_id(e_id) ) return 0.0;
  ti = get_edge_length(e_id);
  ee_id = get_next_tail_edge(e_id); /* assuming exactly two edges per vertex */
  ti = (ti + get_edge_length(ee_id))/2;

  power = globals[exponent_param].value.real;
  for ( v_id = get_edge_headv(e_id) ; v_id != v_info->id ;
             e_id = ee_id, v_id = get_edge_headv(e_id) )
  {
    dist += (tj = get_edge_length(e_id));
    ee_id = inverse_id(get_next_head_edge(e_id));
    tj = (tj + get_edge_length(ee_id))/2;
    energy += ti*tj/pow(dist,power);
  }
  return 2*energy;
}

/******************************************************************
*
*  function: uniform_binormalization()
*
*  purpose: calculates internal knot energy to normalize
*              singular divergence of integral.  Uses shorter distance.
*/

REAL uniform_binormalization(v_info)
struct qinfo *v_info;
{
  vertex_id v_id;
  edge_id e_id,ee_id;
  REAL ti,tj;
  REAL dist=0.,energy=0.,comp_len;
  REAL power;

  e_id = get_vertex_edge(v_info->id);
  if ( !valid_id(e_id) ) return 0.0;
  comp_len = ti = get_edge_length(e_id);
  ee_id = get_next_tail_edge(e_id); /* assuming exactly two edges per vertex */
  ti = (ti + get_edge_length(ee_id))/2;

  power = globals[exponent_param].value.real;
  for ( ee_id = inverse_id(get_next_head_edge(e_id)); ee_id != e_id;
          ee_id = inverse_id(get_next_head_edge(ee_id)) )
     comp_len += get_edge_length(ee_id);
  for ( v_id = get_edge_headv(e_id) ; v_id != v_info->id ;
             e_id = ee_id, v_id = get_edge_headv(e_id) )
  {
    dist += (tj = get_edge_length(e_id));
    ee_id = inverse_id(get_next_head_edge(e_id));
    tj = (tj + get_edge_length(ee_id))/2;
    energy += ti*tj/pow((2*dist<comp_len? dist : comp_len-dist), power);
  }
  return energy;
}

/***************************************************************
*
*  function: facet_knot_energy_init()
*
*  purpose: initialization for knot_energy() and 
*              knot_energy_gradient().
*/

#define SURF_KNOTPOW_NAME "surface_knot_power"

void facet_knot_energy_init(mode,mi)
int mode; /* energy or gradient */
struct method_instance *mi;
{
  REAL *x;
  vertex_id v_id,vv_id; 
  int i;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL rr;
  REAL power;
  REAL tj;  
  facetedge_id fe;
  REAL area,s11,s12,s22,side1[MAXCOORD],side2[MAXCOORD];
  facet_id f_id;

  exponent_param = lookup_global(SURF_KNOTPOW_NAME);
  if ( exponent_param < 0 ) /* missing, so add */
  { exponent_param = add_global(SURF_KNOTPOW_NAME);
    globals[exponent_param].value.real = 4.0;  /* default */
    globals[exponent_param].flags |=  ORDINARY_PARAM | SURFACE_PARAMETER;
  }
  power = globals[exponent_param].value.real;
    

#if defined(PVM) || defined(MPI)
  if ( self_ord == HOST_ORD )
    { /* basic sums */
      if ( f_sums ) myfree((char*)f_sums);
      f_sums = (REAL *)mycalloc(web.skel[VERTEX].max_ord+1,sizeof(REAL));
      pvm_facet_knot_init(f_sums);
    }
#else

  /* will need areas of all vertex stars */
  FOR_ALL_VERTICES(v_id) set_vertex_star(v_id,0.0);
  FOR_ALL_FACETS(f_id)
    { fe = get_facet_fe(f_id);
      get_edge_side(get_fe_edge(fe),side1);
      get_edge_side(get_fe_edge(get_next_edge(fe)),side2);
      s11 = SDIM_dot(side1,side1);
      s22 = SDIM_dot(side2,side2);
      s12 = SDIM_dot(side1,side2);
      area = sqrt(s11*s22 - s12*s12)/2;
      for ( i = 0 ; i < FACET_VERTS ; i++ )
        { vv_id = get_fe_tailv(fe);
          add_vertex_star(vv_id,area/3);
          fe = get_next_edge(fe);
        }
    }

  /* basic sums */
  if ( f_sums ) myfree((char*)f_sums);
  f_sums = (REAL *)mycalloc(web.skel[VERTEX].max_ord+1,sizeof(REAL));

  FOR_ALL_VERTICES(v_id)
  { int ordv = ordinal(v_id);
     REAL val;
     REAL ti = get_vertex_star(v_id);
     x = get_coord(v_id);
     FOR_ALL_VERTICES(vv_id)
     { REAL *y = get_coord(vv_id);
        int ordvv = ordinal(vv_id);
        if ( ordvv <= ordv ) continue;
        tj = get_vertex_star(vv_id);
        for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
        rr = SDIM_dot(d,d);
        val = pow(rr,power/2);  /* inverse power potential */
        f_sums[ordv] += tj/val;
        f_sums[ordvv] += ti/val;
     }
  }
#endif
}

/**************************************************************
*
*  function: facet_knot_energy()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others. Charge at vertex is area of star.
*
*  input: info about vertex is in global qinfo structure.
*
*/

REAL facet_knot_energy(v_info)
struct qinfo *v_info;
{ REAL ti,energy;

  ti = get_vertex_star(v_info->id);
  energy = ti*f_sums[ordinal(v_info->id)];
  return energy; /* because pair counts in both orders */
}


/**************************************************************
*
*  function: facet_knot_energy_gradient()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others.
*
*  input: info about vertex is in qinfo structure.
*
*/

REAL facet_knot_energy_gradient(v_info)
struct qinfo *v_info;
{ REAL *x = get_coord(v_info->id);
  vertex_id v_id;
  int i;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL rr,p;
  REAL power,halfpower;
  REAL ti,tj;  /* length weights of vertices, ti for home vertex */
  facetedge_id fe,start_fe,next_fe;
  REAL sumi,sumj,sumjj,area;
  REAL s11,s12,s22,side1[MAXCOORD],side2[MAXCOORD];
  REAL da[MAXCOORD],sum[MAXCOORD];

  ti = get_vertex_star(v_info->id);
  power = globals[exponent_param].value.real;
  halfpower = power/2;
  for ( i = 0 ; i < SDIM ; i++ ) 
    v_info->grad[0][i] = 0.0; /* intialize gradient */

  /* distant change part */
  for ( i = 0 ; i < SDIM ; i++ ) sum[i] = 0.0;
  FOR_ALL_VERTICES(v_id)
  { REAL *y = get_coord(v_id);

    tj = get_vertex_star(v_id);
    for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
    rr = SDIM_dot(d,d);
    if ( rr > 1e-12 ) /* don't do self */
     { p = power*tj/rr/pow(rr,halfpower);  /* inverse power potential */
       for ( i = 0 ; i < SDIM ; i++ ) 
          sum[i] -= p*d[i];
     }
  }
  for ( i = 0 ; i < SDIM ; i++ )
      v_info->grad[0][i] += 2*ti*sum[i];
 
  /* area change part */
  /* go around all neighbor vertices */
  start_fe = get_vertex_fe(v_info->id);
  sumi = f_sums[ordinal(v_info->id)];
  if ( valid_id(start_fe) )
    for ( fe = start_fe ; ; )
    { 
      next_fe = inverse_id(get_next_facet(get_prev_edge(fe)));
      get_edge_side(get_fe_edge(fe),side1);
      get_edge_side(get_fe_edge(get_next_edge(fe)),side2);
      s11 = SDIM_dot(side1,side1);
      s12 = SDIM_dot(side1,side2);
      s22 = SDIM_dot(side2,side2);
      area = sqrt(s11*s22 - s12*s12);
      for ( i = 0 ; i < SDIM ; i++ )
         da[i] = (s12*side2[i] - s22*side1[i])/area/6;
      sumj = f_sums[ordinal(get_fe_headv(fe))];
      sumjj = f_sums[ordinal(get_fe_headv(next_fe))];
      for ( i = 0 ; i < SDIM ; i++ )
         v_info->grad[0][i] += 2*(sumj*da[i] + sumjj*da[i] + sumi*da[i]);
      if ( next_fe == start_fe ) break;
      fe = next_fe;
    }
  return 2*ti*sumi;
}

/**************************************************************************/
/**************************************************************************/

/****************************************************************************
*
* function: facet_knot_energy_fix()
*
* purpose: provide adjacent vertex correction to facet_knot_energy
*
*/

static REAL *fix_sums;  /* for sums to other vertices connected by edges */

/***************************************************************
*
*  function: facet_knot_energy_fix_init()
*
*/

void facet_knot_energy_fix_init(mode,mi)
int mode; /* energy or gradient */
struct method_instance *mi;
{
  edge_id e_id;
  REAL *x;
  vertex_id v_id,vv_id; 
  int i;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL rr;
  REAL power;
  facetedge_id fe;
  REAL area,s11,s12,s22,side1[MAXCOORD],side2[MAXCOORD];
  facet_id f_id;

  exponent_param = lookup_global(SURF_KNOTPOW_NAME);
  if ( exponent_param < 0 ) /* missing, so add */
   { exponent_param = add_global(SURF_KNOTPOW_NAME);
     globals[exponent_param].value.real = 4.0;  /* default */
     globals[exponent_param].flags |=  ORDINARY_PARAM | SURFACE_PARAMETER;
   }
    
  /* will need areas of all vertex stars */
  FOR_ALL_VERTICES(v_id) set_vertex_star(v_id,0.0);
  FOR_ALL_FACETS(f_id)
    { fe = get_facet_fe(f_id);
      get_edge_side(get_fe_edge(fe),side1);
      get_edge_side(get_fe_edge(get_next_edge(fe)),side2);
      s11 = SDIM_dot(side1,side1);
      s22 = SDIM_dot(side2,side2);
      s12 = SDIM_dot(side1,side2);
      area = sqrt(s11*s22 - s12*s12)/2;
      for ( i = 0 ; i < 3 ; i++ )
        { vv_id = get_fe_tailv(fe);
          add_vertex_star(vv_id,area/3);
          fe = get_next_edge(fe);
        }
    }

  /* get basic sums */
  power = globals[exponent_param].value.real/2; /* since rr already square */
  if ( fix_sums ) myfree((char*)fix_sums);
  fix_sums = (REAL *)mycalloc(web.skel[VERTEX].max_ord+1,sizeof(REAL));
  FOR_ALL_EDGES(e_id)
  { 
     REAL val;
     int ordv;
     REAL *y;
     REAL ti,tj;
     int ordvv;
     v_id = get_edge_tailv(e_id);
     vv_id = get_edge_headv(e_id);
     ordv = ordinal(v_id);
     ordvv = ordinal(vv_id);
     x = get_coord(v_id);
     y = get_coord(vv_id);
     ti = get_vertex_star(v_id);
     tj = get_vertex_star(vv_id);
     for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
     rr = SDIM_dot(d,d);
     val = pow(rr,power);  /* inverse power potential */
     fix_sums[ordv] += tj/val;
     fix_sums[ordvv] += ti/val;
  }
}

/**************************************************************
*
*  function: facet_knot_energy_fix()
*  
*  purpose: calculates energy of one vertex due to potential
*              with its neighbors. Charge at vertex is area of star.
*
*  input: info about vertex is in global qinfo structure.
*
*/

REAL facet_knot_energy_fix(v_info)
struct qinfo *v_info;
{ REAL ti,energy;

  ti = get_vertex_star(v_info->id);
  energy = ti*fix_sums[ordinal(v_info->id)];
  return energy;
}


/**************************************************************
*
*  function: facet_knot_energy_fix_gradient()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others.
*
*  input: info about vertex is in global qinfo structure.
*
*/

REAL facet_knot_energy_fix_gradient(v_info)
struct qinfo *v_info;
{ REAL *x = get_coord(v_info->id);
  vertex_id v_id;
  int i;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL rr,p;
  REAL power,halfpower;
  REAL ti,tj;  /* length weights of vertices, ti for home vertex */
  facetedge_id fe,start_fe,next_fe;
  REAL sumi,sumj,sumjj,area;
  REAL s11,s12,s22,side1[MAXCOORD],side2[MAXCOORD];
  REAL da[MAXCOORD],sum[MAXCOORD];

  ti = get_vertex_star(v_info->id);
  power = globals[exponent_param].value.real;
  halfpower = power/2;
  for ( i = 0 ; i < SDIM ; i++ ) 
    v_info->grad[0][i] = 0.0; /* intialize gradient */

  /* distant change part */
  for ( i = 0 ; i < SDIM ; i++ ) sum[i] = 0.0;
  /* go around all neighbor vertices */
  start_fe = get_vertex_fe(v_info->id);
  if (valid_id(start_fe))
    for ( fe = start_fe ; ; )
    { 
      REAL *y;
      v_id = get_fe_headv(fe);
      y = get_coord(v_id);

      tj = get_vertex_star(v_id);
      for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
      rr = SDIM_dot(d,d);
      if ( rr > 1e-12 ) /* don't do self */
       { p = power*tj/rr/pow(rr,halfpower);  /* inverse power potential */
         for ( i = 0 ; i < SDIM ; i++ ) 
           sum[i] -= p*d[i];
       }
      next_fe = inverse_id(get_next_facet(get_prev_edge(fe)));
      if ( next_fe == start_fe ) break;
      fe = next_fe;
    }
  for ( i = 0 ; i < SDIM ; i++ )
      v_info->grad[0][i] += ti*sum[i];
 
  /* area change part */
  /* go around all neighbor vertices */
  start_fe = get_vertex_fe(v_info->id);
  sumi = fix_sums[ordinal(v_info->id)];
  if (valid_id(start_fe))
    for ( fe = start_fe ; ; )
    { 
      next_fe = inverse_id(get_next_facet(get_prev_edge(fe)));
      get_edge_side(get_fe_edge(fe),side1);
      get_edge_side(get_fe_edge(get_next_edge(fe)),side2);
      s11 = SDIM_dot(side1,side1);
      s12 = SDIM_dot(side1,side2);
      s22 = SDIM_dot(side2,side2);
      area = sqrt(s11*s22 - s12*s12);
      for ( i = 0 ; i < SDIM ; i++ )
         da[i] = (s12*side2[i] - s22*side1[i])/area/6;
      sumj = fix_sums[ordinal(get_fe_headv(fe))];
      sumjj = fix_sums[ordinal(get_fe_headv(next_fe))];
      for ( i = 0 ; i < SDIM ; i++ )
         v_info->grad[0][i] += sumj*da[i] + sumjj*da[i] + sumi*da[i];
      if ( next_fe == start_fe ) break;
      fe = next_fe;
    }
  return ti*sumi;
}


/**************************************************************************/
/**************************************************************************/

/***************************************************************
*
*  function: xfacet_knot_energy_init()
*
*  purpose: initialization for knot_energy() and 
*              knot_energy_gradient().
*  Does not do fsums in attempt to be more efficient with PVM or MPI
*/


void xfacet_knot_energy_init(mode,mi)
int mode; /* energy or gradient */
struct method_instance *mi;
{ vertex_id v_id,vv_id; 
  int i;
  facetedge_id fe;
  REAL area,s11,s12,s22,side1[MAXCOORD],side2[MAXCOORD];
  facet_id f_id;

  exponent_param = lookup_global(SURF_KNOTPOW_NAME);
  if ( exponent_param < 0 ) /* missing, so add */
        { exponent_param = add_global(SURF_KNOTPOW_NAME);
          globals[exponent_param].value.real = 4.0;  /* default */
          globals[exponent_param].flags |=  ORDINARY_PARAM | SURFACE_PARAMETER;
        }


  /* will need areas of all vertex stars */
  FOR_ALL_VERTICES(v_id) set_vertex_star(v_id,0.0);
  FOR_ALL_FACETS(f_id)
    { fe = get_facet_fe(f_id);
      get_edge_side(get_fe_edge(fe),side1);
      get_edge_side(get_fe_edge(get_next_edge(fe)),side2);
      s11 = SDIM_dot(side1,side1);
      s22 = SDIM_dot(side2,side2);
      s12 = SDIM_dot(side1,side2);
      area = sqrt(s11*s22 - s12*s12)/2;
      for ( i = 0 ; i < 3 ; i++ )
        { vv_id = get_fe_tailv(fe);
          add_vertex_star(vv_id,area/3);
          fe = get_next_edge(fe);
        }
    }

}

/**************************************************************
*
*  function: facet_knot_energy()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others. Charge at vertex is area of star.
*
*  input: info about vertex is in global qinfo structure.
*
*/

REAL xfacet_knot_energy(v_info)
struct qinfo *v_info;
{ REAL ti,energy;

  ti = get_vertex_star(v_info->id);
  energy = ti*f_sums[ordinal(v_info->id)];
  return energy;
}


/**************************************************************
*
*  function: facet_knot_energy_gradient()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others.
*
*  input: info about vertex is in qinfo structure.
*
*/

REAL xfacet_knot_energy_gradient(v_info)
struct qinfo *v_info;
{ REAL *x = get_coord(v_info->id);
  vertex_id v_id;
  int i;
  REAL d[MAXCOORD]; /* difference vector between vertices */
  REAL rr,p;
  REAL power,halfpower;
  REAL ti,tj;  /* length weights of vertices, ti for home vertex */
  facetedge_id fe,start_fe,next_fe;
  REAL sumi,sumj,sumjj,area;
  REAL s11,s12,s22,side1[MAXCOORD],side2[MAXCOORD];
  REAL da[MAXCOORD],sum[MAXCOORD];

  ti = get_vertex_star(v_info->id);
  power = globals[exponent_param].value.real;
  halfpower = power/2;
  for ( i = 0 ; i < SDIM ; i++ ) 
    v_info->grad[0][i] = 0.0; /* intialize gradient */

  /* distant change part */
  for ( i = 0 ; i < SDIM ; i++ ) sum[i] = 0.0;
  FOR_ALL_VERTICES(v_id)
  { REAL *y = get_coord(v_id);

    tj = get_vertex_star(v_id);
    for ( i = 0 ; i < SDIM ; i++ ) d[i] = x[i] - y[i];
    rr = SDIM_dot(d,d);
    if ( rr > 1e-12 ) /* don't do self */
     { p = power*tj/rr/pow(rr,halfpower);  /* inverse power potential */
       for ( i = 0 ; i < SDIM ; i++ ) 
          sum[i] -= p*d[i];
     }
  }
  for ( i = 0 ; i < SDIM ; i++ )
      v_info->grad[0][i] += ti*sum[i];
 
  /* area change part */
  /* go around all neighbor vertices */
  start_fe = get_vertex_fe(v_info->id);
  sumi = f_sums[ordinal(v_info->id)];
  if ( valid_id(start_fe) )
    for ( fe = start_fe ; ; )
    { 
      next_fe = inverse_id(get_next_facet(get_prev_edge(fe)));
      get_edge_side(get_fe_edge(fe),side1);
      get_edge_side(get_fe_edge(get_next_edge(fe)),side2);
      s11 = SDIM_dot(side1,side1);
      s12 = SDIM_dot(side1,side2);
      s22 = SDIM_dot(side2,side2);
      area = sqrt(s11*s22 - s12*s12);
      for ( i = 0 ; i < SDIM ; i++ )
         da[i] = (s12*side2[i] - s22*side1[i])/area/6;
      sumj = f_sums[ordinal(get_fe_headv(fe))];
      sumjj = f_sums[ordinal(get_fe_headv(next_fe))];
      for ( i = 0 ; i < SDIM ; i++ )
         v_info->grad[0][i] += sumj*da[i] + sumjj*da[i] + sumi*da[i];
      if ( next_fe == start_fe ) break;
      fe = next_fe;
    }
  return ti*sumi;
}

/**************************************************************************/
/**************************************************************************


buck knot energy 

Suggested by Gregory Buck

Between pairs of edges, energy is 1/(d1+d2+d3+d4-2(L1+l2))
where d's are cross distances between vertices and L's are
edge lengths.  Idea is to provide infinite barrier to edge
crossing.

******************************************************************/

/**************************************************************
*
*  function: buck_knot_energy()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others. Charge at vertex is area of star.
*
*  input: info about vertex is in global qinfo structure.
*
*/

REAL buck_knot_energy(v_info)
struct qinfo *v_info;
{ edge_id e1 = v_info->id,e2;
  REAL *x1,*x2,*yy1,*y2; /* end coordinates */
  REAL d1,d2,d3,d4,L1,L2;
  REAL power,denom;
  REAL energy = 0.0;
  REAL dx[MAXCOORD];
  int j;

  power = globals[exponent_param].value.real;
  x1 = get_coord(get_edge_tailv(e1));
  x2 = get_coord(get_edge_headv(e1));
  for ( j = 0 ; j < SDIM ; j++ ) dx[j] = x2[j] - x1[j];
  L1 = sqrt(SDIM_dot(dx,dx));
  FOR_ALL_EDGES(e2)
    { if ( e2 <= e1 ) continue; /* each pair once */
      yy1 = get_coord(get_edge_tailv(e2));
      y2 = get_coord(get_edge_headv(e2));
      for ( j = 0 ; j < SDIM ; j++ ) dx[j] = y2[j] - yy1[j];
      L2 = sqrt(SDIM_dot(dx,dx));
      for ( j = 0 ; j < SDIM ; j++ ) dx[j] = yy1[j] - x1[j];
      d1 = sqrt(SDIM_dot(dx,dx));
      if ( d1 <= 0.0 ) continue; /* prevent division by 0 */
      for ( j = 0 ; j < SDIM ; j++ ) dx[j] = yy1[j] - x2[j];
      d2 = sqrt(SDIM_dot(dx,dx));
      if ( d2 <= 0.0 ) continue; /* for adjacent edges     */
      for ( j = 0 ; j < SDIM ; j++ ) dx[j] = x2[j] - y2[j];
      d3 = sqrt(SDIM_dot(dx,dx));
      if ( d3 <= 0.0 ) continue; /* prevent division by 0 */
      for ( j = 0 ; j < SDIM ; j++ ) dx[j] = y2[j] - x1[j];
      d4 = sqrt(SDIM_dot(dx,dx));
      if ( d4 <= 0.0 ) continue; /* prevent division by 0 */
      denom = d1+d2+d3+d4-2*(L1+L2);
      if ( denom <= 0.0 )
         kb_error(1543,"Buck denominator nonpositive.\n",RECOVERABLE);

      energy += pow(denom ,-power);
    }
  return 2*energy; /* since each pair once */
}

/**************************************************************
*
*  function: buck_knot_energy_gradient()
*  
*  purpose: calculates energy of one vertex due to potential
*              with all others.
*
*  input: info about vertex is in global qinfo structure.
*
*/

REAL buck_knot_energy_gradient(v_info)
struct qinfo *v_info;
{ edge_id e1 = v_info->id,e2;
  REAL *x1,*x2,*yy1,*y2; /* end coordinates */
  REAL d1,d2,d3,d4,L1,L2;
  REAL power;
  REAL energy = 0.0;
  REAL denom,coeff;
  int i,j;
  REAL dL1[MAXCOORD], dL2[MAXCOORD], dd1[MAXCOORD], dd2[MAXCOORD], 
         dd3[MAXCOORD], dd4[MAXCOORD];

  for ( i = 0 ; i < 2 ; i++ )
     for ( j = 0 ; j < SDIM ; j++ ) v_info->grad[i][j] = 0.0;
  power = globals[exponent_param].value.real;
  x1 = get_coord(get_edge_tailv(e1));
  x2 = get_coord(get_edge_headv(e1));
  for ( j = 0 ; j < SDIM ; j++ ) dL1[j] = x2[j] - x1[j];
  L1 = sqrt(SDIM_dot(dL1,dL1));
  FOR_ALL_EDGES(e2)
    { 
      yy1 = get_coord(get_edge_tailv(e2));
      y2 = get_coord(get_edge_headv(e2));
      for ( j = 0 ; j < SDIM ; j++ ) dL2[j] = y2[j] - yy1[j];
      L2 = sqrt(SDIM_dot(dL2,dL2));
      for ( j = 0 ; j < SDIM ; j++ ) dd1[j] = x1[j] - yy1[j];
      d1 = sqrt(SDIM_dot(dd1,dd1));
      if ( d1 <= 0.0 ) continue; /* prevent division by 0 */
      for ( j = 0 ; j < SDIM ; j++ ) dd2[j] = yy1[j] - x2[j];
      d2 = sqrt(SDIM_dot(dd2,dd2));
      if ( d2 <= 0.0 ) continue; /* for identical vertices */
      for ( j = 0 ; j < SDIM ; j++ ) dd3[j] = x2[j] - y2[j];
      d3 = sqrt(SDIM_dot(dd3,dd3));
      if ( d3 <= 0.0 ) continue; /* since contribution  */
      for ( j = 0 ; j < SDIM ; j++ ) dd4[j] = y2[j] - x1[j];
      d4 = sqrt(SDIM_dot(dd4,dd4));
      if ( d4 <= 0.0 ) continue; /* fixed at 0.  Burma Shave */
      denom = d1+d2+d3+d4-2*(L1+L2);
      if ( denom <= 0.0 )
         kb_error(1544,"Buck denominator nonpositive.\n",RECOVERABLE);

      energy += pow(denom ,-power);
      coeff = -2*power*pow(denom,-power-1);
      for ( j = 0 ; j < SDIM ; j++ )
         { v_info->grad[0][j] += coeff*(dd1[j]/d1 - dd4[j]/d4 + 2*dL1[j]/L1);
            v_info->grad[1][j] += coeff*(dd3[j]/d3 - dd2[j]/d2 - 2*dL1[j]/L1);
         }
    }
  return energy;
}


